/* awkfilt2.c - awk-like filter program */
/*
	This filter program requires three functions to make it into a
	complete program. The three functions should be in ONE source
	file, sharing static global data between them if needed.  In each
	case they should set the return value to zero to tell the main
	program to output whatever is in dest, and non-zero to signal that
	nothing is to be output.  The main program is responsible for
	reading, writing and routing and should be absolutely standard.

	It is intended that the contents of dest will not be arbitrarily
	cleared ... that is to say that it MAY be used to build output
	incrementally during several function calls ... the contents only
	get written to stdout when a zero is returned to main by one of the
	three functions

	int begin(char* dest, char* src);

		This function is called ONCE prior to reading stdin. When called,
		the contents of dest are irrelevant, it is provided to enable
		data to be passed back to main. If this function returns zero (normal)
		then main() writes the contents of dest to standard output. If
		this function returns non-zero then nothing is written to stdout.
      The variable src is initialised to '\0', and it probably not useful
      but it is provided now to facilitate possible future use.

	int process(char* dest, char* source);

		This function is called once for every input record.  The input
		record is pointed to by source. Default processing is to move
		the contents of source to dest and return zero to cause the record
		to be written to stdout.  However, there is no need for the
		contents of dest to bear any relationship to source so that any output
		may be written.  To suppress the writing of output return non-zero.
		The contents of dest are NOT cleared by the standard main line program
		so that dest may be used to build output incrementally, appending data
		during several calls, then writing it based on some
		application-specific condition or trigger.


	int finish(char* dest, char* src);

		This function is called ONCE prior after eof on stdin.
		The contents of dest are irrelevant, it is provided to enable
		data to be passed back to main. If this function returns zero
		then main() writes the contents of dest to standard output. If
		this function returns non-zero then nothing is written to stdout.
      The variable src is initialised to '\0', and it probably not useful
      but it is provided now to facilitate possible future use.

*/

#define MAXLEN 256

#include <stdio.h>
#include <string.h>

int begin(char* dest, char* source);    /* user-supplied function */
int process(char *dest, char *source);  /* user-supplied function */
int finish(char* dest, char* source);   /* user-supplied function */

main(void)
{
	char input[MAXLEN] = "\0";
	char output[MAXLEN] = "\0";

	int err = 0;                   /* error flag  0 = OK */

	/*--- awk-like BEGIN processing */
	err = begin(output, input);     /* pre-file processing */
	puts(output);

	/*--- main processing */
	gets(input);                    /* get first input */
	while (!feof(stdin))
	{
		err = process(output, input);  /* process */
		puts(output);
		gets(input);                  /* get next input */
	}

	/*--- awk-like END processing */
	input[0] = '\0';                 /* reset */
	err = finish(output, input);     /* post-processing */
	err++;
	puts (output);
	return 0;
}
