// Copyright (C) 1999-2000 Id Software, Inc.
//
#include "g_local.h"

#define	MISSILE_PRESTEP_TIME	50
void G_ExplodeCluster( gentity_t *ent );
gentity_t *fire_clustergrenade (gentity_t *self, vec3_t start, vec3_t dir);
extern int maxclients;                    // CCH

/*
================
G_BounceMissile

================
*/
void G_BounceMissile( gentity_t *ent, trace_t *trace ) {
	vec3_t	velocity;
	float	dot;
	int		hitTime;

	// reflect the velocity on the trace plane
	hitTime = level.previousTime + ( level.time - level.previousTime ) * trace->fraction;
	BG_EvaluateTrajectoryDelta( &ent->s.pos, hitTime, velocity );
	dot = DotProduct( velocity, trace->plane.normal );
	VectorMA( velocity, -2*dot, trace->plane.normal, ent->s.pos.trDelta );

	if ( ent->s.eFlags & EF_BOUNCE_HALF ) {
		VectorScale( ent->s.pos.trDelta, 0.65, ent->s.pos.trDelta );
		// check for stop
		if ( trace->plane.normal[2] > 0.2 && VectorLength( ent->s.pos.trDelta ) < 40 ) {
			G_SetOrigin( ent, trace->endpos );
			return;
		}
	}

	VectorAdd( ent->r.currentOrigin, trace->plane.normal, ent->r.currentOrigin);
	VectorCopy( ent->r.currentOrigin, ent->s.pos.trBase );
	ent->s.pos.trTime = level.time;
}


/*
================
G_ExplodeMissile

Explode a missile without an impact
================
*/
void G_ExplodeMissile( gentity_t *ent ) {
	vec3_t		dir;
	vec3_t		origin;

	// Atomic for proxy
	ent->takedamage = qfalse;

	BG_EvaluateTrajectory( &ent->s.pos, level.time, origin );
	SnapVector( origin );
	G_SetOrigin( ent, origin );

	// we don't have a valid direction, so just point straight up
	dir[0] = dir[1] = 0;
	dir[2] = 1;

	ent->s.eType = ET_GENERAL;
	G_AddEvent( ent, EV_MISSILE_MISS, DirToByte( dir ) );

	ent->freeAfterEvent = qtrue;

	// splash damage
	if ( ent->splashDamage ) {
		if( G_RadiusDamage( ent->r.currentOrigin, ent->parent, ent->splashDamage, ent->splashRadius, ent
			, ent->splashMethodOfDeath ) ) {
			g_entities[ent->r.ownerNum].client->accuracy_hits++;
		}
	}

	// CCH: For cluster grenades
	if (!strcmp(ent->classname, "cgrenade")) {
		VectorSet(dir, 20, 20, 40);
		fire_grenade2(ent->parent, ent->r.currentOrigin, dir);
		VectorSet(dir, -20, 20, 40);
		fire_grenade2(ent->parent, ent->r.currentOrigin, dir);
		VectorSet(dir, 20, -20, 40);
		fire_grenade2(ent->parent, ent->r.currentOrigin, dir);
		VectorSet(dir, -20, -20, 40);
		fire_grenade2(ent->parent, ent->r.currentOrigin, dir);
	}
       AdjustRocketCounters( &g_entities[ent->r.ownerNum], ent->target_ent, -1);
	trap_LinkEntity( ent );
}


#ifdef MISSIONPACK
/*
================
ProximityMine_Explode
================
*/
static void ProximityMine_Explode( gentity_t *mine ) {
	G_ExplodeMissile( mine );
	// if the prox mine has a trigger free it
	if (mine->activator) {
		G_FreeEntity(mine->activator);
		mine->activator = NULL;
	}
}

/*
================
ProximityMine_Die
================
*/
static void ProximityMine_Die( gentity_t *ent, gentity_t *inflictor, gentity_t *attacker, int damage, int mod ) {
	ent->think = ProximityMine_Explode;
	ent->nextthink = level.time + 1;
}

/*
================
ProximityMine_Trigger
================
*/
void ProximityMine_Trigger( gentity_t *trigger, gentity_t *other, trace_t *trace ) {
	vec3_t		v;
	gentity_t	*mine;

	if( !other->client ) {
		return;
	}

	// trigger is a cube, do a distance test now to act as if it's a sphere
	VectorSubtract( trigger->s.pos.trBase, other->s.pos.trBase, v );
	if( VectorLength( v ) > trigger->parent->splashRadius ) {
		return;
	}


	if ( g_gametype.integer >= GT_TEAM ) {
		// don't trigger same team mines
		if (trigger->parent->s.generic1 == other->client->sess.sessionTeam) {
			return;
		}
	}

	// ok, now check for ability to damage so we don't get triggered thru walls, closed doors, etc...
	if( !CanDamage( other, trigger->s.pos.trBase ) ) {
		return;
	}

	// trigger the mine!
	mine = trigger->parent;
	mine->s.loopSound = 0;
	G_AddEvent( mine, EV_PROXIMITY_MINE_TRIGGER, 0 );
	mine->nextthink = level.time + 500;

	G_FreeEntity( trigger );
}

/*
================
ProximityMine_Activate
================
*/
static void ProximityMine_Activate( gentity_t *ent ) {
	gentity_t	*trigger;
	float		r;

	ent->think = ProximityMine_Explode;
	ent->nextthink = level.time + g_proxMineTimeout.integer;

	ent->takedamage = qtrue;
	ent->health = 1;
	ent->die = ProximityMine_Die;

	ent->s.loopSound = G_SoundIndex( "sound/weapons/proxmine/wstbtick.wav" );

	// build the proximity trigger
	trigger = G_Spawn ();

	trigger->classname = "proxmine_trigger";

	r = ent->splashRadius;
	VectorSet( trigger->r.mins, -r, -r, -r );
	VectorSet( trigger->r.maxs, r, r, r );

	G_SetOrigin( trigger, ent->s.pos.trBase );

	trigger->parent = ent;
	trigger->r.contents = CONTENTS_TRIGGER;
	trigger->touch = ProximityMine_Trigger;

	trap_LinkEntity (trigger);

	// set pointer to trigger so the entity can be freed when the mine explodes
	ent->activator = trigger;
}

/*
================
ProximityMine_ExplodeOnPlayer
================
*/
static void ProximityMine_ExplodeOnPlayer( gentity_t *mine ) {
	gentity_t	*player;

	player = mine->enemy;
	player->client->ps.eFlags &= ~EF_TICKING;

	if ( player->client->invulnerabilityTime > level.time ) {
		G_Damage( player, mine->parent, mine->parent, vec3_origin, mine->s.origin, 1000, DAMAGE_NO_KNOCKBACK, MOD_JUICED );
		player->client->invulnerabilityTime = 0;
		G_TempEntity( player->client->ps.origin, EV_JUICED );
	}
	else {
		G_SetOrigin( mine, player->s.pos.trBase );
		// make sure the explosion gets to the client
		mine->r.svFlags &= ~SVF_NOCLIENT;
		mine->splashMethodOfDeath = MOD_PROXIMITY_MINE;
		G_ExplodeMissile( mine );
	}
}

/*
================
ProximityMine_Player
================
*/
static void ProximityMine_Player( gentity_t *mine, gentity_t *player ) {
	if( mine->s.eFlags & EF_NODRAW ) {
		return;
	}

	G_AddEvent( mine, EV_PROXIMITY_MINE_STICK, 0 );

	if( player->s.eFlags & EF_TICKING ) {
		player->activator->splashDamage += mine->splashDamage;
		player->activator->splashRadius *= 1.50;
		mine->think = G_FreeEntity;
		mine->nextthink = level.time;
		return;
	}

	player->client->ps.eFlags |= EF_TICKING;
	player->activator = mine;

	mine->s.eFlags |= EF_NODRAW;
	mine->r.svFlags |= SVF_NOCLIENT;
	mine->s.pos.trType = TR_LINEAR;
	VectorClear( mine->s.pos.trDelta );

	mine->enemy = player;
	mine->think = ProximityMine_ExplodeOnPlayer;
	if ( player->client->invulnerabilityTime > level.time ) {
		mine->nextthink = level.time + 2 * 1000;
	}
	else {
		mine->nextthink = level.time + 10 * 1000;
	}
}
#endif

/*
=================
fire_flame
=================
*/
gentity_t *fire_flame (gentity_t *self, vec3_t start, vec3_t dir) {
gentity_t*bolt;

VectorNormalize (dir);

bolt = G_Spawn();
bolt->classname = "flame";
bolt->nextthink = level.time + 10000;
bolt->think = G_ExplodeMissile;
bolt->s.eType = ET_MISSILE;
bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
bolt->s.weapon = WP_FLAME_THROWER;
bolt->r.ownerNum = self->s.number;
bolt->parent = self;
bolt->damage = 30;
bolt->splashDamage = 25;
bolt->splashRadius = 45;
bolt->methodOfDeath = MOD_FLAME_THROWER;
bolt->splashMethodOfDeath = MOD_PLASMA_SPLASH;
bolt->clipmask = MASK_SHOT;

bolt->s.pos.trType = TR_LINEAR;
bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;// move a bit on the very first frame
VectorCopy( start, bolt->s.pos.trBase );
VectorScale( dir, 2000, bolt->s.pos.trDelta );
SnapVector( bolt->s.pos.trDelta );// save net bandwidth

VectorCopy (start, bolt->r.currentOrigin);

return bolt;
}

/*
================
G_MissileImpact
================
*/
void G_MissileImpact( gentity_t *ent, trace_t *trace ) {
	gentity_t		*other;
	qboolean		hitClient = qfalse;
#ifdef MISSIONPACK
	vec3_t			forward, impactpoint, bouncedir;
	int				eFlags;
#endif
       vec3_t               dir;                 // CCH
	other = &g_entities[trace->entityNum];

	// check for bounce
	if ( !other->takedamage &&
		( ent->s.eFlags & ( EF_BOUNCE | EF_BOUNCE_HALF ) ) ) {
              if (grenadetype.integer != 3)
              {
                     G_BounceMissile( ent, trace );
                     G_AddEvent( ent, EV_GRENADE_BOUNCE, 0 );
              }
              if (grenadetype.integer == 3)
                     G_SetOrigin( ent, trace->endpos ); // ZYGOTE NEW LINE
		return;
	}

#ifdef MISSIONPACK
	if ( other->takedamage ) {
		if ( ent->s.weapon != WP_PROX_LAUNCHER ) {
			if ( other->client && other->client->invulnerabilityTime > level.time ) {
				//
				VectorCopy( ent->s.pos.trDelta, forward );
				VectorNormalize( forward );
				if (G_InvulnerabilityEffect( other, forward, ent->s.pos.trBase, impactpoint, bouncedir )) {
					VectorCopy( bouncedir, trace->plane.normal );
					eFlags = ent->s.eFlags & EF_BOUNCE_HALF;
					ent->s.eFlags &= ~EF_BOUNCE_HALF;
					G_BounceMissile( ent, trace );
					ent->s.eFlags |= eFlags;
				}
				ent->target_ent = other;
				return;
			}
		}
	}
#endif
	// Atomic for proxy health
	ent->takedamage = qfalse;

	// impact damage
	if (other->takedamage) {
		// FIXME: wrong damage direction?
		if ( ent->damage ) {
			vec3_t	velocity;

			if( LogAccuracyHit( other, &g_entities[ent->r.ownerNum] ) ) {
				g_entities[ent->r.ownerNum].client->accuracy_hits++;
				hitClient = qtrue;
			}
			BG_EvaluateTrajectoryDelta( &ent->s.pos, level.time, velocity );
			if ( VectorLength( velocity ) == 0 ) {
				velocity[2] = 1;	// stepped on a grenade
			}
			G_Damage (other, ent, &g_entities[ent->r.ownerNum], velocity,
				ent->s.origin, ent->damage, 
				0, ent->methodOfDeath);
		}
	}

#ifdef MISSIONPACK
	if( ent->s.weapon == WP_PROX_LAUNCHER ) {
		if( ent->s.pos.trType != TR_GRAVITY ) {
			return;
		}

		// if it's a player, stick it on to them (flag them and remove this entity)
		if( other->s.eType == ET_PLAYER && other->health > 0 ) {
			ProximityMine_Player( ent, other );
			return;
		}

		SnapVectorTowards( trace->endpos, ent->s.pos.trBase );
		G_SetOrigin( ent, trace->endpos );
		ent->s.pos.trType = TR_STATIONARY;
		VectorClear( ent->s.pos.trDelta );

		G_AddEvent( ent, EV_PROXIMITY_MINE_STICK, trace->surfaceFlags );

		ent->think = ProximityMine_Activate;
		ent->nextthink = level.time + 2000;

		vectoangles( trace->plane.normal, ent->s.angles );
		ent->s.angles[0] += 90;

		// link the prox mine to the other entity
		ent->enemy = other;
		ent->die = ProximityMine_Die;
		VectorCopy(trace->plane.normal, ent->movedir);
		VectorSet(ent->r.mins, -4, -4, -4);
		VectorSet(ent->r.maxs, 4, 4, 4);
		trap_LinkEntity(ent);

		return;
	}
#endif

	if (!strcmp(ent->classname, "hook")) {
		gentity_t *nent;
		vec3_t v;

		nent = G_Spawn();
		if ( other->takedamage && other->client ) {

			G_AddEvent( nent, EV_MISSILE_HIT, DirToByte( trace->plane.normal ) );
			nent->s.otherEntityNum = other->s.number;

			ent->enemy = other;

			v[0] = other->r.currentOrigin[0] + (other->r.mins[0] + other->r.maxs[0]) * 0.5;
			v[1] = other->r.currentOrigin[1] + (other->r.mins[1] + other->r.maxs[1]) * 0.5;
			v[2] = other->r.currentOrigin[2] + (other->r.mins[2] + other->r.maxs[2]) * 0.5;

			SnapVectorTowards( v, ent->s.pos.trBase );	// save net bandwidth
		} else {
			VectorCopy(trace->endpos, v);
			G_AddEvent( nent, EV_MISSILE_MISS, DirToByte( trace->plane.normal ) );
			ent->enemy = NULL;
		}

		SnapVectorTowards( v, ent->s.pos.trBase );	// save net bandwidth

		nent->freeAfterEvent = qtrue;
		// change over to a normal entity right at the point of impact
		nent->s.eType = ET_GENERAL;
		ent->s.eType = ET_GRAPPLE;

		G_SetOrigin( ent, v );
		G_SetOrigin( nent, v );

		ent->think = Weapon_HookThink;
		ent->nextthink = level.time + FRAMETIME;

		ent->parent->client->ps.pm_flags |= PMF_GRAPPLE_PULL;
		VectorCopy( ent->r.currentOrigin, ent->parent->client->ps.grapplePoint);

		trap_LinkEntity( ent );
		trap_LinkEntity( nent );

		return;
	}

	// is it cheaper in bandwidth to just remove this ent and create a new
	// one, rather than changing the missile into the explosion?

	if ( other->takedamage && other->client ) {
		G_AddEvent( ent, EV_MISSILE_HIT, DirToByte( trace->plane.normal ) );
		ent->s.otherEntityNum = other->s.number;
	} else if( trace->surfaceFlags & SURF_METALSTEPS ) {
		G_AddEvent( ent, EV_MISSILE_MISS_METAL, DirToByte( trace->plane.normal ) );
	} else {
		G_AddEvent( ent, EV_MISSILE_MISS, DirToByte( trace->plane.normal ) );
	}

	ent->freeAfterEvent = qtrue;

	// change over to a normal entity right at the point of impact
	ent->s.eType = ET_GENERAL;

	SnapVectorTowards( trace->endpos, ent->s.pos.trBase );	// save net bandwidth

	G_SetOrigin( ent, trace->endpos );

	// splash damage (doesn't apply to person directly hit)
	if ( ent->splashDamage ) {
		if( G_RadiusDamage( trace->endpos, ent->parent, ent->splashDamage, ent->splashRadius, 
			other, ent->splashMethodOfDeath ) ) {
			if( !hitClient ) {
				g_entities[ent->r.ownerNum].client->accuracy_hits++;
			}
		}
	}
	// CCH: For cluster grenades
	if (!strcmp(ent->classname, "cgrenade")) {
		VectorSet(dir, 20, 20, 40);
		fire_grenade2(ent->parent, ent->r.currentOrigin, dir);
		VectorSet(dir, -20, 20, 40);
		fire_grenade2(ent->parent, ent->r.currentOrigin, dir);
		VectorSet(dir, 20, -20, 40);
		fire_grenade2(ent->parent, ent->r.currentOrigin, dir);
		VectorSet(dir, -20, -20, 40);
		fire_grenade2(ent->parent, ent->r.currentOrigin, dir);
	}

	trap_LinkEntity( ent );
}

/*
=================
CCH: fire_grenade2

38: 62. They will also say, `Our Lord, whosoever prepared this for us,
do thou multiply manifold his punishment in the Fire.'
--Holy Quran, translated by Maulvi Sher Ali  
=================
*/
gentity_t *fire_grenade2 (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;
       int    answer;

	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "grenade";
	bolt->nextthink = level.time + 2500;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_GRENADE_LAUNCHER;
	bolt->s.eFlags = EF_BOUNCE_HALF;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
       bolt->damage = gldamage.integer; // 100
       bolt->splashDamage = glsdamage.integer; // 100
       bolt->splashRadius = glsradius.integer; // 150
       bolt->methodOfDeath = MOD_CLUSTER;
       bolt->splashMethodOfDeath = MOD_CLUSTER_SPLASH;
	bolt->clipmask = MASK_SHOT;

	bolt->s.pos.trType = TR_GRAVITY;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
//       VectorScale( dir, 380, bolt->s.pos.trDelta );
       answer = grenadevelocity.integer - 450;
       VectorScale( dir, grenadevelocity.integer-answer, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth

	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}

void G_ExplodeCluster( gentity_t *ent ) {
	vec3_t		dir;
	vec3_t		origin;

	//Gerbil!
	vec3_t		grenade1;
	vec3_t		grenade2;
	vec3_t		grenade3;
	vec3_t		grenade4;

	BG_EvaluateTrajectory( &ent->s.pos, level.time, origin );
	SnapVector( origin );
	G_SetOrigin( ent, origin );

	// we don't have a valid direction, so just point straight up
	dir[0] = dir[1] = 0;
	dir[2] = 10;

	ent->s.eType = ET_GENERAL;
	G_AddEvent( ent, EV_MISSILE_MISS, DirToByte( dir ) );

	ent->freeAfterEvent = qtrue;

	// splash damage
	if ( ent->splashDamage ) {
		if( G_RadiusDamage( ent->r.currentOrigin, ent->parent, ent->splashDamage, ent->splashRadius, NULL
			, ent->splashMethodOfDeath ) ) {
                     g_entities[ent->r.ownerNum].client->accuracy_hits++;
		}
	}

	//Gerbil!
	VectorSet(grenade1, 25, 25, 30);
	VectorSet(grenade2, -25, 25, 30);
	VectorSet(grenade3, 25, -25, 30);
	VectorSet(grenade4, -25, -25, 30);
	fire_clustergrenade (ent, origin, grenade1);
	fire_clustergrenade (ent, origin, grenade2);
	fire_clustergrenade (ent, origin, grenade3);
	fire_clustergrenade (ent, origin, grenade4);

	trap_LinkEntity( ent );
}

/*
==================
wkamkoff - RandomRocketLockMessage
==================
*/
char *RandomGrenadeLockMessage( char *attacker, char *target) {
	static int grenadeMessage;

	// globally cycle through the different messages
	grenadeMessage = ( grenadeMessage + 1 ) % 4;
	switch( grenadeMessage) {
		case 0:
			return va( "print \"%s" S_COLOR_WHITE " aquires a lock on %s with a " S_COLOR_YELLOW "magnetic grenade" S_COLOR_WHITE ".\n\"", attacker, target);
		case 1:
			return va( "print \"%s" S_COLOR_WHITE "'s " S_COLOR_YELLOW "magnetic grenade" S_COLOR_WHITE " homes in on %s" S_COLOR_WHITE ".\n\"", attacker, target);
		case 2:
			return va( "print \"%s" S_COLOR_WHITE " targets %s with a " S_COLOR_YELLOW "magnetic grenade" S_COLOR_WHITE ".\n\"", attacker, target);
		case 3:
			return va( "print \"%s" S_COLOR_WHITE " has been targeted by %s's " S_COLOR_YELLOW "magnetic grenade" S_COLOR_WHITE "'s rocket.\n\"", target, attacker);
		case 4:
		default:
			return va( "print \"%s" S_COLOR_WHITE " throws a " S_COLOR_YELLOW "magnetic grenade" S_COLOR_WHITE " at %s" S_COLOR_WHITE ".\n\"", attacker, target);
	}
}

/*
==================
wkamkoff - AdjustRocketCounters
==================
*/
void AdjustRocketCounters( gentity_t *owner,  gentity_t *target, int changes) {
       static char targetname[32]="name";

	if( target->client) {
//              target->client->ps.stats[STAT_ROCKETLOCK_COUNT]+=changes;
//              if( target->client->ps.stats[STAT_ROCKETLOCK_COUNT] < 0)
//                     target->client->ps.stats[STAT_ROCKETLOCK_COUNT] = 0;
              if( changes==1 && (!(strstr(target->client->pers.netname, targetname)!= NULL))) {
                     strcpy(targetname, target->client->pers.netname);
//              trap_SendServerCommand( owner-g_entities, va("cp \"" "Target -->[%s]\n\n\"",target->client->pers.netname) );
              trap_SendServerCommand( owner-g_entities, va("print \"Target -->[%s]\n\n\"",target->client->pers.netname) );
//              trap_SendServerCommand( -1, va("print \"Target2 -->[%s]\n\n\"",target->client->pers.netname) );
		}
	}
       else if (changes == -1)
             strcpy(targetname, "name");
       /*
	if( owner->client) {
		owner->client->ps.stats[STAT_ROCKET_TARGETS_COUNT]+=changes;
		if( owner->client->ps.stats[STAT_ROCKET_TARGETS_COUNT] < 0)
			owner->client->ps.stats[STAT_ROCKET_TARGETS_COUNT] = 0;
	}
       */
}


/*
=================
G_Proxy
=================
*/
static void G_Proxy( gentity_t *self ) {
	gentity_t *target;
	target = NULL;
       if ((g_gametype.integer == GT_TEAM || g_gametype.integer == GT_CTF) &&
                     target->client->sess.sessionTeam == self->parent->client->sess.sessionTeam)
              return;
	//check if there are any entity's within the radius
	if ((target = findradius(target, self->r.currentOrigin, 150)) != NULL) {
		// make sure we can kill them
              if ((target != self) && (target->client) && target->takedamage && target != self->parent && !target->client->ps.powerups[PW_PROTECT]) {
                     G_Sound(target, CHAN_AUTO, G_SoundIndex("sound/weapons/proxwarn.wav"));
			G_ExplodeMissile( self); // KILL THEM!!!!
		}
	}
	self->nextthink = level.time + 20; 
	// check if time is up
	if (level.time > self->wait) {
		G_ExplodeMissile( self);
	}
}

/*
================
G_RunMissile
================
*/
void G_RunMissile( gentity_t *ent ) {
	vec3_t		origin;
	trace_t		tr;
	int			passent;

	// get current position
	BG_EvaluateTrajectory( &ent->s.pos, level.time, origin );

	// if this missile bounced off an invulnerability sphere
	if ( ent->target_ent ) {
		passent = ent->target_ent->s.number;
	}
#ifdef MISSIONPACK
	// prox mines that left the owner bbox will attach to anything, even the owner
	else if (ent->s.weapon == WP_PROX_LAUNCHER && ent->count) {
		passent = ENTITYNUM_NONE;
	}
#endif
	else {
		// ignore interactions with the missile owner
		passent = ent->r.ownerNum;
	}
	// trace a line from the previous position to the current position
	trap_Trace( &tr, ent->r.currentOrigin, ent->r.mins, ent->r.maxs, origin, passent, ent->clipmask );

	if ( tr.startsolid || tr.allsolid ) {
		// make sure the tr.entityNum is set to the entity we're stuck in
		trap_Trace( &tr, ent->r.currentOrigin, ent->r.mins, ent->r.maxs, ent->r.currentOrigin, passent, ent->clipmask );
		tr.fraction = 0;
	}
	else {
		VectorCopy( tr.endpos, ent->r.currentOrigin );
	}

	trap_LinkEntity( ent );

	if ( tr.fraction != 1 ) {
              AdjustRocketCounters( &g_entities[ent->r.ownerNum], ent->target_ent, -1);
		// never explode or bounce on sky
		if ( tr.surfaceFlags & SURF_NOIMPACT ) {
			// If grapple, reset owner
			// Willi - Offhand Grappling Hook
			if (ent->parent && ent->parent->client->hook == ent)
			{
				ent->parent->client->hook = NULL;
				ent->parent->client->hookhasbeenfired = qfalse;
				ent->parent->client->fireHeld = qfalse;
			}

			G_FreeEntity( ent );
			return;
		}
		G_MissileImpact( ent, &tr );
		if ( ent->s.eType != ET_MISSILE ) {
			return;		// exploded
		}
	}
#ifdef MISSIONPACK
	// if the prox mine wasn't yet outside the player body
	if (ent->s.weapon == WP_PROX_LAUNCHER && !ent->count) {
		// check if the prox mine is outside the owner bbox
		trap_Trace( &tr, ent->r.currentOrigin, ent->r.mins, ent->r.maxs, ent->r.currentOrigin, ENTITYNUM_NONE, ent->clipmask );
		if (!tr.startsolid || tr.entityNum != ent->r.ownerNum) {
			ent->count = 1;
		}
	}
#endif
	// check think function after bouncing
	G_RunThink( ent );
}


//=============================================================================

/*
=================
fire_plasma

=================
*/
gentity_t *fire_plasma (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;

	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "plasma";
	bolt->nextthink = level.time + 10000;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_PLASMAGUN;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
       bolt->damage = pgdamage.integer;
       bolt->splashDamage = pgsdamage.integer;//15
       bolt->splashRadius = pgsradius.integer;//20
	bolt->methodOfDeath = MOD_PLASMA;
	bolt->splashMethodOfDeath = MOD_PLASMA_SPLASH;
	bolt->clipmask = MASK_SHOT;
	bolt->target_ent = NULL;

	bolt->s.pos.trType = TR_LINEAR;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
//       VectorScale( dir, 2000, bolt->s.pos.trDelta );
       VectorScale( dir, plasmavelocity.integer, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth

	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}	

//=============================================================================

#define GSUCK_VELOCITY      1200                  // the amount of kick each second gets
#define GSUCK_RADIUS 500

/*
=================
G_Suck
=================
*/
static void G_Suck( gentity_t *self ) {
	gentity_t *target;
	vec3_t start,dir,end;
       int    answer;
	target = NULL;

//check if there are any entity's within a radius of 500 units.
	while ((target = findradius(target, self->r.currentOrigin, 500)) != NULL)

		{
// target must not be vortex grenade
		if (target == self) 
			continue;

// target must be visible
              if (!visible (self, target))
			continue;

// target must be a client
		if (!target->client) 
			continue;
              if (target->client->ps.powerups[PW_PROTECT])
                     continue;
// target must not be the player who fired the vortex grenade 
		if (target == self->parent) 
			continue;

// target must be able to take damage
		if (!target->takedamage) 
			continue;

// put target position in start
		VectorCopy(target->r.currentOrigin, start); 
// put grenade position in end
		VectorCopy(self->r.currentOrigin, end); 
// subtract start from end to get directional vector
		VectorSubtract(end, start, dir); 
		VectorNormalize(dir); 
// scale directional vector by 2000 and add to the targets velocity
//       VectorScale(dir,GSUCK_VELOCITY, target->client->ps.velocity);
       answer = grenadevelocity.integer - GSUCK_VELOCITY;
       VectorScale(dir,grenadevelocity.integer-answer, target->client->ps.velocity);
// make targets move direction = to directional vector.
		VectorCopy(dir, target->movedir); 
       
		}

              self->nextthink = level.time + 5;

// check if vortext grenade is older than 5 seconds.

              if (level.time > self->wait || (target = findradius(target, self->r.currentOrigin, 20)) != NULL
              && (target != self) && (target->client) && target->takedamage && target != self->parent && !target->client->ps.powerups[PW_PROTECT])
			G_ExplodeMissile( self);
}

/*
================
G_MissileDie

Lancer - Destroy a missile
================
*/
// Atomic
void G_MissileDie( gentity_t *self, gentity_t *inflictor,

gentity_t *attacker, int damage, int mod ) {
	if (inflictor == self)
		return;
	self->takedamage = qfalse;
	self->think = G_ExplodeMissile;
	self->nextthink = level.time + 10;
}

/*
=================
fire_grenade
=================
*/
gentity_t *fire_grenade (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;

	VectorNormalize (dir);
       bolt = G_Spawn();
       bolt->classname = "grenade";
	bolt->nextthink = level.time + 2500;
	bolt->think = G_ExplodeMissile;
	bolt->methodOfDeath = MOD_GRENADE;
	bolt->splashMethodOfDeath = MOD_GRENADE_SPLASH;
       switch(grenadetype.integer)
       {
       case 1:
              bolt->methodOfDeath = MOD_CLUSTER;
              bolt->splashMethodOfDeath = MOD_CLUSTER_SPLASH;
              bolt->classname = "cgrenade";
//              bolt->think = G_ExplodeCluster;
              break;
       case 2:
              bolt->methodOfDeath = MOD_VORTEX;
              bolt->splashMethodOfDeath = MOD_VORTEX_SPLASH;
              bolt->nextthink = level.time + 1000; // call G_Suck in 1 second
              bolt->think = G_Suck;
              bolt->wait = level.time + 5000; // vortext grenade lifetime.
              break;
       case 3:
              bolt->methodOfDeath = MOD_STICKY;
              bolt->splashMethodOfDeath = MOD_STICKY_SPLASH;
              bolt->nextthink = level.time + 1000; // wait 1 second to arm itself
              bolt->think = G_Proxy;
              if (!g_proxytime.integer)
                     bolt->wait = level.time + 10800 * 1000;
              else
                     bolt->wait = level.time + (g_proxytime.integer * 1000); // if still around after 15 seconds, blowup
              bolt->health = 10;
              bolt->takedamage = qtrue;
              bolt->die = G_MissileDie;
              bolt->r.contents = CONTENTS_BODY;
              break;
       case 4:
              bolt->methodOfDeath = MOD_IMPACT;
              bolt->splashMethodOfDeath = MOD_IMPACT_SPLASH;
              break;
       }
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_GRENADE_LAUNCHER;
       if (grenadetype.integer == 0 || grenadetype.integer == 1 || grenadetype.integer == 2 || grenadetype.integer == 3) // then make it a cluster / vortex grenade and allow it to bounce first
             bolt->s.eFlags = EF_BOUNCE_HALF; // bounce, remove it to explode on impact
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
       bolt->damage = gldamage.integer;
       bolt->splashDamage = glsdamage.integer; // 100
       bolt->splashRadius = glsradius.integer; // 150
       /*
       if (grenadetype.integer)
       {
              bolt->damage = 150;
              bolt->splashDamage = 150;
              bolt->splashRadius = 250;
       }
       */
	bolt->clipmask = MASK_SHOT;

	bolt->s.pos.trType = TR_GRAVITY;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
//       VectorScale( dir, 700, bolt->s.pos.trDelta );
       VectorScale( dir, grenadevelocity.integer, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth

	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}

gentity_t *fire_clustergrenade (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;
       int answer;
	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "grenade";
       bolt->nextthink = level.time + 2500;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_GRENADE_LAUNCHER;
	bolt->s.eFlags = EF_BOUNCE_HALF;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
	bolt->damage = 100;
       bolt->splashDamage = glsdamage.integer;
       bolt->splashRadius = glsradius.integer;
	bolt->methodOfDeath = MOD_GRENADE;
	bolt->splashMethodOfDeath = MOD_GRENADE_SPLASH;
	bolt->clipmask = MASK_SHOT;
	bolt->target_ent = NULL;

	bolt->s.pos.trType = TR_GRAVITY;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );

	//Gerbil!  lower the velocity from 700 to 400
//       VectorScale( dir, 500, bolt->s.pos.trDelta );
       answer = grenadevelocity.integer - 500;
       VectorScale( dir, grenadevelocity.integer-answer, bolt->s.pos.trDelta );

	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth

	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}

/*
================
CCH: bfg_think

(PSALM 144:6)  Cast forth lightning, and scatter them: shoot out thine 
  arrows, and destroy them.
--Bible (KJV)
================
*/
void bfg_think( gentity_t *ent ) {
	trace_t		tr;
	vec3_t		vecto;
	gentity_t	*tent, *target;
	int			damage, i;
	float		s_quadFactor;
       int accuracy;
       if (ent->parent->client->accuracy_shots)
              accuracy = ent->parent->client->accuracy_hits * 100 / ent->parent->client->accuracy_shots;
       else
              accuracy = 0;

	for (i = 0; i < maxclients; i++) {
		target = &g_entities[i];

		if (!target->inuse) continue;

		if ( target == ent->parent ) continue;

		if ( OnSameTeam( target, ent->parent ) ) continue;

		VectorSubtract(ent->r.currentOrigin, target->r.currentOrigin, vecto);
		if ( VectorLength(vecto) > (LIGHTNING_RANGE / 2) ) continue;

		trap_Trace( &tr, ent->r.currentOrigin, NULL, NULL, target->r.currentOrigin, ent->r.ownerNum, MASK_SHOT );
		if ( target != &g_entities[tr.entityNum] ) continue;

		if (ent->parent->client->ps.powerups[PW_QUAD] ) {
			s_quadFactor = g_quadfactor.value;
		} else {
			s_quadFactor = 1;
		}
              damage = 3 * s_quadFactor;

              if (target->client->ps.powerups[PW_PROTECT] || target->client->sess.sessionTeam == TEAM_SPECTATOR) break;
		if ((g_gametype.integer == GT_TEAM || g_gametype.integer == GT_CTF) &&
                     target->client->sess.sessionTeam == ent->parent->client->sess.sessionTeam)
              break;

		tent = G_TempEntity( tr.endpos, EV_MISSILE_HIT );
		tent->s.otherEntityNum = target->s.number;
		tent->s.eventParm = DirToByte( tr.plane.normal );
		tent->s.weapon = WP_LIGHTNING;

		if ( target->takedamage) {
			G_Damage( target, ent->parent, ent->parent, vecto, tr.endpos,
                            damage, 0, MOD_ZAP);
			tent = G_TempEntity(ent->r.currentOrigin, EV_LIGHTNING);
			VectorCopy(target->r.currentOrigin, tent->s.origin2);
                     if( LogAccuracyHit( target, ent->parent ) && random() < 0.25 && accuracy < 100) {
                            ent->parent->client->accuracy_hits++;
			}
		}
	}
	ent->nextthink += 50;
       if (level.time > ent->wait) {
              G_ExplodeMissile( ent);
	}
}

/*
=================
fire_bfg
=================
*/
gentity_t *fire_bfg (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;

	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "bfg";
       if (bfgtype.integer)
       {
              bolt->nextthink = level.time + 1;                // CCH
              bolt->think = bfg_think;                         // CCH
       }
       else
       {
              bolt->nextthink = level.time + 10000;
              bolt->think = G_ExplodeMissile;
       }
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_BFG;
	bolt->r.ownerNum = self->s.number;
       bolt->wait = level.time + 15000;
	bolt->parent = self;
	bolt->damage = 100;
       bolt->splashDamage = bfgsdamage.integer;
       bolt->splashRadius = bfgsradius.integer;
	bolt->methodOfDeath = MOD_BFG;
	bolt->splashMethodOfDeath = MOD_BFG_SPLASH;
	bolt->clipmask = MASK_SHOT;

	bolt->s.pos.trType = TR_LINEAR;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
//       VectorScale( dir, 1000, bolt->s.pos.trDelta );
       VectorScale( dir, bfgvelocity.integer, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth
	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}

/*
=================
fire_eshell
=================
*/
gentity_t *fire_eshell (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t *bolt;
	VectorNormalize (dir);
	bolt = G_Spawn();
	bolt->classname = "eshell";
       bolt->nextthink = level.time + 10000;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_SHOTGUN;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
       bolt->damage = 25;
       bolt->splashDamage = 25;
       bolt->methodOfDeath = MOD_SHOTGUN;
       bolt->splashMethodOfDeath = MOD_SHOTGUN;
	bolt->clipmask = MASK_SHOT;
	bolt->s.pos.trType = TR_LINEAR;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME; 
	// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
//       VectorScale( dir, 60000, bolt->s.pos.trDelta );
       VectorScale( dir, 600, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta ); // save net bandwidth
	VectorCopy (start, bolt->r.currentOrigin);
	return bolt;
}


//=============================================================================

/*
=================
fire_bfg
=================
*/
gentity_t *fire_bfg2 (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;

	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "bfg";
	bolt->nextthink = level.time + 10000;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_BFG;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
       bolt->damage = bfgdamage.integer;
       bolt->splashDamage = bfgsdamage.integer;
       bolt->splashRadius = bfgsradius.integer;
	bolt->methodOfDeath = MOD_BFG;
	bolt->splashMethodOfDeath = MOD_BFG_SPLASH;
	bolt->clipmask = MASK_SHOT;
	bolt->target_ent = NULL;

	bolt->s.pos.trType = TR_LINEAR;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
	VectorScale( dir, 2000, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth
	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}

//=============================================================================

void G_HomingMissile( gentity_t *ent )
{
	gentity_t	*target = NULL;
	gentity_t	*blip = NULL;
	vec3_t  dir, blipdir, temp_dir;
	int			msglen;
	int			buflen;
       int    answer;
       int    answer2;
	char		buf[32];
	char		msg[256];
//       static char          targetName[32]="name";
       char netname[MAX_NETNAME];
              char   *s;

       while ((blip = findradius(blip, ent->r.currentOrigin, 2000)) != NULL) {
		if (blip->client==NULL)			continue;
		if (blip==ent->parent)			continue;
		if (blip->health<=0)			continue;
		if (blip->client->sess.sessionTeam == TEAM_SPECTATOR)
			continue;
              if (blip->client->ps.powerups[PW_PROTECT])
                     continue;
		if ((g_gametype.integer == GT_TEAM || g_gametype.integer == GT_CTF) &&
			blip->client->sess.sessionTeam == ent->parent->client->sess.sessionTeam)
			continue;
		//in old code,this ent->parent->cliend-> was blip->parent->client->,
		//so didn't work in CTF and team deathmatch.Now it will work.

		if (!visible (ent, blip))
			continue;

              VectorSubtract(blip->r.currentOrigin, ent->r.currentOrigin, blipdir);
		blipdir[2] += 16;
//              if (blip->client && blip->health >= 0 && (!(strstr(blip->client->pers.netname, targetName)!= NULL)))
//              {
//                     strcpy(targetName, blip->client->pers.netname);
//       G_Printf("Target -->[%s]\n", blip->client->pers.netname);
//              if( blip->client->ps.stats[STAT_ROCKETLOCK_COUNT] == 1)
//                     G_Printf("Target --> [%s] Locked\n", blip->client->pers.netname);
//              }
              AdjustRocketCounters( ent->parent, blip, 1);
		if ((target == NULL) || (VectorLength(blipdir) < VectorLength(dir)))
		{
			//if new target is the nearest
			VectorCopy(blipdir,temp_dir);
			VectorNormalize(temp_dir);
                     VectorAdd(temp_dir,ent->r.currentAngles,temp_dir);
			//now the longer temp_dir length is the more straight path for the rocket.
			if(VectorLength(temp_dir)>1.6)
			{	
				//if this 1.6 were smaller,the rocket also get to target the enemy on his back.
				target = blip;
				VectorCopy(blipdir, dir);
			}
		}
	}
	if (target == NULL)	{
              ent->nextthink = level.time + 5000; // 10000
		// if once the rocket lose target,it will not search new enemy any more,and go away.
	} else {

		ent->s.pos.trTime=level.time;
		VectorCopy(ent->r.currentOrigin, ent->s.pos.trBase );
		//for exact trajectory calculation,set current point to base.
			
		VectorNormalize(dir);
              VectorScale(dir, 0.6,dir);
		VectorAdd(dir,ent->r.currentAngles,dir);
		// this 0.3 is swing rate.this value is cheap,I think.try 0.8 or 1.5.
		// if you want fastest swing,comment out these 3 lines.

		VectorNormalize(dir);
		VectorCopy(dir,ent->r.currentAngles);
		//locate nozzle to target
//              VectorScale (dir,VectorLength(ent->s.pos.trDelta)*1.1,ent->s.pos.trDelta);
              if (rocketvelocity.integer >= 350 && rocketvelocity.integer <= 900)
                     answer = rocketvelocity.integer - 350;
              else if (rocketvelocity.integer > 900)
                     answer = rocketvelocity.integer * 0.6;
              else
                     answer = 0;
              VectorScale (dir,rocketvelocity.integer-answer,ent->s.pos.trDelta);
              if (ent->parent->client->ps.powerups[PW_HASTE] != 0)
              {
              if (rocketvelocity.integer >= 700 && rocketvelocity.integer <= 900)
                            answer2 = rocketvelocity.integer - 700;
                     else if (rocketvelocity.integer > 900)
                            answer2 = rocketvelocity.integer * 0.2;
                     else
                            answer2 = 0;
                     VectorScale (dir,(rocketvelocity.integer)-answer2,ent->s.pos.trDelta);
              }
		//trDelta is actual vector for movement.Because the rockets slow down
		// when swing large angle,so accelalate them.

		SnapVector (ent->s.pos.trDelta);                      // save net bandwidth
              ent->nextthink = level.time + 50; // 100 org. decrease this value also makes fast swing.
	}
       if (ent->parent->client->ps.stats[STAT_HEALTH] <= 0 || ent->parent->health <= 0)
              G_ExplodeMissile( ent);
       if (level.time > ent->wait) {
              G_ExplodeMissile( ent);
	}
}

void Guided_Missile_Think (gentity_t *missile)
{
	vec3_t forward, right, up; 
	vec3_t muzzle;
	float dist;
	gentity_t *player = missile->parent;

	// If our owner can't be found, just return
	if (!player)
	{
		G_Printf ("Guided_Missile_Think : missile has no owner!\n");
		return;
	}

	// Get our forward, right, up vector from the view angle of the player
	AngleVectors (player->client->ps.viewangles, forward, right, up);

	// Calculate the player's eyes origin, and store this origin in muzzle
	CalcMuzzlePoint ( player, forward, right, up, muzzle );
 
	// Tells the engine that our movement starts at the current missile's origin
	VectorCopy (missile->r.currentOrigin, missile->s.pos.trBase );

	// Trajectory type setup (linear move - fly)
	missile->s.pos.trType = TR_LINEAR;
	missile->s.pos.trTime = level.time - 50;

	// Get the dir vector between the player's point of view and the rocket
	// and store it into muzzle again
	VectorSubtract (muzzle, missile->r.currentOrigin, muzzle);
	
	// Add some range to our "line" so we can go behind blocks
	// We could have used a trace function here, but the rocket would
	// have come back if player was aiming on a block while the rocket is behind it
	// as the trace stops on solid blocks
	dist = VectorLength (muzzle) + 400;	 //give the range of our muzzle vector + 400 units
	VectorScale (forward, dist, forward);

	// line straight forward
	VectorAdd (forward, muzzle, muzzle);

	// Normalize the vector so it's 1 unit long, but keep its direction
	VectorNormalize (muzzle);

	// Slow the rocket down a bit, so we can handle it
	VectorScale (muzzle, 300, forward);

	// Set the rockets's velocity so it'll move toward our new direction
	VectorCopy (forward, missile->s.pos.trDelta);

	// Change the rocket's angle so it'll point toward the new direction
	vectoangles (muzzle, missile->s.angles);

	// This should "save net bandwidth" =D
	SnapVector( missile->s.pos.trDelta );

	// Call this function in 0,1 s
	missile->nextthink = level.time + FRAMETIME; 
}
/*
=================
fire_rocket
=================
*/
gentity_t *fire_rocket (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;
       int    answer;
	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "rocket";
	bolt->nextthink = level.time + 15000;
       if (guidedrockets.integer)
       {
              if (self->client)           // && check for bots
              {
                     bolt->think = Guided_Missile_Think;
                     bolt->nextthink = level.time + FRAMETIME;
              }
              else
              {
                     bolt->nextthink = level.time + 10000;
                     bolt->think = G_ExplodeMissile;
              }
       }
       else
       {
              switch(homingmissile.integer)
              {
              case 1:
                     bolt->think = G_HomingMissile;
                     bolt->nextthink = level.time + 60; // 60
                     bolt->wait = level.time + 5000; // if still around after 5 seconds, blowup
                     break;
              default:
                     bolt->nextthink = level.time + 10000;
                     bolt->think = G_ExplodeMissile;
                     break;
              }
       }
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_ROCKET_LAUNCHER;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
       switch(homingmissile.integer)
       {
       case 1:
              if (rlsdamage.integer >= 50 && rlsdamage.integer <= 100 )
                     bolt->splashDamage = 50; // 100
              else
                     bolt->splashDamage = rlsdamage.integer;
              if (rlsradius.integer >= 60 && rlsdamage.integer <= 120)
                     bolt->splashRadius = 60;
              else
                     bolt->splashRadius = rlsdamage.integer; // 120
              bolt->methodOfDeath = MOD_ROCKET_HOMING;
              bolt->splashMethodOfDeath = MOD_ROCKET_HOMING_SPLASH;
              break;
       default:
              bolt->splashDamage = rlsdamage.integer;
              bolt->splashRadius = rlsradius.integer;
              bolt->methodOfDeath = MOD_ROCKET;
              bolt->splashMethodOfDeath = MOD_ROCKET_SPLASH;
       }
       if (doublerockets.integer)
       {
              bolt->splashDamage = rlsdamage.integer;
              bolt->splashRadius = rlsradius.integer;
              bolt->methodOfDeath = MOD_DOUBLE_ROCKETS;
              bolt->splashMethodOfDeath = MOD_DOUBLE_ROCKETS_SPLASH;
       }
       if (doublerockets.integer && homingmissile.integer)
       {
              if (rlsdamage.integer >= 75 && rlsdamage.integer <= 100 )
                     bolt->splashDamage = 75; // 100
              else
                     bolt->splashDamage = rlsdamage.integer;
              if (rlsradius.integer >= 85 && rlsdamage.integer <= 120)
                     bolt->splashRadius = 85;
              else
                     bolt->splashRadius = rlsdamage.integer; // 120
              bolt->methodOfDeath = MOD_DOUBLE_HOMING;
              bolt->splashMethodOfDeath = MOD_DOUBLE_HOMING_SPLASH;
       }
       bolt->damage = rldamage.integer;
	bolt->clipmask = MASK_SHOT;
	bolt->target_ent = NULL;

	bolt->s.pos.trType = TR_LINEAR;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
       if (homingmissile.integer == 1)
//              VectorScale( dir, 450, bolt->s.pos.trDelta );
       {
              if (rocketvelocity.integer >= 450 && rocketvelocity.integer <= 900)
                     answer = rocketvelocity.integer - 450;
              else if (rocketvelocity.integer > 900)
                     answer = rocketvelocity.integer * 0.5;
              else
                     answer = 0;
              VectorScale( dir, rocketvelocity.integer-answer, bolt->s.pos.trDelta );
       }
       else
//              VectorScale( dir, 900, bolt->s.pos.trDelta ); // velocity is 900
              VectorScale( dir, rocketvelocity.integer, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth
	VectorCopy (start, bolt->r.currentOrigin);
       if (homingmissile.integer == 1)
              VectorCopy (dir, bolt->r.currentAngles);         // this is new line.
	return bolt;
}

gentity_t *fire_rocket2 (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;

	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "rocket";
	
	//### NEW ###
	//bolt->nextthink = level.time + 10000;
	//bolt->think = G_ExplodeMissile;
	if (self->client)		// && check for bots
	{
		bolt->think = Guided_Missile_Think;
		bolt->nextthink = level.time + FRAMETIME;
	}
	else
	{
		bolt->nextthink = level.time + 10000;
		bolt->think = G_ExplodeMissile;
	}
	//### END NEW ###

	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_ROCKET_LAUNCHER;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
	bolt->damage = 100;
	bolt->splashDamage = 100;
	bolt->splashRadius = 120;
	bolt->methodOfDeath = MOD_ROCKET;
	bolt->splashMethodOfDeath = MOD_ROCKET_SPLASH;
	bolt->clipmask = MASK_SHOT;
	bolt->target_ent = NULL;

	bolt->s.pos.trType = TR_LINEAR;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
	VectorScale( dir, 900, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth
	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}

/*
=================
fire_grapple
=================
*/
gentity_t *fire_grapple (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*hook;

	VectorNormalize (dir);

	hook = G_Spawn();
	hook->classname = "hook";
	hook->nextthink = level.time + 10000;
	hook->think = Weapon_HookFree;
	hook->s.eType = ET_MISSILE;
	hook->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	hook->s.weapon = WP_GRAPPLING_HOOK;
	hook->r.ownerNum = self->s.number;
	hook->methodOfDeath = MOD_GRAPPLE;
	hook->clipmask = MASK_SHOT;
	hook->parent = self;
	hook->target_ent = NULL;

	hook->s.pos.trType = TR_LINEAR;
	hook->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	hook->s.otherEntityNum = self->s.number; // use to match beam in client
	VectorCopy( start, hook->s.pos.trBase );
       VectorScale( dir, g_gpreload.integer, hook->s.pos.trDelta ); // grapple pull speed
	SnapVector( hook->s.pos.trDelta );			// save net bandwidth
	VectorCopy (start, hook->r.currentOrigin);

	self->client->hook = hook;

	return hook;
}


#ifdef MISSIONPACK
/*
=================
fire_nail
=================
*/
#define NAILGUN_SPREAD	1000

gentity_t *fire_nail( gentity_t *self, vec3_t start, vec3_t forward, vec3_t right, vec3_t up ) {
	gentity_t	*bolt;
	vec3_t		dir;
	vec3_t		end;
	float		r, u, scale;

	bolt = G_Spawn();
	bolt->classname = "nail";
	bolt->nextthink = level.time + 10000;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_NAILGUN;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
	bolt->damage = 20;
	bolt->methodOfDeath = MOD_NAIL;
	bolt->clipmask = MASK_SHOT;
	bolt->target_ent = NULL;

	bolt->s.pos.trType = TR_LINEAR;
	bolt->s.pos.trTime = level.time;
	VectorCopy( start, bolt->s.pos.trBase );

	r = random() * M_PI * 2.0f;
	u = sin(r) * crandom() * NAILGUN_SPREAD * 16;
	r = cos(r) * crandom() * NAILGUN_SPREAD * 16;
	VectorMA( start, 8192 * 16, forward, end);
	VectorMA (end, r, right, end);
	VectorMA (end, u, up, end);
	VectorSubtract( end, start, dir );
	VectorNormalize( dir );

	scale = 555 + random() * 1800;
	VectorScale( dir, scale, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta );

	VectorCopy( start, bolt->r.currentOrigin );

	return bolt;
}	


/*
=================
fire_prox
=================
*/
gentity_t *fire_prox( gentity_t *self, vec3_t start, vec3_t dir ) {
	gentity_t	*bolt;

	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "prox mine";
	bolt->nextthink = level.time + 3000;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_PROX_LAUNCHER;
	bolt->s.eFlags = 0;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
	bolt->damage = 0;
       bolt->splashDamage = glsdamage.integer;
       bolt->splashRadius = glsradius.integer;
	bolt->methodOfDeath = MOD_PROXIMITY_MINE;
	bolt->splashMethodOfDeath = MOD_PROXIMITY_MINE;
	bolt->clipmask = MASK_SHOT;
	bolt->target_ent = NULL;
	// count is used to check if the prox mine left the player bbox
	// if count == 1 then the prox mine left the player bbox and can attack to it
	bolt->count = 0;

	//FIXME: we prolly wanna abuse another field
	bolt->s.generic1 = self->client->sess.sessionTeam;

	bolt->s.pos.trType = TR_GRAVITY;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
	VectorScale( dir, 700, bolt->s.pos.trDelta );
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth

	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}
#endif
