// Copyright (C) 1999-2000 Id Software, Inc.
//
// cg_draw.c -- draw all of the graphical elements during
// active (after loading) gameplay

#include "cg_local.h"

#ifdef MISSIONPACK
#include "../ui/ui_shared.h"

// used for scoreboard
extern displayContextDef_t cgDC;
menuDef_t *menuScoreboard = NULL;
#else
int drawTeamOverlayModificationCount = -1;
#endif

int sortedTeamPlayers[TEAM_MAXOVERLAY];
int	numSortedTeamPlayers;

char systemChat[256];
char teamChat1[256];
char teamChat2[256];

#ifdef MISSIONPACK

int CG_Text_Width(const char *text, float scale, int limit) {
  int count,len;
	float out;
	glyphInfo_t *glyph;
	float useScale;
	const char *s = text;
	fontInfo_t *font = &cgDC.Assets.textFont;
	if (scale <= cg_smallFont.value) {
		font = &cgDC.Assets.smallFont;
	} else if (scale > cg_bigFont.value) {
		font = &cgDC.Assets.bigFont;
	}
	useScale = scale * font->glyphScale;
  out = 0;
  if (text) {
    len = strlen(text);
		if (limit > 0 && len > limit) {
			len = limit;
		}
		count = 0;
		while (s && *s && count < len) {
			if ( Q_IsColorString(s) ) {
				s += 2;
				continue;
			} else {
				glyph = &font->glyphs[*s];
				out += glyph->xSkip;
				s++;
				count++;
			}
    }
  }
  return out * useScale;
}

int CG_Text_Height(const char *text, float scale, int limit) {
  int len, count;
	float max;
	glyphInfo_t *glyph;
	float useScale;
	const char *s = text;
	fontInfo_t *font = &cgDC.Assets.textFont;
	if (scale <= cg_smallFont.value) {
		font = &cgDC.Assets.smallFont;
	} else if (scale > cg_bigFont.value) {
		font = &cgDC.Assets.bigFont;
	}
	useScale = scale * font->glyphScale;
  max = 0;
  if (text) {
    len = strlen(text);
		if (limit > 0 && len > limit) {
			len = limit;
		}
		count = 0;
		while (s && *s && count < len) {
			if ( Q_IsColorString(s) ) {
				s += 2;
				continue;
			} else {
				glyph = &font->glyphs[*s];
	      if (max < glyph->height) {
		      max = glyph->height;
			  }
				s++;
				count++;
			}
    }
  }
  return max * useScale;
}

void CG_Text_PaintChar(float x, float y, float width, float height, float scale, float s, float t, float s2, float t2, qhandle_t hShader) {
  float w, h;
  w = width * scale;
  h = height * scale;
  CG_AdjustFrom640( &x, &y, &w, &h );
  trap_R_DrawStretchPic( x, y, w, h, s, t, s2, t2, hShader );
}

void CG_Text_Paint(float x, float y, float scale, vec4_t color, const char *text, float adjust, int limit, int style) {
  int len, count;
	vec4_t newColor;
	glyphInfo_t *glyph;
	float useScale;
	fontInfo_t *font = &cgDC.Assets.textFont;
	if (scale <= cg_smallFont.value) {
		font = &cgDC.Assets.smallFont;
	} else if (scale > cg_bigFont.value) {
		font = &cgDC.Assets.bigFont;
	}
	useScale = scale * font->glyphScale;
  if (text) {
		const char *s = text;
		trap_R_SetColor( color );
		memcpy(&newColor[0], &color[0], sizeof(vec4_t));
    len = strlen(text);
		if (limit > 0 && len > limit) {
			len = limit;
		}
		count = 0;
		while (s && *s && count < len) {
			glyph = &font->glyphs[*s];
      //int yadj = Assets.textFont.glyphs[text[i]].bottom + Assets.textFont.glyphs[text[i]].top;
      //float yadj = scale * (Assets.textFont.glyphs[text[i]].imageHeight - Assets.textFont.glyphs[text[i]].height);
			if ( Q_IsColorString( s ) ) {
				memcpy( newColor, g_color_table[ColorIndex(*(s+1))], sizeof( newColor ) );
				newColor[3] = color[3];
				trap_R_SetColor( newColor );
				s += 2;
				continue;
			} else {
				float yadj = useScale * glyph->top;
				if (style == ITEM_TEXTSTYLE_SHADOWED || style == ITEM_TEXTSTYLE_SHADOWEDMORE) {
					int ofs = style == ITEM_TEXTSTYLE_SHADOWED ? 1 : 2;
					colorBlack[3] = newColor[3];
					trap_R_SetColor( colorBlack );
					CG_Text_PaintChar(x + ofs, y - yadj + ofs, 
														glyph->imageWidth,
														glyph->imageHeight,
														useScale, 
														glyph->s,
														glyph->t,
														glyph->s2,
														glyph->t2,
														glyph->glyph);
					colorBlack[3] = 1.0;
					trap_R_SetColor( newColor );
				}
				CG_Text_PaintChar(x, y - yadj, 
													glyph->imageWidth,
													glyph->imageHeight,
													useScale, 
													glyph->s,
													glyph->t,
													glyph->s2,
													glyph->t2,
													glyph->glyph);
				// CG_DrawPic(x, y - yadj, scale * cgDC.Assets.textFont.glyphs[text[i]].imageWidth, scale * cgDC.Assets.textFont.glyphs[text[i]].imageHeight, cgDC.Assets.textFont.glyphs[text[i]].glyph);
				x += (glyph->xSkip * useScale) + adjust;
				s++;
				count++;
			}
    }
	  trap_R_SetColor( NULL );
  }
}


#endif

/*
==============
CG_DrawField

Draws large numbers for status bar and powerups
==============
*/
#ifndef MISSIONPACK
static void CG_DrawField (int x, int y, int width, int value) {
	char	num[16], *ptr;
	int		l;
	int		frame;

	if ( width < 1 ) {
		return;
	}

	// draw number string
	if ( width > 5 ) {
		width = 5;
	}

	switch ( width ) {
	case 1:
		value = value > 9 ? 9 : value;
		value = value < 0 ? 0 : value;
		break;
	case 2:
		value = value > 99 ? 99 : value;
		value = value < -9 ? -9 : value;
		break;
	case 3:
		value = value > 999 ? 999 : value;
		value = value < -99 ? -99 : value;
		break;
	case 4:
		value = value > 9999 ? 9999 : value;
		value = value < -999 ? -999 : value;
		break;
	}

	Com_sprintf (num, sizeof(num), "%i", value);
	l = strlen(num);
	if (l > width)
		l = width;
       x += 2 + CHAR_WIDTH/2*(width - l);

	ptr = num;
	while (*ptr && l)
	{
		if (*ptr == '-')
			frame = STAT_MINUS;
		else
			frame = *ptr -'0';

              CG_DrawPic( x,y, CHAR_WIDTH/2, CHAR_HEIGHT/2, cgs.media.numberShaders[frame] );
              x += CHAR_WIDTH/2;
		ptr++;
		l--;
	}
}
#endif

/*
================
CG_Draw3DModel

================
*/
void CG_Draw3DModel( float x, float y, float w, float h, qhandle_t model, qhandle_t skin, vec3_t origin, vec3_t angles ) {
	refdef_t		refdef;
	refEntity_t		ent;

	if ( !cg_draw3dIcons.integer || !cg_drawIcons.integer ) {
		return;
	}

	CG_AdjustFrom640( &x, &y, &w, &h );

	memset( &refdef, 0, sizeof( refdef ) );

	memset( &ent, 0, sizeof( ent ) );
	AnglesToAxis( angles, ent.axis );
	VectorCopy( origin, ent.origin );
	ent.hModel = model;
	ent.customSkin = skin;
	ent.renderfx = RF_NOSHADOW;		// no stencil shadows

	refdef.rdflags = RDF_NOWORLDMODEL;

	AxisClear( refdef.viewaxis );

	refdef.fov_x = 30;
	refdef.fov_y = 30;

	refdef.x = x;
	refdef.y = y;
	refdef.width = w;
	refdef.height = h;

	refdef.time = cg.time;

	trap_R_ClearScene();
	trap_R_AddRefEntityToScene( &ent );
	trap_R_RenderScene( &refdef );
}

/*
================
CG_DrawHead

Used for both the status bar and the scoreboard
================
*/
void CG_DrawHead( float x, float y, float w, float h, int clientNum, vec3_t headAngles ) {
	clipHandle_t	cm;
	clientInfo_t	*ci;
	float			len;
	vec3_t			origin;
	vec3_t			mins, maxs;

	ci = &cgs.clientinfo[ clientNum ];

	if ( cg_draw3dIcons.integer ) {
		cm = ci->headModel;
		if ( !cm ) {
			return;
		}

		// offset the origin y and z to center the head
		trap_R_ModelBounds( cm, mins, maxs );

		origin[2] = -0.5 * ( mins[2] + maxs[2] );
		origin[1] = 0.5 * ( mins[1] + maxs[1] );

		// calculate distance so the head nearly fills the box
		// assume heads are taller than wide
		len = 0.7 * ( maxs[2] - mins[2] );		
		origin[0] = len / 0.268;	// len / tan( fov/2 )

		// allow per-model tweaking
		VectorAdd( origin, ci->headOffset, origin );

		CG_Draw3DModel( x, y, w, h, ci->headModel, ci->headSkin, origin, headAngles );
	} else if ( cg_drawIcons.integer ) {
              CG_DrawPic( x, y, w/=1.1, h/=1.1, ci->modelIcon );
	}

	// if they are deferred, draw a cross out
       CG_LoadDeferredPlayers();
	if ( ci->deferred ) {
		CG_DrawPic( x, y, w, h, cgs.media.deferShader );
	}
}

/*
================
CG_DrawFlagModel

Used for both the status bar and the scoreboard
================
*/
void CG_DrawFlagModel( float x, float y, float w, float h, int team, qboolean force2D ) {
	qhandle_t		cm;
	float			len;
	vec3_t			origin, angles;
	vec3_t			mins, maxs;
	qhandle_t		handle;

	if ( !force2D && cg_draw3dIcons.integer ) {

		VectorClear( angles );

		cm = cgs.media.redFlagModel;

		// offset the origin y and z to center the flag
		trap_R_ModelBounds( cm, mins, maxs );

		origin[2] = -0.5 * ( mins[2] + maxs[2] );
		origin[1] = 0.5 * ( mins[1] + maxs[1] );

		// calculate distance so the flag nearly fills the box
		// assume heads are taller than wide
		len = 0.5 * ( maxs[2] - mins[2] );		
		origin[0] = len / 0.268;	// len / tan( fov/2 )

		angles[YAW] = 60 * sin( cg.time / 2000.0 );;

		if( team == TEAM_RED ) {
			handle = cgs.media.redFlagModel;
		} else if( team == TEAM_BLUE ) {
			handle = cgs.media.blueFlagModel;
		} else if( team == TEAM_FREE ) {
			handle = cgs.media.neutralFlagModel;
		} else {
			return;
		}
              CG_Draw3DModel( x, y, w, h, handle, 0, origin, angles );
	} else if ( cg_drawIcons.integer ) {
		gitem_t *item;

		if( team == TEAM_RED ) {
			item = BG_FindItemForPowerup( PW_REDFLAG );
		} else if( team == TEAM_BLUE ) {
			item = BG_FindItemForPowerup( PW_BLUEFLAG );
		} else if( team == TEAM_FREE ) {
			item = BG_FindItemForPowerup( PW_NEUTRALFLAG );
		} else {
			return;
		}
		if (item) {
                CG_DrawPic( x, y, w, h, cg_items[ ITEM_INDEX(item) ].icon );
		}
	}
}

/*
================
CG_DrawStatusBarHead

================
*/
#ifndef MISSIONPACK

static void CG_DrawStatusBarHead( float x ) {
	vec3_t		angles;
	float		size, stretch;
	float		frac;

	VectorClear( angles );

	if ( cg.damageTime && cg.time - cg.damageTime < DAMAGE_TIME ) {
		frac = (float)(cg.time - cg.damageTime ) / DAMAGE_TIME;
		size = ICON_SIZE * 1.25 * ( 1.5 - frac * 0.5 );

		stretch = size - ICON_SIZE * 1.25;
		// kick in the direction of damage
		x -= stretch * 0.5 + cg.damageX * stretch * 0.5;

		cg.headStartYaw = 180 + cg.damageX * 45;

		cg.headEndYaw = 180 + 20 * cos( crandom()*M_PI );
		cg.headEndPitch = 5 * cos( crandom()*M_PI );

		cg.headStartTime = cg.time;
		cg.headEndTime = cg.time + 100 + random() * 2000;
	} else {
		if ( cg.time >= cg.headEndTime ) {
			// select a new head angle
			cg.headStartYaw = cg.headEndYaw;
			cg.headStartPitch = cg.headEndPitch;
			cg.headStartTime = cg.headEndTime;
			cg.headEndTime = cg.time + 100 + random() * 2000;

			cg.headEndYaw = 180 + 20 * cos( crandom()*M_PI );
			cg.headEndPitch = 5 * cos( crandom()*M_PI );
		}

		size = ICON_SIZE * 1.25;
	}

	// if the server was frozen for a while we may have a bad head start time
	if ( cg.headStartTime > cg.time ) {
		cg.headStartTime = cg.time;
	}

	frac = ( cg.time - cg.headStartTime ) / (float)( cg.headEndTime - cg.headStartTime );
	frac = frac * frac * ( 3 - 2 * frac );
	angles[YAW] = cg.headStartYaw + ( cg.headEndYaw - cg.headStartYaw ) * frac;
	angles[PITCH] = cg.headStartPitch + ( cg.headEndPitch - cg.headStartPitch ) * frac;

       CG_DrawHead( x, 452, size/2.5, size/2.5,
				cg.snap->ps.clientNum, angles );
}

/*
================
CG_DrawStatusBarHead2

================
*/
static void CG_DrawStatusBarHead2( float x, float y, float height, float width ) {
	vec3_t		angles;
	float		size, stretch;
	float		frac;

	VectorClear( angles );
//       x = TEXT_ICON_SPACE;
	if ( cg.damageTime && cg.time - cg.damageTime < DAMAGE_TIME ) {
		frac = (float)(cg.time - cg.damageTime ) / DAMAGE_TIME;
              size = ICON_SIZE * 1.25 * ( 1.5 - frac * 0.5 );

              stretch = size - ICON_SIZE * 1.25;
		// kick in the direction of damage
              x -= stretch * 0.5 + cg.damageX * stretch * 0.5;

		cg.headStartYaw = 180 + cg.damageX * 45;

		cg.headEndYaw = 180 + 20 * cos( crandom()*M_PI );
		cg.headEndPitch = 5 * cos( crandom()*M_PI );

		cg.headStartTime = cg.time;
		cg.headEndTime = cg.time + 100 + random() * 2000;
	} else {
		if ( cg.time >= cg.headEndTime ) {
			// select a new head angle
			cg.headStartYaw = cg.headEndYaw;
			cg.headStartPitch = cg.headEndPitch;
			cg.headStartTime = cg.headEndTime;
			cg.headEndTime = cg.time + 100 + random() * 2000;

			cg.headEndYaw = 180 + 20 * cos( crandom()*M_PI );
			cg.headEndPitch = 5 * cos( crandom()*M_PI );
		}

              size = ICON_SIZE * 1.25;
	}

	// if the server was frozen for a while we may have a bad head start time
	if ( cg.headStartTime > cg.time ) {
		cg.headStartTime = cg.time;
	}

	frac = ( cg.time - cg.headStartTime ) / (float)( cg.headEndTime - cg.headStartTime );
	frac = frac * frac * ( 3 - 2 * frac );
	angles[YAW] = cg.headStartYaw + ( cg.headEndYaw - cg.headStartYaw ) * frac;
	angles[PITCH] = cg.headStartPitch + ( cg.headEndPitch - cg.headStartPitch ) * frac;

       CG_DrawHead( x, y, height, width,
				cg.snap->ps.clientNum, angles );
}
#endif

/*
================
CG_DrawStatusBarFlag

================
*/
#ifndef MISSIONPACK
static void CG_DrawStatusBarFlag( float x, int team ) {
       CG_DrawFlagModel( x, 452, ICON_SIZE/2, ICON_SIZE/2, team, qfalse );
}
#endif

/*
================
CG_DrawTeamBackground

================
*/
void CG_DrawTeamBackground( int x, int y, int w, int h, float alpha, int team )
{
	vec4_t		hcolor;

	hcolor[3] = alpha;
	if ( team == TEAM_RED ) {
		hcolor[0] = 1;
		hcolor[1] = 0;
		hcolor[2] = 0;
	} else if ( team == TEAM_BLUE ) {
		hcolor[0] = 0;
		hcolor[1] = 0;
		hcolor[2] = 1;
	} else {
		return;
	}
	trap_R_SetColor( hcolor );
       if (cg_hud.integer == 1)
              CG_DrawPic( x, y, w, h, cgs.media.teamStatusBar );
       else
              CG_DrawPic( x, y+30, w, h/2, cgs.media.teamStatusBar );
	trap_R_SetColor( NULL );
}

/* q3app
================
CG_DrawDamageMOD

draws icons pointing what kind of damage is being taken (used by CG_DrawDamageIcons)
================
*/
static void CG_DrawDamageMOD( float x, float y ) {
        int     i;
        float   j;
        char    *damstr;
        qhandle_t       icon;

        i = cg.predictedPlayerState.persistant[PERS_MOD];

        switch ( i ) {
        case MOD_GRAPPLE:
                icon = cg_weapons[ WP_GRAPPLING_HOOK ].weaponIcon;
                break;
        case MOD_GAUNTLET:
                icon = cg_weapons[ WP_GAUNTLET ].weaponIcon;
                break;
        case MOD_MACHINEGUN:
                icon = cg_weapons[ WP_MACHINEGUN ].weaponIcon;
                break;
        case MOD_SHOTGUN:
                icon = cg_weapons[ WP_SHOTGUN ].weaponIcon;
                break;
        case MOD_GRENADE: case MOD_GRENADE_SPLASH:
                icon = cg_weapons[ WP_GRENADE_LAUNCHER ].weaponIcon;
                break;
        case MOD_ROCKET: case MOD_ROCKET_SPLASH:
                icon = cg_weapons[ WP_ROCKET_LAUNCHER ].weaponIcon;
                break;
        case MOD_PLASMA: case MOD_PLASMA_SPLASH:
                icon = cg_weapons[ WP_PLASMAGUN ].weaponIcon;
                break;
        case MOD_RAILGUN:
                icon = cg_weapons[ WP_RAILGUN ].weaponIcon;
                break;
        case MOD_LIGHTNING:
                icon = cg_weapons[ WP_LIGHTNING ].weaponIcon;
                break;
        case MOD_BFG: case MOD_BFG_SPLASH: case MOD_ZAP:
                icon = cg_weapons[ WP_BFG ].weaponIcon;
                break;
        case MOD_FLAME_THROWER:
                icon = cg_weapons[ WP_FLAME_THROWER ].weaponIcon;
                break;
        default:
                switch ( i ) {
                case MOD_WATER:
                        damstr = "WATER";
                        j = ( CG_DrawStrlen( damstr ) * BIGCHAR_WIDTH ) / 2;              
                        CG_DrawBigString( x + 16 - j, y + 16, damstr, 1.0F);
                        break;
                case MOD_SLIME:
                        damstr = "SLIME";
                        j = ( CG_DrawStrlen( damstr ) * BIGCHAR_WIDTH ) / 2;              
                        CG_DrawBigString( x + 16 - j, y + 16, damstr, 1.0F);
                        break;
                case MOD_LAVA:
                        damstr = "LAVA";
                        j = ( CG_DrawStrlen( damstr ) * BIGCHAR_WIDTH ) / 2;              
                        CG_DrawBigString( x + 16 - j, y + 16, damstr, 1.0F);
                        break;
                case MOD_FALLING:
                        damstr = "FALL";
                        j = ( CG_DrawStrlen( damstr ) * BIGCHAR_WIDTH ) / 2;              
                        CG_DrawBigString( x + 16 - j, y + 16, damstr, 1.0F);
                        break;
                }
                i = -1;
                break;

        }
        if (i != -1 ) {
                CG_DrawPic( x, y, 32, 32, icon );
        }
}

/* q3app
================
CG_DrawDamageIcons

draws icons pointing from which direction damage is being taken
================
*/
static void CG_DrawDamageIcons( void ) {
        int     dir, dir2, w;
        float   *color, x, y, pitch, yaw, fov;
        float   temp_45, temp_135, temp_225, temp_315;
        char    *values;

        if ( cg.v_dmg_time < cg.time ) {
                return;
        }

        pitch = cg.predictedPlayerState.damagePitch / 255.0 * 360;
        yaw = cg.predictedPlayerState.damageYaw / 255.0 * 360;
        if ( cg.predictedPlayerState.persistant[PERS_ATTACKER] == ENTITYNUM_WORLD ) { // if attacker is world
                dir = 5; dir2 = 0;
        }
        else {
                x = AngleMod ( AngleSubtract( yaw , AngleMod( cg.refdefViewAngles[YAW] + 180 ) ) );
                y = pitch;

                if ( cg.zoomed ) fov = cg_zoomFov.value; else fov = cg_fov.value; // wish I could use cg.refdef.fov_x instead!
                if ( fov < 1 ) {
                        fov = 1;
                } else if ( fov > 160 ) {
                        fov = 160;
                }

                temp_45 = fov * 0.5;
                temp_135 = ( ( 180 - fov ) * 0.5 ) + fov;
                temp_225 = 360 - temp_135;
                temp_315 = 360 - temp_45;

                if ( x < temp_135 && x > temp_45 ) { // left
                        if ( y < temp_135 && y > temp_45 ) {
                                dir = 8;
                        } else if ( y > temp_225 && y < temp_315 ) {
                                dir = 6;
                        } else {
                                dir = 7;
                        }
                } else if ( x > temp_225 && x < temp_315 ) { // right
                        if ( y < temp_135 && y > temp_45 ) {
                                dir = 2;
                        } else if ( y > temp_225 && y < temp_315 ) {
                                dir = 4;
                        } else {
                                dir = 3;
                        }
                } else if ( y < temp_135 && y > temp_45 ) {
                        dir = 1;
                } else if ( y > temp_225 && y < temp_315 ) {
                        dir = 5;
                } else {
                        dir = 0;
                }
        
                if ( x > temp_315 || x < temp_45 ) { // front
                        dir2 = 1;
                } else if ( x < temp_225 && x > temp_135 ) { // back
                        dir2 = 2;
                } else {
                        dir2 = 0;
                }
        }

        color = CG_FadeColor( cg.damageTime, DAMAGE_TIME );
        trap_R_SetColor( color );

        switch ( dir2 ) {
        case 1:
                CG_DrawPic( 288, 132, 64, 24, cgs.media.damageArrow2 );
                CG_DrawDamageMOD( 304, 158 );
                break;
        case 2:
                CG_DrawPic( 288, 346, 64, -24, cgs.media.damageArrow2 );
                CG_DrawDamageMOD( 304, 288 );
                break;
        default:

                switch ( dir ) {
                case 1: // up
                        CG_DrawPic( 256, 78, 128, 16, cgs.media.damageArrow2 );
                        CG_DrawDamageMOD( 304, 96 );
                        break;
                case 2: // upper-right
                        CG_DrawPic( 338+48, 104-18, 96, 96, cgs.media.damageArrow3 );
                        CG_DrawDamageMOD( 354+56, 152-16 );
                        break;
                case 3: // right
                        CG_DrawPic( 488, 176, -16, 128, cgs.media.damageArrow1 );
                        CG_DrawDamageMOD( 438, 224 );
                        break;
                case 4: // lower-right
                        CG_DrawPic( 348+38, 376+18, 96, -96, cgs.media.damageArrow3 );
                        CG_DrawDamageMOD( 364+36, 296+16 );
                        break;
                case 5: // down
                        CG_DrawPic( 256, 402, 128, -16, cgs.media.damageArrow2 );
                        CG_DrawDamageMOD( 304, 352 );
                        break;
                case 6: // lower-left
                        CG_DrawPic( 300-48, 376+18, -96, -96, cgs.media.damageArrow3 );
                        CG_DrawDamageMOD( 252-46, 296+16 );
                        break;
                case 7: // left
                        CG_DrawPic( 152, 176, 16, 128, cgs.media.damageArrow1 );
                        CG_DrawDamageMOD( 170, 224 );
                        break;
                 case 8: // upper-left
                        CG_DrawPic( 300-48, 104-18, -96, 96, cgs.media.damageArrow3 );
                        CG_DrawDamageMOD( 252-46, 152-16 );
                        break;
                }
        }

        trap_R_SetColor( NULL );

}

/*
================
CG_DrawStatusBar

================
*/
#ifndef MISSIONPACK
static void CG_DrawStatusBar( void ) {
	int			color;
	centity_t	*cent;
	playerState_t	*ps;
	int			value;
	vec4_t		hcolor;
	vec3_t		angles;
	vec3_t		origin;
#ifdef MISSIONPACK
	qhandle_t	handle;
#endif
	static float colors[4][4] = { 
//		{ 0.2, 1.0, 0.2, 1.0 } , { 1.0, 0.2, 0.2, 1.0 }, {0.5, 0.5, 0.5, 1} };
		{ 1.0f, 0.69f, 0.0f, 1.0f } ,		// normal
		{ 1.0f, 0.2f, 0.2f, 1.0f },		// low health
		{0.5f, 0.5f, 0.5f, 1.0f},			// weapon firing
		{ 1.0f, 1.0f, 1.0f, 1.0f } };			// health > 100

	if ( cg_drawStatus.integer == 0 ) {
		return;
	}
       if ( CG_DrawScoreboard() ) {
              return;
       }

	// draw the team background
	CG_DrawTeamBackground( 0, 420, 640, 60, 0.33f, cg.snap->ps.persistant[PERS_TEAM] );

	cent = &cg_entities[cg.snap->ps.clientNum];
	ps = &cg.snap->ps;

	VectorClear( angles );

	// draw any 3D icons first, so the changes back to 2D are minimized
	if ( cent->currentState.weapon && cg_weapons[ cent->currentState.weapon ].ammoModel ) {
		origin[0] = 70;
		origin[1] = 0;
		origin[2] = 0;
		angles[YAW] = 90 + 20 * sin( cg.time / 1000.0 );
              CG_Draw3DModel( CHAR_WIDTH*3 + TEXT_ICON_SPACE-40, 452, ICON_SIZE/2, ICON_SIZE/2,
                                      cg_weapons[ cent->currentState.weapon ].ammoModel, 0, origin, angles );
	}

	CG_DrawStatusBarHead( 185 + CHAR_WIDTH*3 + TEXT_ICON_SPACE );

	if( cg.predictedPlayerState.powerups[PW_REDFLAG] ) {
		CG_DrawStatusBarFlag( 185 + CHAR_WIDTH*3 + TEXT_ICON_SPACE + ICON_SIZE, TEAM_RED );
	} else if( cg.predictedPlayerState.powerups[PW_BLUEFLAG] ) {
		CG_DrawStatusBarFlag( 185 + CHAR_WIDTH*3 + TEXT_ICON_SPACE + ICON_SIZE, TEAM_BLUE );
	} else if( cg.predictedPlayerState.powerups[PW_NEUTRALFLAG] ) {
		CG_DrawStatusBarFlag( 185 + CHAR_WIDTH*3 + TEXT_ICON_SPACE + ICON_SIZE, TEAM_FREE );
	}

	if ( ps->stats[ STAT_ARMOR ] ) {
		origin[0] = 90;
		origin[1] = 0;
		origin[2] = -10;
		angles[YAW] = ( cg.time & 2047 ) * 360 / 2048.0;
              if (ps->stats[STAT_ARMOR] >= 100)
                     CG_Draw3DModel( 510 + CHAR_WIDTH*3 + TEXT_ICON_SPACE, 452, ICON_SIZE/2, ICON_SIZE/2,
                                             cgs.media.armorModelRed, 0, origin, angles );
              else
                     CG_Draw3DModel( 510 + CHAR_WIDTH*3 + TEXT_ICON_SPACE, 452, ICON_SIZE/2, ICON_SIZE/2,
                                             cgs.media.armorModel, 0, origin, angles );
	}
#ifdef MISSIONPACK
	if( cgs.gametype == GT_HARVESTER ) {
		origin[0] = 90;
		origin[1] = 0;
		origin[2] = -10;
		angles[YAW] = ( cg.time & 2047 ) * 360 / 2048.0;
		if( cg.snap->ps.persistant[PERS_TEAM] == TEAM_BLUE ) {
			handle = cgs.media.redCubeModel;
		} else {
			handle = cgs.media.blueCubeModel;
		}
		CG_Draw3DModel( 640 - (TEXT_ICON_SPACE + ICON_SIZE), 416, ICON_SIZE, ICON_SIZE, handle, 0, origin, angles );
	}
#endif
	//
	// ammo
	//
	if ( cent->currentState.weapon ) {
		value = ps->ammo[cent->currentState.weapon];
              if ( value > -1 && value < 1500) {
			if ( cg.predictedPlayerState.weaponstate == WEAPON_FIRING
				&& cg.predictedPlayerState.weaponTime > 100 ) {
				// draw as dark grey when reloading
				color = 2;	// dark grey
			} else {
				if ( value >= 0 ) {
					color = 0;	// green
				} else {
					color = 1;	// red
				}
			}
			trap_R_SetColor( colors[color] );
			
                     CG_DrawField (0, 452, 3, value);
			trap_R_SetColor( NULL );

			// if we didn't draw a 3D icon, draw a 2D icon for ammo
			if ( !cg_draw3dIcons.integer && cg_drawIcons.integer ) {
				qhandle_t	icon;

				icon = cg_weapons[ cg.predictedPlayerState.weapon ].ammoIcon;
				if ( icon ) {
                                   CG_DrawPic( CHAR_WIDTH*2 + TEXT_ICON_SPACE, 452, ICON_SIZE/2, ICON_SIZE/2, icon );
				}
			}
		}
	}

	//
	// health
	//
	value = ps->stats[STAT_HEALTH];
	if ( value > 100 ) {
		trap_R_SetColor( colors[3] );		// white
	} else if (value > 25) {
		trap_R_SetColor( colors[0] );	// green
	} else if (value > 0) {
		color = (cg.time >> 8) & 1;	// flash
		trap_R_SetColor( colors[color] );
	} else {
		trap_R_SetColor( colors[1] );	// red
	}

	// stretch the health up when taking damage
       CG_DrawField ( 225, 452, 3, value); // 185
	CG_ColorForHealth( hcolor );
	trap_R_SetColor( hcolor );


	//
	// armor
	//
	value = ps->stats[STAT_ARMOR];
	if (value > 0 ) {
		trap_R_SetColor( colors[0] );
              CG_DrawField (550, 452, 3, value);
		trap_R_SetColor( NULL );
		// if we didn't draw a 3D icon, draw a 2D icon for armor
		if ( !cg_draw3dIcons.integer && cg_drawIcons.integer ) {
                     if (value >= 100)
                            CG_DrawPic( 510 + CHAR_WIDTH*3 + TEXT_ICON_SPACE, 452, ICON_SIZE/2, ICON_SIZE/2, cgs.media.armorIconRed );
                     else
                            CG_DrawPic( 510 + CHAR_WIDTH*3 + TEXT_ICON_SPACE, 452, ICON_SIZE/2, ICON_SIZE/2, cgs.media.armorIcon );
		}

	}
#ifdef MISSIONPACK
	//
	// cubes
	//
	if( cgs.gametype == GT_HARVESTER ) {
		value = ps->generic1;
		if( value > 99 ) {
			value = 99;
		}
		trap_R_SetColor( colors[0] );
		CG_DrawField (640 - (CHAR_WIDTH*2 + TEXT_ICON_SPACE + ICON_SIZE), 432, 2, value);
		trap_R_SetColor( NULL );
		// if we didn't draw a 3D icon, draw a 2D icon for armor
		if ( !cg_draw3dIcons.integer && cg_drawIcons.integer ) {
			if( cg.snap->ps.persistant[PERS_TEAM] == TEAM_BLUE ) {
				handle = cgs.media.redCubeIcon;
			} else {
				handle = cgs.media.blueCubeIcon;
			}
			CG_DrawPic( 640 - (TEXT_ICON_SPACE + ICON_SIZE), 432, ICON_SIZE, ICON_SIZE, handle );
		}
	}
#endif
}
#endif

/*
===========================================================================================

  UPPER RIGHT CORNER

===========================================================================================
*/

/*
================
CG_DrawAttacker

================
*/
static float CG_DrawAttacker( float y ) {
	int			t;
	float		size;
	vec3_t		angles;
	const char	*info;
	const char	*name;
	int			clientNum;

	if ( cg.predictedPlayerState.stats[STAT_HEALTH] <= 0 ) {
		return y;
	}

	if ( !cg.attackerTime ) {
		return y;
	}

	clientNum = cg.predictedPlayerState.persistant[PERS_ATTACKER];
	if ( clientNum < 0 || clientNum >= MAX_CLIENTS || clientNum == cg.snap->ps.clientNum ) {
		return y;
	}

	t = cg.time - cg.attackerTime;
	if ( t > ATTACKER_HEAD_TIME ) {
		cg.attackerTime = 0;
		return y;
	}

       size = (ICON_SIZE/2 * 1.25);

	angles[PITCH] = 0;
	angles[YAW] = 180;
	angles[ROLL] = 0;
	CG_DrawHead( 640 - size, y, size, size, clientNum, angles );

	info = CG_ConfigString( CS_PLAYERS + clientNum );
	name = Info_ValueForKey(  info, "n" );
	y += size;
       CG_DrawSmallString( 640 - ( Q_PrintStrlen( name ) * BIGCHAR_WIDTH/2), y, name, 0.5 );

       return y + BIGCHAR_HEIGHT/2 + 2;
}

/*
==================
CG_DrawSnapshot
==================
*/
static float CG_DrawSnapshot( float y ) {
	char		*s;
	int			w;

	s = va( "time:%i snap:%i cmd:%i", cg.snap->serverTime, 
		cg.latestSnapshotNum, cgs.serverCommandSequence );
	w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH;

	CG_DrawBigString( 635 - w, y + 2, s, 1.0F);

	return y + BIGCHAR_HEIGHT + 4;
}

/*
==================
CG_DrawFPS
==================
*/
#define	FPS_FRAMES	4
static float CG_DrawFPS( float y ) {
	char		*s;
	int			w;
	static int	previousTimes[FPS_FRAMES];
	static int	index;
	int		i, total;
	int		fps;
	static	int	previous;
	int		t, frameTime;

	// don't use serverTime, because that will be drifting to
	// correct for internet lag changes, timescales, timedemos, etc
	t = trap_Milliseconds();
	frameTime = t - previous;
	previous = t;

	previousTimes[index % FPS_FRAMES] = frameTime;
	index++;
	if ( index > FPS_FRAMES ) {
		// average multiple frames together to smooth changes out a bit
		total = 0;
		for ( i = 0 ; i < FPS_FRAMES ; i++ ) {
			total += previousTimes[i];
		}
		if ( !total ) {
			total = 1;
		}
		fps = 1000 * FPS_FRAMES / total;

		s = va( "%ifps", fps );
		w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH;

		CG_DrawBigString( 635 - w, y + 2, s, 1.0F);
	}

	return y + BIGCHAR_HEIGHT + 4;
}

/*
=================
CG_DrawTimer
=================
*/
static float CG_DrawTimer( float y ) {
	char		*s;
	int			w;
	int			mins, seconds, tens;
	int			msec;

	msec = cg.time - cgs.levelStartTime;

	seconds = msec / 1000;
	mins = seconds / 60;
	seconds -= mins * 60;
	tens = seconds / 10;
	seconds -= tens * 10;

	s = va( "%i:%i%i", mins, tens, seconds );
	w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH;

	CG_DrawBigString( 635 - w, y + 2, s, 1.0F);

	return y + BIGCHAR_HEIGHT + 4;
}


/*
=================
CG_DrawTeamOverlay
=================
*/

static float CG_DrawTeamOverlay( float y, qboolean right, qboolean upper ) {
	int x, w, h, xx;
	int i, j, len;
	const char *p;
	vec4_t		hcolor;
	int pwidth, lwidth;
	int plyrs;
	char st[16];
	clientInfo_t *ci;
	gitem_t	*item;
	int ret_y, count;

	if ( !cg_drawTeamOverlay.integer ) {
		return y;
	}

	if ( cg.snap->ps.persistant[PERS_TEAM] != TEAM_RED && cg.snap->ps.persistant[PERS_TEAM] != TEAM_BLUE ) {
		return y; // Not on any team
	}

	plyrs = 0;

	// max player name width
	pwidth = 0;
	count = (numSortedTeamPlayers > 8) ? 8 : numSortedTeamPlayers;
	for (i = 0; i < count; i++) {
		ci = cgs.clientinfo + sortedTeamPlayers[i];
		if ( ci->infoValid && ci->team == cg.snap->ps.persistant[PERS_TEAM]) {
			plyrs++;
			len = CG_DrawStrlen(ci->name);
			if (len > pwidth)
				pwidth = len;
		}
	}

	if (!plyrs)
		return y;

	if (pwidth > TEAM_OVERLAY_MAXNAME_WIDTH)
		pwidth = TEAM_OVERLAY_MAXNAME_WIDTH;

	// max location name width
	lwidth = 0;
	for (i = 1; i < MAX_LOCATIONS; i++) {
		p = CG_ConfigString(CS_LOCATIONS + i);
		if (p && *p) {
			len = CG_DrawStrlen(p);
			if (len > lwidth)
				lwidth = len;
		}
	}

	if (lwidth > TEAM_OVERLAY_MAXLOCATION_WIDTH)
		lwidth = TEAM_OVERLAY_MAXLOCATION_WIDTH;

	w = (pwidth + lwidth + 4 + 7) * TINYCHAR_WIDTH;

	if ( right )
		x = 640 - w;
	else
		x = 0;

	h = plyrs * TINYCHAR_HEIGHT;

	if ( upper ) {
		ret_y = y + h;
	} else {
		y -= h;
		ret_y = y;
	}

	if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_RED ) {
		hcolor[0] = 1;
		hcolor[1] = 0;
		hcolor[2] = 0;
		hcolor[3] = 0.33f;
	} else { // if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_BLUE )
		hcolor[0] = 0;
		hcolor[1] = 0;
		hcolor[2] = 1;
		hcolor[3] = 0.33f;
	}
	trap_R_SetColor( hcolor );
	CG_DrawPic( x, y, w, h, cgs.media.teamStatusBar );
	trap_R_SetColor( NULL );

	for (i = 0; i < count; i++) {
		ci = cgs.clientinfo + sortedTeamPlayers[i];
		if ( ci->infoValid && ci->team == cg.snap->ps.persistant[PERS_TEAM]) {

			hcolor[0] = hcolor[1] = hcolor[2] = hcolor[3] = 1.0;

			xx = x + TINYCHAR_WIDTH;

			CG_DrawStringExt( xx, y,
				ci->name, hcolor, qfalse, qfalse,
				TINYCHAR_WIDTH, TINYCHAR_HEIGHT, TEAM_OVERLAY_MAXNAME_WIDTH);

			if (lwidth) {
				p = CG_ConfigString(CS_LOCATIONS + ci->location);
				if (!p || !*p)
					p = "unknown";
				len = CG_DrawStrlen(p);
				if (len > lwidth)
					len = lwidth;

//				xx = x + TINYCHAR_WIDTH * 2 + TINYCHAR_WIDTH * pwidth + 
//					((lwidth/2 - len/2) * TINYCHAR_WIDTH);
				xx = x + TINYCHAR_WIDTH * 2 + TINYCHAR_WIDTH * pwidth;
				CG_DrawStringExt( xx, y,
					p, hcolor, qfalse, qfalse, TINYCHAR_WIDTH, TINYCHAR_HEIGHT,
					TEAM_OVERLAY_MAXLOCATION_WIDTH);
			}

			CG_GetColorForHealth( ci->health, ci->armor, hcolor );

			Com_sprintf (st, sizeof(st), "%3i %3i", ci->health,	ci->armor);

			xx = x + TINYCHAR_WIDTH * 3 + 
				TINYCHAR_WIDTH * pwidth + TINYCHAR_WIDTH * lwidth;

			CG_DrawStringExt( xx, y,
				st, hcolor, qfalse, qfalse,
				TINYCHAR_WIDTH, TINYCHAR_HEIGHT, 0 );

			// draw weapon icon
			xx += TINYCHAR_WIDTH * 3;

			if ( cg_weapons[ci->curWeapon].weaponIcon ) {
				CG_DrawPic( xx, y, TINYCHAR_WIDTH, TINYCHAR_HEIGHT, 
					cg_weapons[ci->curWeapon].weaponIcon );
			} else {
				CG_DrawPic( xx, y, TINYCHAR_WIDTH, TINYCHAR_HEIGHT, 
					cgs.media.deferShader );
			}

			// Draw powerup icons
			if (right) {
				xx = x;
			} else {
				xx = x + w - TINYCHAR_WIDTH;
			}
			for (j = 0; j <= PW_NUM_POWERUPS; j++) {
				if (ci->powerups & (1 << j)) {

					item = BG_FindItemForPowerup( j );

					if (item) {
						CG_DrawPic( xx, y, TINYCHAR_WIDTH, TINYCHAR_HEIGHT, 
						trap_R_RegisterShader( item->icon ) );
						if (right) {
							xx -= TINYCHAR_WIDTH;
						} else {
							xx += TINYCHAR_WIDTH;
						}
					}
				}
			}

			y += TINYCHAR_HEIGHT;
		}
	}

	return ret_y;
//#endif
}


/*
=====================
CG_DrawUpperRight

=====================
*/
static void CG_DrawUpperRight( void ) {
	float	y;

	y = 0;

	if ( cgs.gametype >= GT_TEAM && cg_drawTeamOverlay.integer == 1 ) {
		y = CG_DrawTeamOverlay( y, qtrue, qtrue );
	} 
	if ( cg_drawSnapshot.integer ) {
		y = CG_DrawSnapshot( y );
	}
	if ( cg_drawFPS.integer ) {
		y = CG_DrawFPS( y );
	}
	if ( cg_drawTimer.integer ) {
		y = CG_DrawTimer( y );
	}
	if ( cg_drawAttacker.integer ) {
		y = CG_DrawAttacker( y );
	}

}

/*
=====================
CG_DrawUpperLeft

=====================
*/
static float CG_DrawUpperLeft( float y ) {
//       float  y;

//       y = 0;

	if ( cgs.gametype >= GT_TEAM && cg_drawTeamOverlay.integer == 1 ) {
		y = CG_DrawTeamOverlay( y, qtrue, qtrue );
	} 
	if ( cg_drawSnapshot.integer ) {
		y = CG_DrawSnapshot( y );
	}
	if ( cg_drawFPS.integer ) {
		y = CG_DrawFPS( y );
	}
	if ( cg_drawTimer.integer ) {
		y = CG_DrawTimer( y );
	}
	if ( cg_drawAttacker.integer ) {
		y = CG_DrawAttacker( y );
	}
       return y;
}

/*
===========================================================================================

  LOWER RIGHT CORNER

===========================================================================================
*/

/*
=================
CG_DrawMiniScore

Draw the small scoreboard display
=================
*/
void CG_DrawMiniScore( int y, score_t *score, qboolean smallfont ) {
	char	string[1024];
	float	hcolor[4];
	float	color[4];
	clientInfo_t	*ci;
	int		x;
	int		w;
	int		sw;
	int		sh;
	const char *s;

	ci = &cgs.clientinfo[score->client];

	// draw the score line
	// Use S_COLOR_WHITE so colored names don't change score color.
	Com_sprintf(string, sizeof(string),
		"%s "S_COLOR_WHITE" %3i", ci->name, score->score);

	// colorize line
	if ( ci->team == TEAM_BLUE ) {
		hcolor[0] = 0;
		hcolor[1] = 0;
		hcolor[2] = 1;
	} else if ( ci->team == TEAM_RED ) {
		hcolor[0] = 1;
		hcolor[1] = 0;
		hcolor[2] = 0;
	} else {
		hcolor[0] = 0.7;
		hcolor[1] = 0.7;
		hcolor[2] = 0.7;
	}

	hcolor[3] = 0.33;

	if(smallfont)
	{
		sw = TINYCHAR_WIDTH;
		sh = TINYCHAR_HEIGHT;
	}
	else
	{
		sw = SMALLCHAR_WIDTH;
		sh = SMALLCHAR_HEIGHT;
	}

	x = 640;
	s = va( "%s", string );
	w = CG_DrawStrlen( s ) * sw + 8;
	x -= w;

	CG_FillRect( x+1, y, w, sh+1, hcolor );

	if(smallfont)
	{
		color[0] = color[1] = color[2] = color[3] = 1.0;
		CG_DrawStringExt( x, y, string, color, qtrue, qfalse, TINYCHAR_WIDTH, TINYCHAR_HEIGHT, 0 );
	}
	else
	{
		CG_DrawSmallString( x, y, string, 1.0 );
	}
}

int CG_MakeMiniScoreboard( int y)
{
	int		i;
	score_t	*score;
	int		count;
	int players;
	clientInfo_t	*ci;
	qboolean smallfont;
	int lineHeight;

	players = 0;
	count = 0;

	for ( i = 0 ; i < cg.numScores ; i++ )
	{
		score = &cg.scores[i];
		ci = &cgs.clientinfo[ score->client ];

		if ( score->client < 0 || score->client >= cgs.maxclients ) {
			Com_Printf( "Bad score->client: %i\n", score->client );
			return 0;
		}

		// Don't show connecting people or spectators
		if ( score->ping == -1 || ci->team == TEAM_SPECTATOR ) {
			continue;
		}

		players++;
	}

	if (!players)
		return 0;

	if(players > 7) {
		lineHeight = (TINYCHAR_HEIGHT - 4)*2+1;
		smallfont = qtrue;
	}
	else {
              lineHeight = (SMALLCHAR_HEIGHT - 8)*2+1;
		smallfont = qfalse;
	}	

	// draw (at most) the top 11 players
	for ( i = 0 ; i < cg.numScores && count < 11 ; i++ )
	{
		score = &cg.scores[i];
		ci = &cgs.clientinfo[ score->client ];
		if ( score->ping == -1 || ci->team == TEAM_SPECTATOR ) {
			continue;
		}

              CG_DrawMiniScore( y + lineHeight * count, score, smallfont);
		count++;
	}

	return count;
}

/*
=================
CG_DrawScores

Draw the small two score display
=================
*/
#ifndef MISSIONPACK
static float CG_DrawScores( float y ) {
	const char	*s;
	int			s1, s2, score;
	int			x, w;
	int			v;
       int                  oldy;
	vec4_t		color;
	float		y1;
	gitem_t		*item;

       if ( CG_DrawScoreboard() ) {
              return;
       }

	s1 = cgs.scores1;
	s2 = cgs.scores2;
       if (cg_hud.integer == 5)
       {
              oldy = y;
              oldy -=  BIGCHAR_HEIGHT + 8;
              oldy -=  BIGCHAR_HEIGHT * 12; // * 12
              y = 457;
       }
       else
       {
              y -=  BIGCHAR_HEIGHT + 8;
              y -=  BIGCHAR_HEIGHT * 12; // * 12
       }
       
	y1 = y;

	// draw from the right side to left
	if ( cgs.gametype >= GT_TEAM ) {
		x = 640;

		color[0] = 0;
		color[1] = 0;
		color[2] = 1;
		color[3] = 0.33f;
		s = va( "%2i", s2 );
		w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
		x -= w;
//              if (cg_hud.integer == 5)
//                     x -= 5;
              CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
		if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_BLUE ) {
			CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
		}
		CG_DrawBigString( x + 4, y, s, 1.0F);

              if ( cgs.gametype == GT_CTF && (cg_hud.integer == 0 || cg_hud.integer == 4)) {
			// Display flag status
			item = BG_FindItemForPowerup( PW_BLUEFLAG );

			if (item) {
				y1 = y - BIGCHAR_HEIGHT - 8;
				if( cgs.blueflag >= 0 && cgs.blueflag <= 2 ) {
					CG_DrawPic( x, y1-4, w, BIGCHAR_HEIGHT+8, cgs.media.blueFlagShader[cgs.blueflag] );
				}
			}
		}

		color[0] = 1;
		color[1] = 0;
		color[2] = 0;
		color[3] = 0.33f;
		s = va( "%2i", s1 );
		w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
		x -= w;
//              if (cg_hud.integer == 5)
//                     x -= 5;
		CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
		if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_RED ) {
			CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
		}
		CG_DrawBigString( x + 4, y, s, 1.0F);

              if ( cgs.gametype == GT_CTF && (cg_hud.integer == 0 || cg_hud.integer == 4)) {
			// Display flag status
			item = BG_FindItemForPowerup( PW_REDFLAG );

			if (item) {
				y1 = y - BIGCHAR_HEIGHT - 8;
				if( cgs.redflag >= 0 && cgs.redflag <= 2 ) {
					CG_DrawPic( x, y1-4, w, BIGCHAR_HEIGHT+8, cgs.media.redFlagShader[cgs.redflag] );
				}
			}
		}

#ifdef MISSIONPACK
		if ( cgs.gametype == GT_1FCTF ) {
			// Display flag status
			item = BG_FindItemForPowerup( PW_NEUTRALFLAG );

			if (item) {
				y1 = y - BIGCHAR_HEIGHT - 8;
				if( cgs.flagStatus >= 0 && cgs.flagStatus <= 3 ) {
					CG_DrawPic( x, y1-4, w, BIGCHAR_HEIGHT+8, cgs.media.flagShader[cgs.flagStatus] );
				}
			}
		}
#endif
		if ( cgs.gametype >= GT_CTF ) {
			v = cgs.capturelimit;
		} else {
			v = cgs.fraglimit;
		}
		if ( v ) {
			s = va( "%2i", v );
			w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
			x -= w;
                     if (cg_hud.integer == 5)
                            x -= 5;
			CG_DrawBigString( x + 4, y, s, 1.0F);
		}

	} else {
		qboolean	spectator;

		x = 640;
		score = cg.snap->ps.persistant[PERS_SCORE];
		spectator = ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR );

		// always show your score in the second box if not in first place
		if ( s1 != score ) {
			s2 = score;
		}
		if ( s2 != SCORE_NOT_PRESENT ) {
			s = va( "%2i", s2 );
			w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
			x -= w;
                     if (cg_hud.integer == 5)
                            x -= 5;
			if ( !spectator && score == s2 && score != s1 ) {
				color[0] = 1;
				color[1] = 0;
				color[2] = 0;
				color[3] = 0.33f;
				CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
				CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
			} else {
				color[0] = 0.5f;
				color[1] = 0.5f;
				color[2] = 0.5f;
				color[3] = 0.33f;
				CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
			}	
			CG_DrawBigString( x + 4, y, s, 1.0F);
		}

		// first place
		if ( s1 != SCORE_NOT_PRESENT ) {
			s = va( "%2i", s1 );
			w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
			x -= w;
//                     if (cg_hud.integer == 5)
//                            x -= 3;
			if ( !spectator && score == s1 ) {
				color[0] = 0;
				color[1] = 0;
				color[2] = 1;
				color[3] = 0.33f;
				CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
				CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
			} else {
				color[0] = 0.5f;
				color[1] = 0.5f;
				color[2] = 0.5f;
				color[3] = 0.33f;
				CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
			}	
			CG_DrawBigString( x + 4, y, s, 1.0F);
		}

		if ( cgs.fraglimit ) {
			s = va( "%2i", cgs.fraglimit );
			w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
			x -= w;
                     if (cg_hud.integer == 5)
                           x -= 5;
			CG_DrawBigString( x + 4, y, s, 1.0F);
		}

	}
       if ( cg.scoresRequestTime + 2000 < cg.time )
       {
            // the scores are more than five seconds out of date,
			// so request new ones
			cg.scoresRequestTime = cg.time;
			trap_SendClientCommand( "score" );
        }
        if (cg_drawMiniScore.integer)
        {
               if (cg_hud.integer == 5)
                     y = oldy+10;
               CG_MakeMiniScoreboard( y + ((BIGCHAR_HEIGHT - 8)*2)+8 ); // *2
        }

	return y1 - 8;
}
#endif

static float CG_DrawScores2( float y ) {
	const char	*s;
       int                  s1, s2,s3, score;
	int			x, w;
	int			v;
       int                  oldy;
       float  hcolor[4];
	vec4_t		color;
	float		y1;
              qboolean      spectator;
	gitem_t		*item;

       if ( CG_DrawScoreboard() ) {
              return;
       }

	s1 = cgs.scores1;
	s2 = cgs.scores2;
       oldy = y;
       oldy -=  SMALLCHAR_HEIGHT + 8;
       oldy -=  SMALLCHAR_HEIGHT * 12; // * 12
       y = 457;
	y1 = y;

              x = 605;
		score = cg.snap->ps.persistant[PERS_SCORE];
              s3 = score;
		spectator = ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR );
              if (spectator)
                     y = 439;
// team / ctf
	// draw from the right side to left
	if ( cgs.gametype >= GT_TEAM ) {
              x = 640;
		color[0] = 0;
		color[1] = 0;
		color[2] = 1;
		color[3] = 0.33f;
              s = va( " Blue:%-3i ", s2 );
              w = CG_DrawStrlen( s ) * SMALLCHAR_WIDTH + 8;
		x -= w;
              CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
		if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_BLUE ) {
                     CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
		}
//              CG_DrawBigString( x + 4, y, s, 1.0F);
              CG_DrawStringExt( x + 4, y, s, 
                                 colorWhite, qfalse, qtrue, SMALLCHAR_WIDTH,SMALLCHAR_HEIGHT, 0 );


              if ( cgs.gametype == GT_CTF && (cg_hud.integer == 0 || cg_hud.integer == 4)) {
			// Display flag status
			item = BG_FindItemForPowerup( PW_BLUEFLAG );

			if (item) {
				y1 = y - BIGCHAR_HEIGHT - 8;
				if( cgs.blueflag >= 0 && cgs.blueflag <= 2 ) {
					CG_DrawPic( x, y1-4, w, BIGCHAR_HEIGHT+8, cgs.media.blueFlagShader[cgs.blueflag] );
				}
			}
		}

		color[0] = 1;
		color[1] = 0;
		color[2] = 0;
		color[3] = 0.33f;
              s = va( " Red:%-3i ", s1 );
              w = CG_DrawStrlen( s ) * SMALLCHAR_WIDTH + 8;
		x -= w;
              CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
		if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_RED ) {
                     CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
		}
//              CG_DrawBigString( x + 4, y, s, 1.0F);
              CG_DrawStringExt( x + 4, y, s, 
                                 colorWhite, qfalse, qtrue, SMALLCHAR_WIDTH,SMALLCHAR_HEIGHT, 0 );


              if ( cgs.gametype == GT_CTF && (cg_hud.integer == 0 || cg_hud.integer == 4)) {
			// Display flag status
			item = BG_FindItemForPowerup( PW_REDFLAG );

			if (item) {
				y1 = y - BIGCHAR_HEIGHT - 8;
				if( cgs.redflag >= 0 && cgs.redflag <= 2 ) {
					CG_DrawPic( x, y1-4, w, BIGCHAR_HEIGHT+8, cgs.media.redFlagShader[cgs.redflag] );
				}
			}
		}

		if ( cgs.gametype >= GT_CTF ) {
			v = cgs.capturelimit;
		} else {
			v = cgs.fraglimit;
		}
		if ( v ) {
			s = va( "%2i", v );
                     w = CG_DrawStrlen( s ) * SMALLCHAR_WIDTH + 8;
			x -= w;
//                     CG_DrawBigString( x + 4, y, s, 1.0F);
              CG_DrawStringExt( x + 4, y, s, 
                                 colorWhite, qfalse, qtrue, SMALLCHAR_WIDTH,SMALLCHAR_HEIGHT, 0 );
		}
}
// ffa / tourney
else
{
		// always show your score in the second box if not in first place
                     if ( s2 == SCORE_NOT_PRESENT )
                            s2 = 0;
                     s = va( "2nd:%-3i", s2 );
                     w = CG_DrawStrlen( s ) * SMALLCHAR_WIDTH + 8;
			x -= w;
                     if (cg_hud.integer == 5)
                            x -= 5;
                     hcolor[0] = hcolor[3] = 1;
                     hcolor[2] = hcolor[1] = 0;
                     CG_DrawStringExt( x + 4, y, s,
                                       hcolor, qfalse, qtrue, SMALLCHAR_WIDTH,SMALLCHAR_HEIGHT, 0 );
//              }
		// first place
                     if ( s1 == SCORE_NOT_PRESENT )
                            s1 = 0;
                     s = va( "1st:%-3i", s1 );
                     w = CG_DrawStrlen( s ) * SMALLCHAR_WIDTH + 8;
			x -= w;
//                     hcolor[0] = hcolor[2] = hcolor[3] = 1;
//                     hcolor[1] = 0.3;
/*
                     hcolor[0] = 1;
                     hcolor[1] = 0.5;
                     hcolor[2] = 1;
                     hcolor[3] = 1;
*/
              hcolor[0] = .25;
              hcolor[1] = .25;
              hcolor[2] = 1;
              hcolor[3] = 1; // 0.5
              trap_R_SetColor( hcolor );
                     CG_DrawStringExt( x + 4, y, s, 
                                       hcolor, qfalse, qtrue, SMALLCHAR_WIDTH,SMALLCHAR_HEIGHT, 0 );
              trap_R_SetColor( NULL );
		if ( cgs.fraglimit ) {
                     hcolor[0] = hcolor[3] = 1;
                     hcolor[2] = hcolor[1] = 0;
                     s = va( "%2i", cgs.fraglimit );
                     w = CG_DrawStrlen( s ) * SMALLCHAR_WIDTH + 8;
			x -= w;
                     if (cg_hud.integer == 5)
                           x -= 5;
                     CG_DrawStringExt( x + 4, y, s, 
                                       colorWhite, qfalse, qtrue, SMALLCHAR_WIDTH,SMALLCHAR_HEIGHT, 0 );
		}
              // own score
                     s = va( "%-3i", s3 );
                     w = CG_DrawStrlen( s ) * SMALLCHAR_WIDTH + 8;
			x -= w;
                     CG_DrawStringExt( 600 + 4, y, s, 
                                       colorWhite, qfalse, qtrue, SMALLCHAR_WIDTH,SMALLCHAR_HEIGHT, 0 );

}
       if ( cg.scoresRequestTime + 2000 < cg.time )
       {
            // the scores are more than five seconds out of date,
			// so request new ones
			cg.scoresRequestTime = cg.time;
			trap_SendClientCommand( "score" );
        }
        if (cg_drawMiniScore.integer)
        {
               if (cg_hud.integer == 5)
                     y = oldy+10;
               CG_MakeMiniScoreboard( y + ((BIGCHAR_HEIGHT - 8)*2)+8 ); // *2
        }
	return y1 - 8;
}

/*
================
CG_DrawPowerups
================
*/
#ifndef MISSIONPACK
static float CG_DrawPowerups( float y ) {
	int		sorted[MAX_POWERUPS];
	int		sortedTime[MAX_POWERUPS];
	int		i, j, k;
	int		active;
	playerState_t	*ps;
	int		t;
	gitem_t	*item;
	int		x;
	int		color;
	float	size;
	float	f;
	static float colors[2][4] = { 
		{ 0.2f, 1.0f, 0.2f, 1.0f } , { 1.0f, 0.2f, 0.2f, 1.0f } };

       if ( CG_DrawScoreboard() ) {
              return;
       }
	ps = &cg.snap->ps;

	if ( ps->stats[STAT_HEALTH] <= 0 ) {
		return y;
	}

	// sort the list by time remaining
	active = 0;
	for ( i = 0 ; i < MAX_POWERUPS ; i++ ) {
		if ( !ps->powerups[ i ] ) {
			continue;
		}
		t = ps->powerups[ i ] - cg.time;
		// ZOID--don't draw if the power up has unlimited time (999 seconds)
		// This is true of the CTF flags
		if ( t < 0 || t > 999000) {
			continue;
		}

		// insert into the list
		for ( j = 0 ; j < active ; j++ ) {
			if ( sortedTime[j] >= t ) {
				for ( k = active - 1 ; k >= j ; k-- ) {
					sorted[k+1] = sorted[k];
					sortedTime[k+1] = sortedTime[k];
				}
				break;
			}
		}
		sorted[j] = i;
		sortedTime[j] = t;
		active++;
	}

	// draw the icons and timers
       x = 640 - ICON_SIZE/2 - CHAR_WIDTH * 1; // 2
	for ( i = 0 ; i < active ; i++ ) {
		item = BG_FindItemForPowerup( sorted[i] );

    if (item) {

		  color = 1;

		  y -= ICON_SIZE;

		  trap_R_SetColor( colors[color] );
                CG_DrawField( x, y/2 + 90 , 2, sortedTime[ i ] / 1000 );

		  t = ps->powerups[ sorted[i] ];
		  if ( t - cg.time >= POWERUP_BLINKS * POWERUP_BLINK_TIME ) {
			  trap_R_SetColor( NULL );
		  } else {
			  vec4_t	modulate;

			  f = (float)( t - cg.time ) / POWERUP_BLINK_TIME;
			  f -= (int)f;
			  modulate[0] = modulate[1] = modulate[2] = modulate[3] = f;
			  trap_R_SetColor( modulate );
		  }

		  if ( cg.powerupActive == sorted[i] && 
			  cg.time - cg.powerupTime < PULSE_TIME ) {
			  f = 1.0 - ( ( (float)cg.time - cg.powerupTime ) / PULSE_TIME );
			  size = ICON_SIZE * ( 1.0 + ( PULSE_SCALE - 1.0 ) * f );
		  } else {
			  size = ICON_SIZE;
		  }

              CG_DrawPic( 640 - size/2, y/2 + 90 + ICON_SIZE / 2 - size / 2, 
                     size/2, size/2, trap_R_RegisterShader( item->icon ) );
    }
	}
	trap_R_SetColor( NULL );

	return y;
}
#endif

/*
=====================
CG_DrawLowerRight

=====================
*/
#ifndef MISSIONPACK
static void CG_DrawLowerRight( void ) {
	float	y;

	y = 480 - ICON_SIZE;

	if ( cgs.gametype >= GT_TEAM && cg_drawTeamOverlay.integer == 2 ) {
		y = CG_DrawTeamOverlay( y, qtrue, qfalse );
	} 

	y = CG_DrawScores( y );
	y = CG_DrawPowerups( y );
}
#endif

/*
===================
CG_DrawPickupItem
===================
*/
#ifndef MISSIONPACK
static int CG_DrawPickupItem( int y ) {
	int		value;
	float	*fadeColor;

	if ( cg.snap->ps.stats[STAT_HEALTH] <= 0 ) {
		return y;
	}

	y -= ICON_SIZE;

	value = cg.itemPickup;
	if ( value ) {
		fadeColor = CG_FadeColor( cg.itemPickupTime, 3000 );
		if ( fadeColor ) {
			CG_RegisterItemVisuals( value );
			trap_R_SetColor( fadeColor );
                     if (cg_hud.integer == 1)
                     {
                     /*
                            if (cg_showAllWeapons.integer)
                            {
                                   CG_DrawPic( 8, 400, ICON_SIZE/2.5, ICON_SIZE/2.5, cg_items[ value ].icon );
                                   CG_DrawSmallString( ICON_SIZE/2 + 8, 385 + (ICON_SIZE/2 - SMALLCHAR_HEIGHT/2), bg_itemlist[ value ].pickup_name, fadeColor[0] );
                            }
                            else
                            {
                     */
                                   CG_DrawPic( 8, y+18, ICON_SIZE/2.5, ICON_SIZE/2.5, cg_items[ value ].icon );
                                   CG_DrawBigString( ICON_SIZE/2 + 16, y+3 + (ICON_SIZE/2 - BIGCHAR_HEIGHT/2), bg_itemlist[ value ].pickup_name, fadeColor[0] );
//                            }
                     }
                     else if (cg_hud.integer == 5)
                     {
                            CG_DrawPic( 8, 455, ICON_SIZE/2.5, ICON_SIZE/2.5, cg_items[ value ].icon );
                            CG_DrawSmallString( ICON_SIZE/2 + 8, 440 + (ICON_SIZE/2 - SMALLCHAR_HEIGHT/2), bg_itemlist[ value ].pickup_name, fadeColor[0] );
                     }
                     else
                     {
                     /*
                            if (cg_showAllWeapons.integer)
                            {
                                   CG_DrawPic( 8, y+39, ICON_SIZE/2.5, ICON_SIZE/2.5, cg_items[ value ].icon );
                                   CG_DrawSmallString( ICON_SIZE/2 + 8, y+23 + (ICON_SIZE/2 - SMALLCHAR_HEIGHT/2), bg_itemlist[ value ].pickup_name, fadeColor[0] );
                            }
                            else
                            {
                     */
                                   CG_DrawPic( 8, y+39, ICON_SIZE/2.5, ICON_SIZE/2.5, cg_items[ value ].icon );
                                   CG_DrawBigString( ICON_SIZE/2 + 16, y+23 + (ICON_SIZE/2 - BIGCHAR_HEIGHT/2), bg_itemlist[ value ].pickup_name, fadeColor[0] );
//                            }
                     }
			trap_R_SetColor( NULL );
		}
	}
	
	return y;
}
#endif

/*
=====================
CG_DrawLowerLeft

=====================
*/
#ifndef MISSIONPACK
static void CG_DrawLowerLeft( void ) {
	float	y;

	y = 480 - ICON_SIZE;

	if ( cgs.gametype >= GT_TEAM && cg_drawTeamOverlay.integer == 3 ) {
		y = CG_DrawTeamOverlay( y, qfalse, qfalse );
	} 


	y = CG_DrawPickupItem( y );
}
#endif


//===========================================================================================

/*
=================
CG_DrawTeamInfo
=================
*/
#ifndef MISSIONPACK
static void CG_DrawTeamInfo( void ) {
	int w, h;
	int i, len;
	vec4_t		hcolor;
	int		chatHeight;

#define CHATLOC_Y 420 // bottom end
#define CHATLOC_X 0

	if (cg_teamChatHeight.integer < TEAMCHAT_HEIGHT)
		chatHeight = cg_teamChatHeight.integer;
	else
		chatHeight = TEAMCHAT_HEIGHT;
	if (chatHeight <= 0)
		return; // disabled

	if (cgs.teamLastChatPos != cgs.teamChatPos) {
		if (cg.time - cgs.teamChatMsgTimes[cgs.teamLastChatPos % chatHeight] > cg_teamChatTime.integer) {
			cgs.teamLastChatPos++;
		}

		h = (cgs.teamChatPos - cgs.teamLastChatPos) * TINYCHAR_HEIGHT;

		w = 0;

		for (i = cgs.teamLastChatPos; i < cgs.teamChatPos; i++) {
			len = CG_DrawStrlen(cgs.teamChatMsgs[i % chatHeight]);
			if (len > w)
				w = len;
		}
		w *= TINYCHAR_WIDTH;
		w += TINYCHAR_WIDTH * 2;

		if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_RED ) {
			hcolor[0] = 1;
			hcolor[1] = 0;
			hcolor[2] = 0;
			hcolor[3] = 0.33f;
		} else if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_BLUE ) {
			hcolor[0] = 0;
			hcolor[1] = 0;
			hcolor[2] = 1;
			hcolor[3] = 0.33f;
		} else {
			hcolor[0] = 0;
			hcolor[1] = 1;
			hcolor[2] = 0;
			hcolor[3] = 0.33f;
		}

		trap_R_SetColor( hcolor );
		CG_DrawPic( CHATLOC_X, CHATLOC_Y - h, 640, h, cgs.media.teamStatusBar );
		trap_R_SetColor( NULL );

		hcolor[0] = hcolor[1] = hcolor[2] = 1.0;
		hcolor[3] = 1.0;

		for (i = cgs.teamChatPos - 1; i >= cgs.teamLastChatPos; i--) {
			CG_DrawStringExt( CHATLOC_X + TINYCHAR_WIDTH, 
				CHATLOC_Y - (cgs.teamChatPos - i)*TINYCHAR_HEIGHT, 
				cgs.teamChatMsgs[i % chatHeight], hcolor, qfalse, qfalse,
				TINYCHAR_WIDTH, TINYCHAR_HEIGHT, 0 );
		}
	}
}
#endif

/*
===================
CG_DrawHoldableItem
===================
*/
#ifndef MISSIONPACK
static void CG_DrawHoldableItem( void ) { 
	int		value;

	value = cg.snap->ps.stats[STAT_HOLDABLE_ITEM];
	if ( value ) {
		CG_RegisterItemVisuals( value );
              CG_DrawPic( 510 + CHAR_WIDTH*3 + TEXT_ICON_SPACE, (SCREEN_HEIGHT-ICON_SIZE*1.2), ICON_SIZE/2, ICON_SIZE/2, cg_items[ value ].icon );
	}

}
#endif

/*
===================
CG_DrawModule
Matchmod Willi - runes
===================
*/
static void CG_DrawModule (void)
{ 
	float *fadecolor;
	char	string[1024];
       int num = 375;
	fadecolor = colorWhite;
	Com_sprintf( string, sizeof( string ), "Rune:" );
       if (cg_hud.integer == 5)
              num = 432;
       CG_DrawSmallStringColor( 5, num, string, fadecolor );

	fadecolor = colorGreen;

	if (cg.snap->ps.stats[STAT_MODULE] == BG_FindItemForModule (MD_AMPLIFIER) - bg_itemlist)
              Com_sprintf( string, sizeof( string ), "Amplifier" );
	else if (cg.snap->ps.stats[STAT_MODULE] == BG_FindItemForModule (MD_DAMPENER) - bg_itemlist)
		Com_sprintf( string, sizeof( string ), "Resist" );
	else if (cg.snap->ps.stats[STAT_MODULE] == BG_FindItemForModule (MD_HEALER) - bg_itemlist)
		Com_sprintf( string, sizeof( string ), "Auto Doc" );
	else if (cg.snap->ps.stats[STAT_MODULE] == BG_FindItemForModule (MD_BOOSTER) - bg_itemlist)
		Com_sprintf( string, sizeof( string ), "Haste" );
	else if (cg.snap->ps.stats[STAT_MODULE] == BG_FindItemForModule (MD_LIFTER) - bg_itemlist)
		Com_sprintf( string, sizeof( string ), "Anti Gravity" );
	else if (cg.snap->ps.stats[STAT_MODULE] == BG_FindItemForModule (MD_CLOAKER) - bg_itemlist)
		Com_sprintf( string, sizeof( string ), "Cloak" );
	else
		Com_sprintf( string, sizeof( string ), "None" );

       CG_DrawSmallStringColor( 55, num, string, fadecolor );
}


#ifdef MISSIONPACK
/*
===================
CG_DrawPersistantPowerup
===================
*/
/*
static void CG_DrawPersistantPowerup( void ) { 
	int		value;

	value = cg.snap->ps.stats[STAT_PERSISTANT_POWERUP];
	if ( value ) {
		CG_RegisterItemVisuals( value );
		CG_DrawPic( 640-ICON_SIZE, (SCREEN_HEIGHT-ICON_SIZE)/2 - ICON_SIZE, ICON_SIZE, ICON_SIZE, cg_items[ value ].icon );
	}
}
*/

#endif

/*
===================
CG_DrawModule for The SARACEN's Modules
===================
*/
static void CG_DrawModule2 (void)
{ 
	int		value;

	value =  cg.snap->ps.stats[STAT_MODULE];
	if (value)
	{
		CG_RegisterItemVisuals (value);
		CG_DrawPic (640-ICON_SIZE, ((SCREEN_HEIGHT-ICON_SIZE)/2)+(ICON_SIZE*1.5), ICON_SIZE, ICON_SIZE, cg_items[value].icon);
	}
}


/*
===================
CG_DrawReward
===================
*/
static void CG_DrawReward( void ) { 
	float	*color;
	int		i, count;
	float	x, y;
	char	buf[32];

	if ( !cg_drawRewards.integer ) {
		return;
	}

	color = CG_FadeColor( cg.rewardTime, REWARD_TIME );
	if ( !color ) {
		if (cg.rewardStack > 0) {
			for(i = 0; i < cg.rewardStack; i++) {
				cg.rewardSound[i] = cg.rewardSound[i+1];
				cg.rewardShader[i] = cg.rewardShader[i+1];
				cg.rewardCount[i] = cg.rewardCount[i+1];
			}
			cg.rewardTime = cg.time;
			cg.rewardStack--;
			color = CG_FadeColor( cg.rewardTime, REWARD_TIME );
			trap_S_StartLocalSound(cg.rewardSound[0], CHAN_ANNOUNCER);
		} else {
			return;
		}
	}

	trap_R_SetColor( color );

	/*
	count = cg.rewardCount[0]/10;				// number of big rewards to draw

	if (count) {
		y = 4;
		x = 320 - count * ICON_SIZE;
		for ( i = 0 ; i < count ; i++ ) {
			CG_DrawPic( x, y, (ICON_SIZE*2)-4, (ICON_SIZE*2)-4, cg.rewardShader[0] );
			x += (ICON_SIZE*2);
		}
	}

	count = cg.rewardCount[0] - count*10;		// number of small rewards to draw
	*/

	if ( cg.rewardCount[0] >= 10 ) {
		y = 56;
		x = 320 - ICON_SIZE/2;
		CG_DrawPic( x, y, ICON_SIZE-4, ICON_SIZE-4, cg.rewardShader[0] );
		Com_sprintf(buf, sizeof(buf), "%d", cg.rewardCount[0]);
		x = ( SCREEN_WIDTH - SMALLCHAR_WIDTH * CG_DrawStrlen( buf ) ) / 2;
		CG_DrawStringExt( x, y+ICON_SIZE, buf, color, qfalse, qtrue,
								SMALLCHAR_WIDTH, SMALLCHAR_HEIGHT, 0 );
	}
	else {

		count = cg.rewardCount[0];

		y = 56;
		x = 320 - count * ICON_SIZE/2;
		for ( i = 0 ; i < count ; i++ ) {
			CG_DrawPic( x, y, ICON_SIZE-4, ICON_SIZE-4, cg.rewardShader[0] );
			x += ICON_SIZE;
		}
	}
	trap_R_SetColor( NULL );
}


/*
===============================================================================

LAGOMETER

===============================================================================
*/

#define	LAG_SAMPLES		128


typedef struct {
	int		frameSamples[LAG_SAMPLES];
	int		frameCount;
	int		snapshotFlags[LAG_SAMPLES];
	int		snapshotSamples[LAG_SAMPLES];
	int		snapshotCount;
} lagometer_t;

lagometer_t		lagometer;

/*
==============
CG_AddLagometerFrameInfo

Adds the current interpolate / extrapolate bar for this frame
==============
*/
void CG_AddLagometerFrameInfo( void ) {
	int			offset;

	offset = cg.time - cg.latestSnapshotTime;
	lagometer.frameSamples[ lagometer.frameCount & ( LAG_SAMPLES - 1) ] = offset;
	lagometer.frameCount++;
}

/*
==============
CG_AddLagometerSnapshotInfo

Each time a snapshot is received, log its ping time and
the number of snapshots that were dropped before it.

Pass NULL for a dropped packet.
==============
*/
void CG_AddLagometerSnapshotInfo( snapshot_t *snap ) {
	// dropped packet
	if ( !snap ) {
		lagometer.snapshotSamples[ lagometer.snapshotCount & ( LAG_SAMPLES - 1) ] = -1;
		lagometer.snapshotCount++;
		return;
	}

	// add this snapshot's info
	lagometer.snapshotSamples[ lagometer.snapshotCount & ( LAG_SAMPLES - 1) ] = snap->ping;
	lagometer.snapshotFlags[ lagometer.snapshotCount & ( LAG_SAMPLES - 1) ] = snap->snapFlags;
	lagometer.snapshotCount++;
}

/*
==============
CG_DrawDisconnect

Should we draw something differnet for long lag vs no packets?
==============
*/
static void CG_DrawDisconnect( void ) {
	float		x, y;
	int			cmdNum;
	usercmd_t	cmd;
	const char		*s;
	int			w;

	// draw the phone jack if we are completely past our buffers
	cmdNum = trap_GetCurrentCmdNumber() - CMD_BACKUP + 1;
	trap_GetUserCmd( cmdNum, &cmd );
	if ( cmd.serverTime <= cg.snap->ps.commandTime
		|| cmd.serverTime > cg.time ) {	// special check for map_restart
		return;
	}

	// also add text in center of screen
	s = "Connection Interrupted";
	w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH;
	CG_DrawBigString( 320 - w/2, 100, s, 1.0F);

	// blink the icon
	if ( ( cg.time >> 9 ) & 1 ) {
		return;
	}

	x = 640 - 48;
	y = 480 - 48;

	CG_DrawPic( x, y, 48, 48, trap_R_RegisterShader("gfx/2d/net.tga" ) );
}


#define	MAX_LAGOMETER_PING	900
#define	MAX_LAGOMETER_RANGE	300

/*
==============
CG_DrawLagometer
==============
*/
static void CG_DrawLagometer( void ) {
	int		a, x, y, i;
	float	v;
	float	ax, ay, aw, ah, mid, range;
	int		color;
	float	vscale;

	if ( !cg_lagometer.integer || cgs.localServer ) {
		CG_DrawDisconnect();
		return;
	}

	//
	// draw the graph
	//
#ifdef MISSIONPACK
	x = 640 - 48;
	y = 480 - 144;
#else
	x = 640 - 48;
	y = 480 - 48;
#endif
       if (cg_hud.integer == 5)
       y = 480 - 96;

	trap_R_SetColor( NULL );
	CG_DrawPic( x, y, 48, 48, cgs.media.lagometerShader );

	ax = x;
	ay = y;
	aw = 48;
	ah = 48;
	CG_AdjustFrom640( &ax, &ay, &aw, &ah );

	color = -1;
	range = ah / 3;
	mid = ay + range;

	vscale = range / MAX_LAGOMETER_RANGE;

	// draw the frame interpoalte / extrapolate graph
	for ( a = 0 ; a < aw ; a++ ) {
		i = ( lagometer.frameCount - 1 - a ) & (LAG_SAMPLES - 1);
		v = lagometer.frameSamples[i];
		v *= vscale;
		if ( v > 0 ) {
			if ( color != 1 ) {
				color = 1;
				trap_R_SetColor( g_color_table[ColorIndex(COLOR_YELLOW)] );
			}
			if ( v > range ) {
				v = range;
			}
			trap_R_DrawStretchPic ( ax + aw - a, mid - v, 1, v, 0, 0, 0, 0, cgs.media.whiteShader );
		} else if ( v < 0 ) {
			if ( color != 2 ) {
				color = 2;
				trap_R_SetColor( g_color_table[ColorIndex(COLOR_BLUE)] );
			}
			v = -v;
			if ( v > range ) {
				v = range;
			}
			trap_R_DrawStretchPic( ax + aw - a, mid, 1, v, 0, 0, 0, 0, cgs.media.whiteShader );
		}
	}

	// draw the snapshot latency / drop graph
	range = ah / 2;
	vscale = range / MAX_LAGOMETER_PING;

	for ( a = 0 ; a < aw ; a++ ) {
		i = ( lagometer.snapshotCount - 1 - a ) & (LAG_SAMPLES - 1);
		v = lagometer.snapshotSamples[i];
		if ( v > 0 ) {
			if ( lagometer.snapshotFlags[i] & SNAPFLAG_RATE_DELAYED ) {
				if ( color != 5 ) {
					color = 5;	// YELLOW for rate delay
					trap_R_SetColor( g_color_table[ColorIndex(COLOR_YELLOW)] );
				}
			} else {
				if ( color != 3 ) {
					color = 3;
					trap_R_SetColor( g_color_table[ColorIndex(COLOR_GREEN)] );
				}
			}
			v = v * vscale;
			if ( v > range ) {
				v = range;
			}
			trap_R_DrawStretchPic( ax + aw - a, ay + ah - v, 1, v, 0, 0, 0, 0, cgs.media.whiteShader );
		} else if ( v < 0 ) {
			if ( color != 4 ) {
				color = 4;		// RED for dropped snapshots
				trap_R_SetColor( g_color_table[ColorIndex(COLOR_RED)] );
			}
			trap_R_DrawStretchPic( ax + aw - a, ay + ah - range, 1, range, 0, 0, 0, 0, cgs.media.whiteShader );
		}
	}

	trap_R_SetColor( NULL );

	if ( cg_nopredict.integer || cg_synchronousClients.integer ) {
		CG_DrawBigString( ax, ay, "snc", 1.0 );
	}

	CG_DrawDisconnect();
}



/*
===============================================================================

CENTER PRINTING

===============================================================================
*/


/*
==============
CG_CenterPrint

Called for important messages that should stay in the center of the screen
for a few moments
==============
*/
void CG_CenterPrint( const char *str, int y, int charWidth ) {
	char	*s;

	Q_strncpyz( cg.centerPrint, str, sizeof(cg.centerPrint) );

	cg.centerPrintTime = cg.time;
	cg.centerPrintY = y;
	cg.centerPrintCharWidth = charWidth;

	// count the number of lines for centering
	cg.centerPrintLines = 1;
	s = cg.centerPrint;
	while( *s ) {
		if (*s == '\n')
			cg.centerPrintLines++;
		s++;
	}
}


/*
===================
CG_DrawCenterString
===================
*/
static void CG_DrawCenterString( void ) {
	char	*start;
	int		l;
	int		x, y, w, h;
	float	*color;

	if ( !cg.centerPrintTime ) {
		return;
	}

	color = CG_FadeColor( cg.centerPrintTime, 1000 * cg_centertime.value );
	if ( !color ) {
		return;
	}

	trap_R_SetColor( color );

	start = cg.centerPrint;

	y = cg.centerPrintY - cg.centerPrintLines * BIGCHAR_HEIGHT / 2;

	while ( 1 ) {
		char linebuffer[1024];

		for ( l = 0; l < 50; l++ ) {
			if ( !start[l] || start[l] == '\n' ) {
				break;
			}
			linebuffer[l] = start[l];
		}
		linebuffer[l] = 0;

#ifdef MISSIONPACK
		w = CG_Text_Width(linebuffer, 0.5, 0);
		h = CG_Text_Height(linebuffer, 0.5, 0);
		x = (SCREEN_WIDTH - w) / 2;
		CG_Text_Paint(x, y + h, 0.5, color, linebuffer, 0, 0, ITEM_TEXTSTYLE_SHADOWEDMORE);
		y += h + 6;
#else
		w = cg.centerPrintCharWidth * CG_DrawStrlen( linebuffer );

		x = ( SCREEN_WIDTH - w ) / 2;

		CG_DrawStringExt( x, y, linebuffer, color, qfalse, qtrue,
			cg.centerPrintCharWidth, (int)(cg.centerPrintCharWidth * 1.5), 0 );

		y += cg.centerPrintCharWidth * 1.5;
#endif
		while ( *start && ( *start != '\n' ) ) {
			start++;
		}
		if ( !*start ) {
			break;
		}
		start++;
	}

	trap_R_SetColor( NULL );
}



/*
================================================================================

CROSSHAIR

================================================================================
*/


/*
=================
CG_DrawCrosshair
=================
*/
static void CG_DrawCrosshair(void) {
	float		w, h;
	qhandle_t	hShader;
	float		f;
	float		x, y;
	int			ca;
//EQ--
   int count, max, health;
   float healthcolour[4];
   float ex, ey, ew, eh;

   // calculate the total points of damage that can
   // be sustained at the current health / armor level
   health = cg.snap->ps.stats[STAT_HEALTH];
   count = cg.snap->ps.stats[STAT_ARMOR];
   max = health * ARMOR_PROTECTION / ( 1.0 - ARMOR_PROTECTION );
   if ( max < count ) count = max;
   health += count;
//--EQ

	if ( !cg_drawCrosshair.integer ) {
		return;
	}

	if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR) {
		return;
	}

       if ( cg.renderingThirdPerson && cg_3rdpersoncrosshair.integer == 0) {
              return;
       }
	// set color based on health
	if ( cg_crosshairHealth.integer ) {
		vec4_t		hcolor;

		CG_ColorForHealth( hcolor );
		trap_R_SetColor( hcolor );

//EQ--
	   // set the health string colour based on health
      healthcolour[3] = 1;
      healthcolour[2] = 0;
      if (health >= 100)
      {
         healthcolour[0] = 0;
         healthcolour[1] = 1;
      }
      else if (health > 50)
      {
         healthcolour[0] = (100-health)/50.0;
         healthcolour[1] = 1;
      }
      else
      {
         healthcolour[0] = 1;
         healthcolour[1] = health/50.0;
      }
//--EQ

	} else {
//EQ--
      healthcolour[0] = healthcolour[1] = healthcolour[2] = 1.0;
      healthcolour[3] = 0.9;
//--EQ

		trap_R_SetColor( NULL );
	}

	w = h = cg_crosshairSize.value;

	// pulse the size of the crosshair when picking up items
	f = cg.time - cg.itemPickupBlendTime;
	if ( f > 0 && f < ITEM_BLOB_TIME ) {
		f /= ITEM_BLOB_TIME;
		w *= ( 1 + f );
		h *= ( 1 + f );
	}

	x = cg_crosshairX.integer;
	y = cg_crosshairY.integer;
//EQ--
   ex=x; ey=y; ew=w; eh=h;
//--EQ
	CG_AdjustFrom640( &x, &y, &w, &h );

	ca = cg_drawCrosshair.integer;
	if (ca < 0) {
		ca = 0;
	}
	hShader = cgs.media.crosshairShader[ ca % NUM_CROSSHAIRS ];
	trap_R_DrawStretchPic( x + cg.refdef.x + 0.5 * (cg.refdef.width - w), 
		y + cg.refdef.y + 0.5 * (cg.refdef.height - h), 
		w, h, 0, 0, 1, 1, hShader );

      if (cg_centerhud.integer)
      {
      int   healthheight    = (int)(0.9 + cg.snap->ps.stats[STAT_HEALTH] / 5.0),
            armourheight    = (int)(0.9 + cg.snap->ps.stats[STAT_ARMOR] / 5.0),
            ammoheight      = (int)(0.9 + cg.snap->ps.ammo[cg_entities[cg.snap->ps.clientNum].currentState.weapon] / 5.0);
      float bgcolour[4]     = {0.7, 0.6, 0.1, 0.2},
            bgcolour2[4]    = {0.7, 0.6, 0.1, 0.15},
            armourcolour[4] = {1.0, 0.8, 0.2, 1.0},
            ammocolour[4]   = {1.0, 1.0, 1.0, 0.9};
      int   font_width      = SMALLCHAR_WIDTH,     //SMALLCHAR_WIDTH = TINYCHAR_WIDTH
            font_height,
            health_x        = ex - ew - font_width*3 +320 +4,
            health_y        = ey - 8 +240,
            ammo_x          = ex + ew  +320 -4,
            ammo_y          = health_y;

      char  healthstring[64], ammostring[64]="NO LIMIT";

      if (ammoheight > 40)
             ammoheight = 40;
      if (armourheight > 40)
             armourheight = 40;
      if (healthheight > 60)
             healthheight = 60;

      Com_sprintf(healthstring, sizeof(healthstring), "%-4d", health);
      if (cg.snap->ps.ammo[cg_entities[cg.snap->ps.clientNum].currentState.weapon] >= 0 && cg.snap->ps.ammo[cg_entities[cg.snap->ps.clientNum].currentState.weapon] < 1500)
             Com_sprintf(ammostring, sizeof(ammostring), "%d", cg.snap->ps.ammo[cg_entities[cg.snap->ps.clientNum].currentState.weapon]);

//      if (e_drawCrosshairStats.integer == 2)
         font_height = TINYCHAR_HEIGHT;
//      else
//         font_height = SMALLCHAR_HEIGHT;
       CG_DrawStringExt(health_x, health_y + 4*(1), healthstring, healthcolour,
                       qtrue, qfalse, font_width, font_height, 0);
       CG_DrawStringExt(ammo_x,   ammo_y + 4*(1),   ammostring,   ammocolour,
                         qtrue, qfalse, font_width, font_height, 0);
//       CG_DrawStringExt(health_x, health_y + 4*(e_drawCrosshairStats.integer-1), healthstring, healthcolour,
//                       qtrue, qfalse, font_width, font_height, 0);
//       CG_DrawStringExt(ammo_x,   ammo_y + 4*(e_drawCrosshairStats.integer-1),   ammostring,   ammocolour,
//                       qtrue, qfalse, font_width, font_height, 0);

      //Health...
      CG_FillRect(health_x -12,            health_y -22,                   2, 19,              bgcolour2);
      CG_FillRect(health_x -12,            health_y -2,                    2, 20-healthheight, bgcolour);
      CG_FillRect(health_x -12,            health_y -2 +(20-healthheight), 2, healthheight,    healthcolour);
      
      //Armour...
      CG_FillRect(health_x -8,             health_y -22,                   2, 19,              bgcolour2);
      CG_FillRect(health_x -8,             health_y -2,                    2, 20-armourheight, bgcolour);
      CG_FillRect(health_x -8,             health_y -2 +(20-armourheight), 2, armourheight,    armourcolour);

      //Ammo...
      if (cg.snap->ps.ammo[cg_entities[cg.snap->ps.clientNum].currentState.weapon] >= 0 && cg.snap->ps.ammo[cg_entities[cg.snap->ps.clientNum].currentState.weapon] < 1500)
      {
      CG_FillRect(ammo_x +font_width*3 +4, health_y -22,                   2, 19,            bgcolour2);
      CG_FillRect(ammo_x +font_width*3 +4, health_y -2,                    2, 20-ammoheight, bgcolour);
      CG_FillRect(ammo_x +font_width*3 +4, health_y -2 +(20-ammoheight),   2, ammoheight,    ammocolour);
      }
   }

}



/*
=================
CG_ScanForCrosshairEntity
=================
*/
static void CG_ScanForCrosshairEntity( void ) {
	trace_t		trace;
	vec3_t		start, end;
	int			content;

//       cg.targetter = 0;
	VectorCopy( cg.refdef.vieworg, start );
	VectorMA( start, 131072, cg.refdef.viewaxis[0], end );

	CG_Trace( &trace, start, vec3_origin, vec3_origin, end, 
		cg.snap->ps.clientNum, CONTENTS_SOLID|CONTENTS_BODY );
	if ( trace.entityNum >= MAX_CLIENTS ) {
		return;
	}

	// if the player is in fog, don't show it
	content = trap_CM_PointContents( trace.endpos, 0 );
	if ( content & CONTENTS_FOG ) {
		return;
	}

	// if the player is invisible, don't show it
	if ( cg_entities[ trace.entityNum ].currentState.powerups & ( 1 << PW_INVIS ) ) {
		return;
	}

	// update the fade timer
	cg.crosshairClientNum = trace.entityNum;
	cg.crosshairClientTime = cg.time;
//EQ--
//       cg.targetter = 1;
//--EQ
}


/*
=====================
CG_DrawCrosshairNames
=====================
*/
static void CG_DrawCrosshairNames( void ) {
	float		*color;
	char		*name;
	float		w;
       char        temp[128];
       float       h = 270;
       int         iconsize = ICON_SIZE / 3;
	char *teamstat;
       clientInfo_t target;
//EQ--
   char  ename[100];
   int   i;
	float ecolor[4];
//--EQ

	if ( !cg_drawCrosshair.integer ) {
		return;
	}
	if ( !cg_drawCrosshairNames.integer ) {
		return;
	}
//       if ( cg.renderingThirdPerson ) {
//              return;
//       }

	// scan the known entities to see if the crosshair is sighted on one
	CG_ScanForCrosshairEntity();

	// draw the name of the player being looked at
	color = CG_FadeColor( cg.crosshairClientTime, 1000 );
	if ( !color ) {
		trap_R_SetColor( NULL );
//EQ--
      cg.crosshairClientNum = 0;
//--EQ
		return;
	}
    target = cgs.clientinfo[ cg.crosshairClientNum ];
       if (cgs.clientinfo[ cg.crosshairClientNum ].powerups & ( 1 << PW_PROTECT ))
              name = va("%s - Spawn Protection",cgs.clientinfo[ cg.crosshairClientNum ].name);
       else
           name = va("%s",cgs.clientinfo[ cg.crosshairClientNum ].name);
#ifdef MISSIONPACK
	color[3] *= 0.5;
	w = CG_Text_Width(name, 0.3f, 0);
	CG_Text_Paint( 320 - w / 2, 190, 0.3f, color, name, 0, 0, ITEM_TEXTSTYLE_SHADOWED);
#else
	w = CG_DrawStrlen( name ) * BIGCHAR_WIDTH;
	CG_DrawBigString( 320 - w / 2, 170, name, color[3] * 0.5 );
#endif
    if ( cgs.gametype >= GT_TEAM && target.team == cgs.clientinfo[ cg.snap->ps.clientNum ].team && cg_drawTeamOverlay.integer)
    {
        h += BIGCHAR_HEIGHT;
        Com_sprintf(temp, sizeof(temp), "[  Health:%3i   Armor:%3i]", target.health, target.armor);
        w = 320 - (CG_DrawStrlen(temp) * SMALLCHAR_WIDTH) / 2;
        CG_DrawSmallString( w, h, temp, color[3] * 0.5 );
        w += SMALLCHAR_WIDTH;
        CG_DrawPic(w, h, iconsize, iconsize, target.modelIcon);
        w += (SMALLCHAR_WIDTH * 4.0) + iconsize;
        if (target.armor >= 100)
               CG_DrawPic(w+50, h, iconsize, iconsize, cgs.media.armorIconRed);
        else
               CG_DrawPic(w+50, h, iconsize, iconsize, cgs.media.armorIcon);
    }
//EQ--
   	for (i=0; i<cg.numScores; i++)
   		if (cg.scores[i].client == cg.crosshairClientNum)
            break;

      switch (i)
      {
         case 0:
            ecolor[0] = 0;
            ecolor[1] = 0;
            ecolor[2] = 1;
            Com_sprintf(ename, sizeof(ename), "[1st with %d frags]", cgs.clientinfo[cg.crosshairClientNum].score);
            break;
         case 1:
            ecolor[0] = 1;
            ecolor[1] = 0;
            ecolor[2] = 0;
            Com_sprintf(ename, sizeof(ename), "[2nd with %d frags]", cgs.clientinfo[cg.crosshairClientNum].score);
            break;
         case 2:
            ecolor[0] = 1;
            ecolor[1] = 1;
            ecolor[2] = 0;
            Com_sprintf(ename, sizeof(ename), "[3rd with %d frags]", cgs.clientinfo[cg.crosshairClientNum].score);
            break;
         default:
            ecolor[0] = 1;
            ecolor[1] = 1;
            ecolor[2] = 1;
            Com_sprintf(ename, sizeof(ename), "[%dth with %d frags]", i+1, cgs.clientinfo[cg.crosshairClientNum].score);
      }
//      ecolor[3] = color[3] * 0.5;
      ecolor[3] = 1;
      if (cg_draw3dIcons.integer)
      {
         w = CG_DrawStrlen(ename) * SMALLCHAR_WIDTH;
          CG_DrawSmallStringColor(320 - w / 2, h+BIGCHAR_HEIGHT+2, ename, ecolor);
      }
      else
      {
         w = CG_DrawStrlen(ename) * BIGCHAR_WIDTH;
          CG_DrawBigStringColor(320 - w / 2, h+BIGCHAR_HEIGHT+2, ename, ecolor);
      }
//--EQ
	trap_R_SetColor( NULL );
}


//==============================================================================

/*
=================
CG_DrawSpectator
=================
*/
static void CG_DrawSpectator(void) {
	CG_DrawBigString(320 - 9 * 8, 440, "SPECTATOR", 1.0F);
	if ( cgs.gametype == GT_TOURNAMENT ) {
		CG_DrawBigString(320 - 15 * 8, 460, "waiting to play", 1.0F);
	}
	else if ( cgs.gametype >= GT_TEAM ) {
		CG_DrawBigString(320 - 39 * 8, 460, "press ESC and use the JOIN menu to play", 1.0F);
	}
}

// the text's height
#define TEXT_HEIGHT 10
// the struct to hold our menu information
// hence the name of it, menuInfo :)
typedef struct{
int  number;
char *string;
vec4_t *color;
char *command;}
menuInfo;

// this array holds what    will be in the specMenu
    menuInfo specMenu[] = { 
    // Edit this to add new choices in your menu
    // here are what the fields do:
    // 1. This is the number the choice
    // is in the menu
    // 2. The text to appear for the choice
    // 3. What color it is
    // 4. What command to execute when the user
    // selects it.
    { 1, "Team Red", &colorWhite, "team r" },
    { 2, "Team Blue", &colorWhite, "team b" },
    { 3, "Team Free", &colorWhite, "team f" },
    // This lets the CG_DrawSpecMenu know
    // there is no more to draw
    {0, NULL}
    };
// Function we use to    make drawing any text for the menu easier
    void CG_DrawMenuText( int x, int y, const char *s, vec4_t color ) {
    CG_DrawStringExt( x, y, s, color, qfalse, qfalse, TEXT_HEIGHT, 12, 0 );
    }
/*
    =================
    CG_DrawSpecMenu
    Draws the spec menu
    =================
    */
    void CG_DrawSpecMenu( void ) {
    vec4_t color;
    char *menuString;
    int n;
    int w = 125;
    // Amount of items in the menu
    int count = 3;
    int h = 230 - (( count * TEXT_HEIGHT+1 )/2);
    int y = 240 - (( count * TEXT_HEIGHT+1 )/2);
 // lets CG_Weapon_f    know to
    // use the users keys to make a selection
    cg.specMenu = qtrue;
    
    // color of the menu's background
    color[0] = 0.00f;
    color[1] = 0.75f;
    color[2] = 0.25f;
    color[3] = 0.33f;
    
    // draw the rectangular background
    CG_FillRect( 0, h, w, 480-( h * 2 ), color );
 // goes through all    the choices in our menu
    // and draws them
    for(n = 0; specMenu[n].string; n++) {
    // tells it to stop when it reaches
    // the choice with the text of NULL
    if(specMenu[n].string == NULL) { 
    return;
    }
    // Combines the selection's # and
    // it't text string into a string to draw
    menuString = va("%i.%s", specMenu[n].number, specMenu[n].string );
    // Draws it with our above function
    CG_DrawMenuText(0, y, menuString, *specMenu[n].color );
    // add to the y number so next time it draws
    // a selection, it is below the first one
    y += TEXT_HEIGHT+1;
    }
    }
/*
    =================
    CG_SpecMenuCmd
    =================
    */
    void CG_SpecMenuCmd( int num ) {
    int n;
   // goes through all the choices in our menu	// and matches the incoming choice to	// the choice's command
    for(n = 0; specMenu[n].command; n++) {
    // tells it to stop when it reaches
    // the choice with the text of NULL
    if(specMenu[n].command == NULL) { 
    return;
    }
    // found the right choice
    if( num == specMenu[n].number ) {
    // Execute the command
    trap_SendClientCommand( specMenu[n].command );
    // Tells CG_Weapon_f to not monitor keys anymore
    cg.specMenu = qfalse;
    }
    }
    }

/*
=================
CG_DrawVote
=================
*/
static void CG_DrawVote(void) {
	char	*s;
	int		sec;

	if ( !cgs.voteTime ) {
		return;
	}

	// play a talk beep whenever it is modified
	if ( cgs.voteModified ) {
		cgs.voteModified = qfalse;
		trap_S_StartLocalSound( cgs.media.talkSound, CHAN_LOCAL_SOUND );
	}

	sec = ( VOTE_TIME - ( cg.time - cgs.voteTime ) ) / 1000;
	if ( sec < 0 ) {
		sec = 0;
	}
#ifdef MISSIONPACK
	s = va("VOTE(%i):%s yes:%i no:%i", sec, cgs.voteString, cgs.voteYes, cgs.voteNo);
	CG_DrawSmallString( 0, 58, s, 1.0F );
	s = "or press ESC then click Vote";
	CG_DrawSmallString( 0, 58 + SMALLCHAR_HEIGHT + 2, s, 1.0F );
#else
	s = va("VOTE(%i):%s yes:%i no:%i", sec, cgs.voteString, cgs.voteYes, cgs.voteNo );
	CG_DrawSmallString( 0, 58, s, 1.0F );
#endif
}

/*
=================
CG_DrawTeamVote
=================
*/
static void CG_DrawTeamVote(void) {
	char	*s;
	int		sec, cs_offset;

	if ( cgs.clientinfo->team == TEAM_RED )
		cs_offset = 0;
	else if ( cgs.clientinfo->team == TEAM_BLUE )
		cs_offset = 1;
	else
		return;

	if ( !cgs.teamVoteTime[cs_offset] ) {
		return;
	}

	// play a talk beep whenever it is modified
	if ( cgs.teamVoteModified[cs_offset] ) {
		cgs.teamVoteModified[cs_offset] = qfalse;
		trap_S_StartLocalSound( cgs.media.talkSound, CHAN_LOCAL_SOUND );
	}

	sec = ( VOTE_TIME - ( cg.time - cgs.teamVoteTime[cs_offset] ) ) / 1000;
	if ( sec < 0 ) {
		sec = 0;
	}
	s = va("TEAMVOTE(%i):%s yes:%i no:%i", sec, cgs.teamVoteString[cs_offset],
							cgs.teamVoteYes[cs_offset], cgs.teamVoteNo[cs_offset] );
	CG_DrawSmallString( 0, 90, s, 1.0F );
}


static qboolean CG_DrawScoreboard() {
#ifdef MISSIONPACK
	static qboolean firstTime = qtrue;
	float fade, *fadeColor;

	if (menuScoreboard) {
		menuScoreboard->window.flags &= ~WINDOW_FORCED;
	}
	if (cg_paused.integer) {
		cg.deferredPlayerLoading = 0;
		firstTime = qtrue;
		return qfalse;
	}

	// should never happen in Team Arena
	if (cgs.gametype == GT_SINGLE_PLAYER && cg.predictedPlayerState.pm_type == PM_INTERMISSION ) {
		cg.deferredPlayerLoading = 0;
		firstTime = qtrue;
		return qfalse;
	}

	// don't draw scoreboard during death while warmup up
	if ( cg.warmup && !cg.showScores ) {
		return qfalse;
	}

	if ( cg.showScores || cg.predictedPlayerState.pm_type == PM_DEAD || cg.predictedPlayerState.pm_type == PM_INTERMISSION ) {
		fade = 1.0;
		fadeColor = colorWhite;
	} else {
		fadeColor = CG_FadeColor( cg.scoreFadeTime, FADE_TIME );
		if ( !fadeColor ) {
			// next time scoreboard comes up, don't print killer
			cg.deferredPlayerLoading = 0;
			cg.killerName[0] = 0;
			firstTime = qtrue;
			return qfalse;
		}
		fade = *fadeColor;
	}																					  


	if (menuScoreboard == NULL) {
		if ( cgs.gametype >= GT_TEAM ) {
			menuScoreboard = Menus_FindByName("teamscore_menu");
		} else {
			menuScoreboard = Menus_FindByName("score_menu");
		}
	}

	if (menuScoreboard) {
		if (firstTime) {
			CG_SetScoreSelection(menuScoreboard);
			firstTime = qfalse;
		}
		Menu_Paint(menuScoreboard, qtrue);
	}

	// load any models that have been deferred
	if ( ++cg.deferredPlayerLoading > 10 ) {
		CG_LoadDeferredPlayers();
	}

	return qtrue;
#else
	return CG_DrawOldScoreboard();
#endif
}

/*
=================
CG_DrawIntermission
=================
*/
static void CG_DrawIntermission( void ) {
//	int key;
#ifdef MISSIONPACK
	//if (cg_singlePlayer.integer) {
	//	CG_DrawCenterString();
	//	return;
	//}
#else
	if ( cgs.gametype == GT_SINGLE_PLAYER ) {
		CG_DrawCenterString();
		return;
	}
#endif
	cg.scoreFadeTime = cg.time;
	cg.scoreBoardShowing = CG_DrawScoreboard();
}

/*
=================
CG_DrawFollow
=================
*/
static qboolean CG_DrawFollow( void ) {
	float		x;
	vec4_t		color;
	const char	*name;

	if ( !(cg.snap->ps.pm_flags & PMF_FOLLOW) ) {
		return qfalse;
	}
	color[0] = 1;
	color[1] = 1;
	color[2] = 1;
	color[3] = 1;


	CG_DrawBigString( 320 - 9 * 8, 24, "following", 1.0F );

	name = cgs.clientinfo[ cg.snap->ps.clientNum ].name;

       x = 0.5 * ( 640 - (GIANT_WIDTH/2) * CG_DrawStrlen( name ) );

       CG_DrawStringExt( x, 40, name, color, qtrue, qtrue, GIANT_WIDTH/2, GIANT_HEIGHT/2, 0 );

	return qtrue;
}



/*
=================
CG_DrawAmmoWarning
=================
*/
static void CG_DrawAmmoWarning( void ) {
	const char	*s;
	int			w;

	if ( cg_drawAmmoWarning.integer == 0 ) {
		return;
	}

	if ( !cg.lowAmmoWarning ) {
		return;
	}

	if ( cg.lowAmmoWarning == 2 ) {
		s = "OUT OF AMMO";
	} else {
		s = "LOW AMMO WARNING";
	}
	w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH;
	CG_DrawBigString(320 - w / 2, 64, s, 1.0F);
}


#ifdef MISSIONPACK
/*
=================
CG_DrawProxWarning
=================
*/
static void CG_DrawProxWarning( void ) {
	char s [32];
	int			w;
  static int proxTime;
  static int proxCounter;
  static int proxTick;

	if( !(cg.snap->ps.eFlags & EF_TICKING ) ) {
    proxTime = 0;
		return;
	}

  if (proxTime == 0) {
    proxTime = cg.time + 5000;
    proxCounter = 5;
    proxTick = 0;
  }

  if (cg.time > proxTime) {
    proxTick = proxCounter--;
    proxTime = cg.time + 1000;
  }

  if (proxTick != 0) {
    Com_sprintf(s, sizeof(s), "INTERNAL COMBUSTION IN: %i", proxTick);
  } else {
    Com_sprintf(s, sizeof(s), "YOU HAVE BEEN MINED");
  }

	w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH;
	CG_DrawBigStringColor( 320 - w / 2, 64 + BIGCHAR_HEIGHT, s, g_color_table[ColorIndex(COLOR_RED)] );
}
#endif


/*
=================
CG_DrawWarmup
=================
*/
static void CG_DrawWarmup( void ) {
	int			w;
	int			sec;
	int			i;
	float scale;
	clientInfo_t	*ci1, *ci2;
	int			cw;
	const char	*s;

	sec = cg.warmup;
	if ( !sec ) {
		return;
	}

	if ( sec < 0 ) {
		s = "Waiting for players";		
		w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH;
		CG_DrawBigString(320 - w / 2, 24, s, 1.0F);
		cg.warmupCount = 0;
		return;
	}

	if (cgs.gametype == GT_TOURNAMENT) {
		// find the two active players
		ci1 = NULL;
		ci2 = NULL;
		for ( i = 0 ; i < cgs.maxclients ; i++ ) {
			if ( cgs.clientinfo[i].infoValid && cgs.clientinfo[i].team == TEAM_FREE ) {
				if ( !ci1 ) {
					ci1 = &cgs.clientinfo[i];
				} else {
					ci2 = &cgs.clientinfo[i];
				}
			}
		}

		if ( ci1 && ci2 ) {
			s = va( "%s vs %s", ci1->name, ci2->name );
#ifdef MISSIONPACK
			w = CG_Text_Width(s, 0.6f, 0);
			CG_Text_Paint(320 - w / 2, 60, 0.6f, colorWhite, s, 0, 0, ITEM_TEXTSTYLE_SHADOWEDMORE);
#else
			w = CG_DrawStrlen( s );
			if ( w > 640 / GIANT_WIDTH ) {
				cw = 640 / w;
			} else {
				cw = GIANT_WIDTH;
			}
			CG_DrawStringExt( 320 - w * cw/2, 20,s, colorWhite, 
					qfalse, qtrue, cw, (int)(cw * 1.5f), 0 );
#endif
		}
	} else {
		if ( cgs.gametype == GT_FFA ) {
			s = "Free For All";
		} else if ( cgs.gametype == GT_TEAM ) {
			s = "Team Deathmatch";
		} else if ( cgs.gametype == GT_CTF ) {
			s = "Capture the Flag";
#ifdef MISSIONPACK
		} else if ( cgs.gametype == GT_1FCTF ) {
			s = "One Flag CTF";
		} else if ( cgs.gametype == GT_OBELISK ) {
			s = "Overload";
		} else if ( cgs.gametype == GT_HARVESTER ) {
			s = "Harvester";
#endif
		} else {
			s = "";
		}
#ifdef MISSIONPACK
		w = CG_Text_Width(s, 0.6f, 0);
		CG_Text_Paint(320 - w / 2, 90, 0.6f, colorWhite, s, 0, 0, ITEM_TEXTSTYLE_SHADOWEDMORE);
#else
		w = CG_DrawStrlen( s );
		if ( w > 640 / GIANT_WIDTH ) {
			cw = 640 / w;
		} else {
			cw = GIANT_WIDTH;
		}
		CG_DrawStringExt( 320 - w * cw/2, 25,s, colorWhite, 
				qfalse, qtrue, cw, (int)(cw * 1.1), 0 );
#endif
	}

	sec = ( sec - cg.time ) / 1000;
	if ( sec < 0 ) {
		cg.warmup = 0;
		sec = 0;
	}
	s = va( "Starts in: %i", sec + 1 );
	if ( sec != cg.warmupCount ) {
		cg.warmupCount = sec;
		switch ( sec ) {
		case 0:
			trap_S_StartLocalSound( cgs.media.count1Sound, CHAN_ANNOUNCER );
			break;
		case 1:
			trap_S_StartLocalSound( cgs.media.count2Sound, CHAN_ANNOUNCER );
			break;
		case 2:
			trap_S_StartLocalSound( cgs.media.count3Sound, CHAN_ANNOUNCER );
			break;
		default:
			break;
		}
	}
	scale = 0.45f;
	switch ( cg.warmupCount ) {
	case 0:
		cw = 28;
		scale = 0.54f;
		break;
	case 1:
		cw = 24;
		scale = 0.51f;
		break;
	case 2:
		cw = 20;
		scale = 0.48f;
		break;
	default:
		cw = 16;
		scale = 0.45f;
		break;
	}

#ifdef MISSIONPACK
		w = CG_Text_Width(s, scale, 0);
		CG_Text_Paint(320 - w / 2, 125, scale, colorWhite, s, 0, 0, ITEM_TEXTSTYLE_SHADOWEDMORE);
#else
	w = CG_DrawStrlen( s );
	CG_DrawStringExt( 320 - w * cw/2, 70, s, colorWhite, 
			qfalse, qtrue, cw, (int)(cw * 1.5), 0 );
#endif
}

static void CG_DrawCampWarning( void ) {
	int			w;
       int sec;
       static               int warning = 0;
       static               int counter = 0;
	int			i;
       char m [32];
       static int                  timer = 0;
	const char	*s;
       const char    *s2;
       char tempmsg[128];

       if (cg.predictedPlayerState.powerups[PW_CAMP])
       {
              s = va ("No Camping Allowed");
              s2 = va( "Move In 5 Seconds Or Be Killed!");
              w = CG_DrawStrlen( s );
              CG_DrawBigString (180, 70, s, 1.0F);
              CG_DrawBigString (60, 90, s2, 1.0F);
       }
}

static void CG_DrawWeaponWarning( void ) {
	int			w;
       int sec;
       static               int warning = 0;
       static               int counter = 0;
	int			i;
       static int                  timer = 0;
	const char	*s;

       if (cg.time - cgs.levelStartTime <= 2000 && timer != 0 )
       {
              timer = 0;
              counter = 0;
       }
       sec = counter+(cg_weaponchangetime.integer*1000);
       sec = (sec - (cg.time-cgs.levelStartTime)) / 1000;
       if (cg.time - cgs.levelStartTime >= (cg_weaponchangetime.integer*1000)-(5000)+timer && warning != 1)
       {
       s = va( "Weapon Change In %i Seconds", sec + 1);
       if (sec + 1 == 1)
              s = va( "Weapon Change In %i Second", sec + 1);
       if ( cg.time - cgs.levelStartTime >= (cg_weaponchangetime.integer*1000)+timer )
              warning = 1;
       w = CG_DrawStrlen( s );
       CG_DrawBigString (110, 70, s, 1.0F);
       }
       if ( cg.time - cgs.levelStartTime >= (cg_weaponchangetime.integer*1000)+timer )
       {
              counter += cg_weaponchangetime.integer*1000;
              timer += cg_weaponchangetime.integer*1000;
              warning = 0;
       }
}

static void CG_DrawPowerupWarning( void ) {
	int			w;
       int sec2;
       static               int warning2 = 0;
       static               int counter2 = 0;
	int			i;
       static int                  timer2 = 0;
	const char	*s;

       if (cg.time - cgs.levelStartTime <= 2000 && timer2 != 0 )
       {
              timer2 = 0;
              counter2 = 0;
       }
       sec2 = counter2+(cg_powerupchangetime.integer*1000);
       sec2 = (sec2 - (cg.time-cgs.levelStartTime)) / 1000;
       if (cg.time - cgs.levelStartTime >= (cg_powerupchangetime.integer*1000)-(5000)+timer2 && warning2 != 1)
       {
       s = va( "Powerup Change In %i Seconds", sec2 + 1);
       if (sec2 + 1 == 1)
              s = va( "Powerup Change In %i Second", sec2 + 1);
       if (cg.time - cgs.levelStartTime >= (cg_powerupchangetime.integer*1000)+timer2)
              warning2 = 1;
       w = CG_DrawStrlen( s );
       CG_DrawBigString (100, 86, s, 1.0F);
       }
       if ( cg.time - cgs.levelStartTime >= (cg_powerupchangetime.integer*1000)+timer2 )
       {
              counter2 += cg_powerupchangetime.integer*1000;
              timer2 += cg_powerupchangetime.integer*1000;
              warning2 = 0;
       }
}

//==================================================================================
#ifdef MISSIONPACK
/* 
=================
CG_DrawTimedMenus
=================
*/
void CG_DrawTimedMenus() {
	if (cg.voiceTime) {
		int t = cg.time - cg.voiceTime;
		if ( t > 2500 ) {
			Menus_CloseByName("voiceMenu");
			trap_Cvar_Set("cl_conXOffset", "0");
			cg.voiceTime = 0;
		}
	}
}
#endif

/*
=================
CG_DrawNHLine
=================
*/
static float CG_DrawNHLine( float y ) {
    int x=0;
    if (cg_hud.integer == 5)
       x = 564;
    CG_DrawPic(x, y, TEXT_ICON_SPACE + ICON_SIZE / 2 + BIGCHAR_WIDTH * 3, 1, cgs.media.whiteShader );
    y += TEXT_ICON_SPACE;
    return y;
}

// KHAILE
// Used for the armor/health font
#define VALUECHAR_HEIGHT BIGCHAR_HEIGHT
#define VALUECHAR_WIDTH BIGCHAR_WIDTH

/*
=================
CG_DrawNHBars
=================
*/
static void CG_DrawNHBars( int value, int maxvalue, qhandle_t icon, int top, int barheight, qboolean isleft, int modeltype ) {
    int     left, savetop;
    vec4_t  hcolor;
    char    temp[10];
    float   width;
    float   divisor;
    vec3_t        angles;
    vec3_t        origin;
    playerState_t *ps;

    hcolor[0] = hcolor[1] = hcolor[2] = hcolor[3] = 1.0;
    left = 260; //640 / 2;
	savetop = top;
    if ( isleft ) {
        left -= 2 + 200;
    } else {
        left += 122;
    }
    CG_FillRect( left, top + BIGCHAR_HEIGHT-1, 200, 1, hcolor );
    CG_FillRect( 320, top, 1, 33, hcolor );
    /*
    hcolor[0] = 0;
    hcolor[1] = 0;
    hcolor[2] = 1;
    hcolor[3] = 0.33;
//    CG_FillRect( 58, top, 506,(BIGCHAR_HEIGHT*2)+5, hcolor );
    trap_R_SetColor( hcolor );
    CG_DrawPic( 58, top, 526, 34, cgs.media.colorStatusBar );
    hcolor[0] = hcolor[1] = hcolor[2] = hcolor[3] = 1.0;
    CG_FillRect( 58, top, 526, 1, hcolor );
    CG_FillRect( 58, top, 1, 34, hcolor );
    CG_FillRect( 58, top+34, 526, 1, hcolor );
    CG_FillRect( 526+57, top, 1, 34, hcolor );
    */
    // color goes from red to white for 0-100
    // and from white to icy cool blue for 101-200
	// KHAILE (changed some settings in this function)
    //hcolor[3] = 1.0; //value < 30 ? 1.0 : 0.5;
    hcolor[3] = 0.5;

    if ( value > maxvalue/2 ) {
        // interpolate red to 0
        hcolor[0] = (maxvalue - value) / (maxvalue/2.0);
    } else {
        hcolor[0] = 1.0;
    }

    if ( value >= maxvalue/2 ) {
		hcolor[2] = 1.0;
       } else if ( value < 66 ) {
		hcolor[2] = 0;
	} else {
              hcolor[2] = ( value - 66) / 33.0;
	}

       if ( value > maxvalue/2 ) {
        // interpolate green to 0.5
              hcolor[1] = ((maxvalue - value) / maxvalue) + 0.5;
    } else if ( value > 60) {
		hcolor[1] = 1.0;
       } else if ( value < 30 ) {
		hcolor[1] = 0;
	} else {
              hcolor[1] = ( value - 30 ) / 30.0;
	}

    left = 260; //640 / 2;

       hcolor[3] = value < 30 ? 1.0 : 0.5;
    if ( value ) {
        // if value > 200
        //    value = 200;
        if (cg.snap->ps.stats[ STAT_MAX_HEALTH] >= 200)
        {
               divisor = maxvalue/200.0;
               width = ((value > maxvalue ? maxvalue : value)/divisor);
               width = ceil(width);
        }
        else
        {
               divisor = 200.0/cg.snap->ps.stats[ STAT_MAX_HEALTH];
               width = ((value > cg.snap->ps.stats[ STAT_MAX_HEALTH] ? cg.snap->ps.stats[ STAT_MAX_HEALTH] : value)*divisor);
               width = ceil(width);
        }
        left = 260;
        if ( isleft ) {
            left -= 2 + width;
        } else {
            left += 122;
        }
        CG_FillRect( left, top, width, barheight, hcolor );
        if (value > cg.snap->ps.stats[ STAT_MAX_HEALTH])
        {
            if (cg.snap->ps.stats[ STAT_MAX_HEALTH] >= 200)
            {
                   divisor = maxvalue/200.0;
                   width = (value - maxvalue)/divisor;
                   width = ceil(width);
            }
            else
            {
                   divisor = 200.0/cg.snap->ps.stats[ STAT_MAX_HEALTH];
                   width = (value - cg.snap->ps.stats[ STAT_MAX_HEALTH])*divisor;
                   width = ceil(width);
            }
            if (width >= 200)
                  width = 200;
            left = 260;
            if ( isleft ) {
                left -= 2 + width;
            } else {
                left += 122;
            }
            top += barheight + (TEXT_ICON_SPACE-4);
            CG_FillRect( left, top, width, barheight, hcolor );
        }
        if ( isleft ) {
            left -= TEXT_ICON_SPACE + barheight;
        } else {
            left += TEXT_ICON_SPACE + width;
        }
        if ( icon > 0 ) {
                if (modeltype == 1)
                       CG_DrawStatusBarHead2(left,top,barheight,barheight);
                else if (modeltype == 2)
                {
                   ps = &cg.snap->ps;
                   VectorClear( angles );
                   if ( ps->stats[ STAT_ARMOR ] )
                   {
                            if (cg_draw3dIcons.integer)
                            {
                                   origin[0] = 90;
                                   origin[1] = 0;
                                   origin[2] = -10;
                                   angles[YAW] = ( cg.time & 2047 ) * 360 / 2048.0;
                                   if (ps->stats[STAT_ARMOR] >= 100)
                                          CG_Draw3DModel( left, top, barheight, barheight,
                                                                  cgs.media.armorModelRed, 0, origin, angles );
                                   else
                                          CG_Draw3DModel( left, top, barheight, barheight,
                                                                  cgs.media.armorModel, 0, origin, angles );
                            }
                            else
                                   CG_DrawPic( left, top, barheight, barheight, icon );
                   }
                }

        }
    } else {
        if ( icon > 0 ) {
            left = 260;
            if ( isleft ) {
                left -= 2 + TEXT_ICON_SPACE;
            } else {
                left += 122 + TEXT_ICON_SPACE;
            }
            if (modeltype == 1)
                     CG_DrawPic( left, top, barheight, barheight, icon );
            else if (modeltype == 2)
            {
                   ps = &cg.snap->ps;
                   VectorClear( angles );
                   if ( ps->stats[ STAT_ARMOR ] )
                   {
                            if (cg_draw3dIcons.integer)
                            {
                                   origin[0] = 90;
                                   origin[1] = 0;
                                   origin[2] = -10;
                                   angles[YAW] = ( cg.time & 2047 ) * 360 / 2048.0;
                                   if (ps->stats[STAT_ARMOR] >= 100)
                                          CG_Draw3DModel( left, top, barheight, barheight,
                                                                  cgs.media.armorModelRed, 0, origin, angles );
                                   else
                                          CG_Draw3DModel( left, top, barheight, barheight,
                                                                  cgs.media.armorModel, 0, origin, angles );
                            }
                            else
                                   CG_DrawPic( left, top, barheight, barheight, icon );
                   }
            }
        }
    }

    hcolor[3] = 1.0; //value < 30 ? 1.0 : 0.5;
    Com_sprintf(temp, sizeof(temp), "%i", value);
    if ( isleft ) {
//        left = 128; // 320 - 192
        left = 260;
    } else {
//        left = 512 - (CG_DrawStrlen(temp) * VALUECHAR_WIDTH);
        left = 380 - (CG_DrawStrlen(temp) * BIGCHAR_WIDTH);
    }
//    CG_DrawStringExt(left, savetop - VALUECHAR_HEIGHT + 5, temp, hcolor, qfalse, qtrue, VALUECHAR_WIDTH, VALUECHAR_HEIGHT, 3);
    CG_DrawStringExt(left, savetop - BIGCHAR_HEIGHT + 23, temp, hcolor, qfalse, qtrue, BIGCHAR_WIDTH, BIGCHAR_HEIGHT, 3);
}


/*
=================
CG_DrawNHHealth
=================
*/
static void CG_DrawNHHealth( void ) {
    qhandle_t icon;
    int maxvalue;
    int value;

    value = cg.snap->ps.stats[ STAT_HEALTH ];
    maxvalue = cg.snap->ps.stats[ STAT_MAX_HEALTH];
    if (maxvalue < 200)
       maxvalue = 200;
    if ( value > 30 || (cg.time % 1000) < 500 ) {
        icon = cgs.clientinfo[ cg.snap->ps.clientNum ].modelIcon;
    } else {
        icon = -1;
    }
    CG_DrawNHBars( value, maxvalue, icon, 425, 16, qtrue, 1 );

    value = cg.snap->ps.stats[ STAT_ARMOR ];
    if (value >= 100)
           icon = cgs.media.armorIconRed;
    else
           icon = cgs.media.armorIcon;
    CG_DrawNHBars( value, 200, icon, 425, 16, qfalse, 2 );
}

/*
=================
CG_DrawNHAmmo
=================
*/
static float CG_DrawNHAmmo( float y ) {
    int         weap,t,count,i;
    vec4_t      hcolor;
//    float       *color;
    int         value;
    qhandle_t   icon;
    int         iconsize;
    float       charwidth, charheight;
    char        temp[10];
    int bits;
    centity_t     *cent;
    playerState_t *ps;

	cent = &cg_entities[cg.snap->ps.clientNum];
      bits = cg.snap->ps.stats[ STAT_WEAPONS ];
	ps = &cg.snap->ps;

      if ( cg.predictedPlayerState.stats[STAT_HEALTH] <= 0 ) { 
        return; 
      }

      for ( i = 1 ; i < NUM_WEAPS ; i++ ) { //WarZone 
        if ( bits & ( 1 << i ) ) { 
          count++; 
        } 
      } 

    weap = weaponRawOrder[NUM_WEAPS - 1]; //WarZone -- select last weapon 
    for ( t = 1; t < NUM_WEAPS+1; t++ ) {
        weap = NextWeapon( weap );

        if ( !( bits & ( 1 << weap ) ) ) {
          continue; 
        } 

        if ( cg.predictedPlayerState.stats[ STAT_WEAPONS ] & ( 1 << weap ) ) {
            value = cg.predictedPlayerState.ammo[weap];
            hcolor[0] = 1.0;
            hcolor[1] = hcolor[2] = value < 10 ? 0.3 : 1.0;
            hcolor[3] = 0.7;
            icon = cg_weapons[weap].ammoIcon;

            if (cent->currentState.weapon == weap) {
                hcolor[0] = hcolor[1] = hcolor[2] = hcolor[3] = 1.0;
                hcolor[0] = 0;
                hcolor[1] = 0;
                hcolor[2] = 1;
                hcolor[3] = 0.5;
                iconsize = ICON_SIZE / 2; 
                charwidth = BIGCHAR_WIDTH;
                charheight = BIGCHAR_HEIGHT;
                CG_FillRect( TEXT_ICON_SPACE-4, y, charwidth*5, charheight*1.5, hcolor );
                hcolor[0] = 1.0;
                hcolor[1] = hcolor[2] = value < 10 ? 0.3 : 1.0;
                hcolor[3] = 0.7;
            } else {
                iconsize = ICON_SIZE / 3; 
                charwidth = TINYCHAR_WIDTH;
                charheight = TINYCHAR_HEIGHT;
            }

            CG_DrawPic( TEXT_ICON_SPACE, y, iconsize, iconsize, icon );
            if ( !value ) {
                CG_DrawPic( TEXT_ICON_SPACE, y, iconsize, iconsize, cgs.media.noammoShader );
            }

            if (value > 0 && value < 1500) { // Don't draw negative ammo
                Com_sprintf (temp, sizeof(temp), "%i", value);
                CG_DrawStringExt( TEXT_ICON_SPACE * 2 + iconsize, y + ((iconsize / 2) - (charheight / 2)), temp, 
                                  hcolor, qfalse, qtrue, charwidth,charheight, 3 );
            }
            y += iconsize + TEXT_ICON_SPACE;

           }
       }
	value = cg.snap->ps.stats[STAT_HOLDABLE_ITEM];
	if ( value ) {
		CG_RegisterItemVisuals( value );
        iconsize = ICON_SIZE / 3;
        CG_DrawPic( TEXT_ICON_SPACE, y, iconsize, iconsize, cg_items[ value ].icon );
        y += iconsize + TEXT_ICON_SPACE;
	}
    y = CG_DrawNHLine(y);
    return y;
}

/*
=================
CG_DrawNHAmmo2
=================
*/
void CG_DrawNHAmmo2() {
    int         t;
    vec4_t      hcolor;
//    float       *color;
    int         value;
    int count;
    int i;
    qhandle_t   icon;
    int a,b;
    float       y,x;
    int         iconsize;
    float       charwidth, charheight;
    char        temp[10];
    centity_t     *cent;
    int size;
   int  bits; 
    int weap;
      float *color;
      char *name;

    playerState_t *ps;
    y = 410;
    if (cg_hud.integer == 1)
    y = 385;
	cent = &cg_entities[cg.snap->ps.clientNum];
	ps = &cg.snap->ps;
      bits = cg.snap->ps.stats[ STAT_WEAPONS ];

      if ( cg.predictedPlayerState.stats[STAT_HEALTH] <= 0 ) { 
        return; 
      }
//      if (!cg_showAllWeapons.integer)
//      {
             color = CG_FadeColor( cg.weaponSelectTime, WEAPON_SELECT_TIME ); 
             if ( !color ) { 
               return; 
             } 
             trap_R_SetColor( color ); 
             // showing weapon select clears pickup item display, but not the blend blob 
             cg.itemPickupTime = 0; 
//      }

      count = 0; 
      for ( i = 1 ; i < NUM_WEAPS ; i++ ) { //WarZone 
        if ( bits & ( 1 << i ) ) { 
          count++; 
        } 
      } 
    x = 320 - count * 20;
    weap = weaponRawOrder[NUM_WEAPS - 1]; //WarZone -- select last weapon 
    for ( t = 1; t < NUM_WEAPS+1; t++ ) {
        weap = NextWeapon( weap );

        if ( !( bits & ( 1 << weap ) ) ) {
          continue; 
        } 

        CG_RegisterWeapon( weap ); 
        if ( cg.predictedPlayerState.stats[ STAT_WEAPONS ] & ( 1 << weap ) ) {
            value = cg.predictedPlayerState.ammo[weap];
            hcolor[0] = 1.0;
            hcolor[1] = hcolor[2] = value < 10 ? 0.3 : 1.0;
            hcolor[3] = 0.7;
            icon = cg_weapons[weap].ammoIcon;

            if (cent->currentState.weapon == weap) {
                hcolor[0] = hcolor[1] = hcolor[2] = hcolor[3] = 1.0;
                hcolor[0] = 0;
                hcolor[1] = 0;
                hcolor[2] = 1;
                hcolor[3] = 0.5;
                iconsize = ICON_SIZE / 2; 
                charwidth = BIGCHAR_WIDTH;
                charheight = BIGCHAR_HEIGHT;
                if (cgs.gametype < GT_TEAM) //  && cg_showAllWeapons.integer == 0
                {
                       CG_DrawPic( x-2, y-2, iconsize+4, iconsize+4, cgs.media.selectShader );
                }
                CG_FillRect( x, y, iconsize, charheight*1.5, hcolor );
                hcolor[0] = 1.0;
                hcolor[1] = hcolor[2] = value < 10 ? 0.3 : 1.0;
                hcolor[3] = 0.7;
                charwidth/=1.2;
                charheight/=1.2;
            } else {
                iconsize = ICON_SIZE / 3; 
                charwidth = TINYCHAR_WIDTH;
                charheight = TINYCHAR_HEIGHT;
            }

            CG_DrawPic( x, y, iconsize, iconsize, icon );
            if ( !value ) {
                CG_DrawPic( x, y, iconsize, iconsize, cgs.media.noammoShader );
            }
            if (iconsize == 24)
            {
              if (value > 99)
                  size = 11;
              else if (value > 9)
                  size = 5;
              else
                  size = - 2;
                hcolor[0] = 1.0;
                hcolor[1] = hcolor[2] = value < 10 ? 0.3 : 1.0;
                hcolor[3] = 0.7;
            }
            else // 16
            {
              if (value > 99)
                  size = 11/2;
              else if (value > 9)
                  size = 1.5;
              else
                  size = - 4;
//                hcolor[0] = cg_test.value;
//                hcolor[1] = cg_test2.value;
//                hcolor[2] = cg_test3.value;
//                hcolor[3] = cg_test4.value;
                hcolor[0] = 1;
                hcolor[1] = 0.3;
                hcolor[2] = 0.5;
                hcolor[3] = 1;
            }
            if (value > 0 && value < 1500) { // Don't draw negative ammo
                Com_sprintf (temp, sizeof(temp), "%i", value);
                CG_DrawStringExt( x-size+2, y+(iconsize)+2, temp, 
                                  hcolor, qfalse, qtrue, charwidth,charheight, 3 );
            }
            x += (iconsize*2) + TEXT_ICON_SPACE;

           }
       }
      if ( cg_weapons[ cg.weaponSelect ].item ) { 
        name = cg_weapons[ cg.weaponSelect ].item->pickup_name; 
        if ( name ) { 
          a = CG_DrawStrlen( name ) * BIGCHAR_WIDTH; 
          b = ( SCREEN_WIDTH - a ) / 2; 
          CG_DrawBigStringColor(b, y - 22, name, color);
        } 
      } 

      trap_R_SetColor( NULL ); 
}

/*
=================
CG_DrawNHHoldable
=================
*/
static float CG_DrawNHHoldable(float y) {
    int         t;
    vec4_t      hcolor;
//    float       *color;
    int         value;
    int count;
    int i;
    qhandle_t   icon;
    int         iconsize;
    float       charwidth, charheight;
    char        temp[10];
    centity_t     *cent;
    int size;
   int  bits; 
    int weap;
    playerState_t *ps;

	value = cg.snap->ps.stats[STAT_HOLDABLE_ITEM];
	if ( value ) {
        y = CG_DrawNHLine(y);
		CG_RegisterItemVisuals( value );
        iconsize = ICON_SIZE / 3;
        CG_DrawPic( TEXT_ICON_SPACE, y, iconsize, iconsize, cg_items[ value ].icon );
        y += iconsize + TEXT_ICON_SPACE;
        y = CG_DrawNHLine(y);
	}

    return y;
}

/*
=================
CG_DrawNHHealth2
=================
*/
static float CG_DrawNHHealth2( float y ) {
    int         t;
    vec4_t      hcolor;
    //float       *color;
    int         value;
    qhandle_t   icon;
    int         iconsize;
    float       charwidth, charheight;
    char        temp[10];
    vec3_t        angles;
    vec3_t        origin;
    playerState_t *ps;

       ps = &cg.snap->ps;
       iconsize = ICON_SIZE / 3; 
       charwidth = BIGCHAR_WIDTH;
       charheight = BIGCHAR_HEIGHT;

       hcolor[0] = hcolor[1] = hcolor[2] = hcolor[3] = 1.0;
       Com_sprintf(temp, sizeof(temp), "%i", cg.snap->ps.stats[ STAT_HEALTH ]);
       CG_DrawStringExt( TEXT_ICON_SPACE * 2 + iconsize, y + ((iconsize / 2) - (BIGCHAR_HEIGHT / 2)), temp, 
                                  hcolor, qfalse, qtrue, charwidth,charheight, 3 );
       CG_DrawStatusBarHead2(TEXT_ICON_SPACE,y,ICON_SIZE / 3,ICON_SIZE / 3);
       y += iconsize + TEXT_ICON_SPACE;

       VectorClear( angles );
       if ( ps->stats[ STAT_ARMOR ] ) {
              if (cg_draw3dIcons.integer)
              {
                     iconsize = ICON_SIZE / 3;
                     origin[0] = 90;
                     origin[1] = 0;
                     origin[2] = -10;
                     angles[YAW] = ( cg.time & 2047 ) * 360 / 2048.0;
                     if (ps->stats[STAT_ARMOR] >= 100)
                            CG_Draw3DModel( TEXT_ICON_SPACE, y, iconsize, iconsize,
                                                    cgs.media.armorModelRed, 0, origin, angles );
                     else
                            CG_Draw3DModel( TEXT_ICON_SPACE, y, iconsize, iconsize,
                                                    cgs.media.armorModel, 0, origin, angles );
              }
              else
                     if (ps->stats[STAT_ARMOR] >= 100)
                            CG_DrawPic( TEXT_ICON_SPACE, y, iconsize, iconsize, cgs.media.armorIconRed );
                     else
                            CG_DrawPic( TEXT_ICON_SPACE, y, iconsize, iconsize, cgs.media.armorIcon );

              Com_sprintf(temp, sizeof(temp), "%i", cg.snap->ps.stats[ STAT_ARMOR ]);
              CG_DrawStringExt( TEXT_ICON_SPACE * 2 + iconsize, y + ((iconsize / 2) - (BIGCHAR_HEIGHT / 2)), temp,
                     hcolor, qfalse, qtrue, charwidth,charheight, 3 );
              y += iconsize + TEXT_ICON_SPACE;
       }
       y = CG_DrawNHLine(y);
       return y;
}

/*
=================
CG_DrawNHHealth3
=================
*/
static float CG_DrawNHHealth3( void ) {
    int         t;
    vec4_t      hcolor;
    vec4_t      hcolor2;
    //float       *color;
    int         value;
    qhandle_t   icon;
    int         iconsize;
    float       charwidth, charheight;
    char        temp[10];
    float y;
    vec3_t        angles;
    vec3_t        origin;
    playerState_t *ps;
    centity_t     *cent;
    y = 452;
       cent = &cg_entities[cg.snap->ps.clientNum];
       ps = &cg.snap->ps;
       iconsize = ICON_SIZE/2; 
       charwidth = SMALLCHAR_WIDTH;
       charheight = BIGCHAR_HEIGHT;

//       hcolor[0] = hcolor[1] = hcolor[2] = hcolor[3] = 1.0;
//       hcolor[0] = 0;
//       hcolor[1] = 0;
//       hcolor[2] = 1;
//       hcolor[0] = cg_test.value;
//       hcolor[1] = cg_test2.value;
//       hcolor[2] = cg_test3.value;
       hcolor[0] = 0.1;
       hcolor[1] = 0;
       hcolor[2] = 0.1;
       hcolor[3] = 0.33f; // 0.5
       CG_FillRect( 0, 450, 640, 60, hcolor );
//       hcolor[0] = hcolor[1] = hcolor[2] = hcolor[3] = 1.0;
       hcolor[0] = hcolor[3] = 1;
       hcolor[2] = hcolor[1] = 0;

       hcolor2[0] = hcolor2[1] = hcolor2[2] = hcolor2[3] = 1.0;
       CG_FillRect( 0, y-1, 640, 1, hcolor2 );
       CG_FillRect( 0, y-1, 1, 33, hcolor2 );
       CG_FillRect( 639, y-1, 1, 33, hcolor2 );
       CG_FillRect( 0, 479, 640, 1, hcolor2 );
       CG_FillRect( 200-15, y-1, 1, 33, hcolor2 );
       CG_FillRect( 440-15, y-1, 1, 33, hcolor2 );
       CG_FillRect( 470, y-1, 1, 33, hcolor2 );
       if (cgs.gametype < GT_TEAM)
              CG_FillRect( 600, y-1, 1, 33, hcolor2 );
       /*
              CG_FillRect( 200-15, y-1, 300, 1, hcolor2 );
              CG_FillRect( 200-15, y-1, 1, 33, hcolor2 );
              CG_FillRect( 500-15, y-1, 1, 33, hcolor2 );
              CG_FillRect( 200-15, 479, 300, 1, hcolor2 );
       */
       Com_sprintf(temp, sizeof(temp), "%i", cg.snap->ps.stats[ STAT_HEALTH ]);
       CG_DrawStringExt( 200 + iconsize, y + ((iconsize / 2) - (BIGCHAR_HEIGHT / 2)), temp, 
                                  hcolor, qfalse, qtrue, charwidth,charheight, 3 );
       if (cg_draw3dIcons.integer)
              CG_DrawStatusBarHead2( 200,y+1,ICON_SIZE / 2,ICON_SIZE / 2);
       else
              CG_DrawStatusBarHead2( 200,y+2,ICON_SIZE / 2,ICON_SIZE / 2);

       VectorClear( angles );
       CG_FillRect( 280-15, y-1, 1, 33, hcolor2 );
       if ( ps->stats[ STAT_ARMOR ] ) {
              if (cg_draw3dIcons.integer)
              {
                     origin[0] = 90;
                     origin[1] = 0;
                     origin[2] = -10;
                     angles[YAW] = ( cg.time & 2047 ) * 360 / 2048.0;
                     if (ps->stats[STAT_ARMOR] >= 100)
                            CG_Draw3DModel( 280, y+3, iconsize, iconsize,
                                                    cgs.media.armorModelRed, 0, origin, angles );
                     else
                            CG_Draw3DModel( 280, y+3, iconsize, iconsize,
                                                    cgs.media.armorModel, 0, origin, angles );
              }
              else
                     if (ps->stats[STAT_ARMOR] >= 100)
                            CG_DrawPic( 280, y+2, iconsize, iconsize, cgs.media.armorIconRed );
                     else
                            CG_DrawPic( 280, y+2, iconsize, iconsize, cgs.media.armorIcon );

              Com_sprintf(temp, sizeof(temp), "%i", cg.snap->ps.stats[ STAT_ARMOR ]);
              hcolor[0] = hcolor[1] = hcolor[3] = 1;
              hcolor[2] = 0;
              CG_DrawStringExt( 280 + iconsize, y + ((iconsize / 2) - (BIGCHAR_HEIGHT / 2)), temp,
                     hcolor, qfalse, qtrue, charwidth,charheight, 3 );
       }
       for (t = 0; t < MAX_WEAPONS; t++)
       {
//       CG_RegisterWeapon( weap );
        if ( cg.predictedPlayerState.stats[ STAT_WEAPONS ] & ( 1 << t ) ) {
            value = cg.predictedPlayerState.ammo[t];
             hcolor[0] = hcolor[2] = hcolor[3] = 1;
             hcolor[1] = 0.3;
//           hcolor[0] = cg_test.value;
//           hcolor[1] = cg_test2.value;
//           hcolor[2] = cg_test3.value;
//           hcolor[3] = cg_test4.value;
            icon = cg_weapons[t].ammoIcon;
            if (cent->currentState.weapon == t) {
                   CG_FillRect( 360-15, y-1, 1, 33, hcolor2 );
                   if (cg_draw3dIcons.integer)
                          CG_DrawPic( 360, y+3, ICON_SIZE/2.5, ICON_SIZE/2.5, icon );
                   else
                          CG_DrawPic( 360, y+4, ICON_SIZE/2.5, ICON_SIZE/2.5, icon );
                   if ( !value ) {
                       if (cg_draw3dIcons.integer)
                              CG_DrawPic( 360, y+3,ICON_SIZE/2.5,ICON_SIZE/2.5, cgs.media.noammoShader );
                       else
                              CG_DrawPic( 360, y+4,ICON_SIZE/2.5,ICON_SIZE/2.5, cgs.media.noammoShader );
                   }
                   if (value > 0 && value < 1500) { // Don't draw negative ammo
                       Com_sprintf (temp, sizeof(temp), "%i", value);
                       CG_DrawStringExt( 360+iconsize, y + ((iconsize / 2) - (BIGCHAR_HEIGHT / 2)), temp, 
                                         hcolor, qfalse, qtrue, charwidth,charheight, 3 );
                   }
            }
           }
       }
}

/*
================
CG_DrawNHPowerups
================
*/
static float CG_DrawNHPowerups( float y ) {
	int		sorted[MAX_POWERUPS];
	int		sortedTime[MAX_POWERUPS];
    char    temp[10];
	int		i, j, k;
	int		active;
	playerState_t	*ps;
	int		t;
	gitem_t	*item;
	int		x;
	float	size;
	float	f;
    int         value;
    int     iconsize;
    int     color;
	static float colors[2][4] = { 
		{ 0.2, 1.0, 0.2, 1.0 } , { 1.0, 0.2, 0.2, 1.0 } };

	ps = &cg.snap->ps;

	if ( ps->stats[STAT_HEALTH] <= 0 ) {
		return y;
	}

	// sort the list by time remaining
	active = 0;
	for ( i = 0 ; i < MAX_POWERUPS ; i++ ) {
		if ( !ps->powerups[ i ] ) {
			continue;
		}
		t = ps->powerups[ i ] - cg.time;
		// ZOID--don't draw if the power up has unlimited time (999 seconds)
		// This is true of the CTF flags
		if ( t < 0 || t > 999000) {
			continue;
		}

		// insert into the list
		for ( j = 0 ; j < active ; j++ ) {
			if ( sortedTime[j] >= t ) {
				for ( k = active - 1 ; k >= j ; k-- ) {
					sorted[k+1] = sorted[k];
					sortedTime[k+1] = sortedTime[k];
				}
				break;
			}
		}
		sorted[j] = i;
		sortedTime[j] = t;
		active++;
	}

	// draw the icons and timers
       x = TEXT_ICON_SPACE;
       iconsize = ICON_SIZE / 2;
	for ( i = 0 ; i < active ; i++ ) {
		item = BG_FindItemForPowerup( sorted[i] );
    if (item) {
		if ( cg.powerupActive == sorted[i] && 
			cg.time - cg.powerupTime < PULSE_TIME ) {
			f = 1.0 - ( ( (float)cg.time - cg.powerupTime ) / PULSE_TIME );
			size = (ICON_SIZE / 2)* ( 1.0 + ( PULSE_SCALE - 1.0 ) * f );
		} else {
			size = (ICON_SIZE / 2);
		}

                     CG_DrawPic( TEXT_ICON_SPACE, y - ((size - (ICON_SIZE / 2)) / 2),
			size, size, trap_R_RegisterShader( item->icon ) );
    
        if (sortedTime[i] / 1000 > 99)
               Com_sprintf(temp, sizeof(temp), "99");
        else
               Com_sprintf(temp, sizeof(temp), "%i", sortedTime[ i ] / 1000);
        color = 1;
                  CG_DrawStringExt( x + ICON_SIZE / 2, y + ((ICON_SIZE / 4) - (BIGCHAR_HEIGHT / 2)), temp, colors[color], qfalse, qtrue, BIGCHAR_WIDTH, BIGCHAR_HEIGHT, 2 );
                  y += ICON_SIZE / 2 + TEXT_ICON_SPACE;
	}
    }
    if ( active) {
        y = CG_DrawNHLine(y);
    }
	return y;
}

/*
================
CG_DrawNHPowerups2
================
*/
static float CG_DrawNHPowerups2( float y ) {
	int		sorted[MAX_POWERUPS];
	int		sortedTime[MAX_POWERUPS];
    char    temp[8192];
	int		i, j, k;
	int		active;
	playerState_t	*ps;
	int		t;
	gitem_t	*item;
	int		x;
	float	size;
	float	f;
    int         value;
    int     iconsize;
    int     color;
	static float colors[2][4] = { 
		{ 0.2, 1.0, 0.2, 1.0 } , { 1.0, 0.2, 0.2, 1.0 } };

	ps = &cg.snap->ps;

	if ( ps->stats[STAT_HEALTH] <= 0 ) {
		return y;
	}

	// sort the list by time remaining
	active = 0;
	for ( i = 0 ; i < MAX_POWERUPS ; i++ ) {
		if ( !ps->powerups[ i ] ) {
			continue;
		}
		t = ps->powerups[ i ] - cg.time;
		// ZOID--don't draw if the power up has unlimited time (999 seconds)
		// This is true of the CTF flags
		if ( t < 0 || t > 999000) {
			continue;
		}

		// insert into the list
		for ( j = 0 ; j < active ; j++ ) {
			if ( sortedTime[j] >= t ) {
				for ( k = active - 1 ; k >= j ; k-- ) {
					sorted[k+1] = sorted[k];
					sortedTime[k+1] = sortedTime[k];
				}
				break;
			}
		}
		sorted[j] = i;
		sortedTime[j] = t;
		active++;
	}
       if (active && cgs.gametype != GT_CTF)
       {
              y += ICON_SIZE /2 + TEXT_ICON_SPACE+15;
              y = CG_DrawNHLine(y);
              y -= 15;
       }
	// draw the icons and timers
       x = 570;
       iconsize = ICON_SIZE / 2;
	for ( i = 0 ; i < active ; i++ ) {
		item = BG_FindItemForPowerup( sorted[i] );
    if (item) {
		if ( cg.powerupActive == sorted[i] && 
			cg.time - cg.powerupTime < PULSE_TIME ) {
			f = 1.0 - ( ( (float)cg.time - cg.powerupTime ) / PULSE_TIME );
			size = (ICON_SIZE / 2)* ( 1.0 + ( PULSE_SCALE - 1.0 ) * f );
		} else {
			size = (ICON_SIZE / 2);
		}

                     CG_DrawPic( x, y - ((size - (ICON_SIZE / 2)) / 2)-ICON_SIZE / 2 + TEXT_ICON_SPACE,
			size, size, trap_R_RegisterShader( item->icon ) );

        if (sortedTime[i] / 1000 > 99)
               Com_sprintf(temp, sizeof(temp), "99");
        else
               Com_sprintf(temp, sizeof(temp), "%i", sortedTime[ i ] / 1000);
        color = 1;
                  CG_DrawStringExt( x + ICON_SIZE / 2, y + ((ICON_SIZE / 4) - (BIGCHAR_HEIGHT / 2))-ICON_SIZE / 2 + TEXT_ICON_SPACE, temp, colors[color], qfalse, qtrue, BIGCHAR_WIDTH, BIGCHAR_HEIGHT, 2 );
                  y -= ICON_SIZE / 2 + TEXT_ICON_SPACE;
	}
    }
    if ( active) {
        y = CG_DrawNHLine(y);
    }
	value = cg.snap->ps.stats[STAT_HOLDABLE_ITEM];
	if ( value ) {
        if (!active && cgs.gametype != GT_CTF)
        {
               y+= ICON_SIZE - TEXT_ICON_SPACE;
               y = CG_DrawNHLine(y); // bottom line
        }
        else if (!active && cgs.gametype == GT_CTF)
              y+= ICON_SIZE/2 - TEXT_ICON_SPACE;
        CG_RegisterItemVisuals( value );
        iconsize = ICON_SIZE / 3;
        CG_DrawPic( 570, y-ICON_SIZE/2, iconsize, iconsize, cg_items[ value ].icon );
        y -= ICON_SIZE / 2 + TEXT_ICON_SPACE;
        y = CG_DrawNHLine(y); // top line
	}
	return y;
}

/*
=================
CG_DrawNHFlag
=================
*/
static float CG_DrawNHFlag( float x, float y, int flag, int status, qhandle_t icon ) {
    gitem_t     *item;
   // Display flag status
    item = BG_FindItemForPowerup( flag );
    if (item) {
        // draw icon
        if ( status == 0 || (cg.time % 1000) < 500 ) {
            // blink flag once a second
            CG_DrawPic( x, y, ICON_SIZE / 2, ICON_SIZE / 2, icon );
        }                
        if ( cg.predictedPlayerState.powerups[flag] )
        {
            CG_DrawPic( x, y, ICON_SIZE / 2, ICON_SIZE / 2, cgs.media.selectShader );
        }
	}
    return y;
}

/*
=================
CG_DrawNHFlagStatus
=================
*/
static float CG_DrawNHFlagStatus( float y ) {
       int x;
       x = TEXT_ICON_SPACE;
	if ( cgs.gametype == GT_CTF ) {
        if ( cgs.blueflag < 0 || cgs.blueflag > 2 )
        {
            // no leet info, just draw flag if we have it;
            if ( cg.predictedPlayerState.powerups[PW_BLUEFLAG] ) {
                CG_DrawNHFlag(x, y, PW_BLUEFLAG, 1, cgs.media.blueFlagShader[ 1 ]);
                y += ICON_SIZE / 2 + TEXT_ICON_SPACE;
                y = CG_DrawNHLine(y);
            } else if ( cg.predictedPlayerState.powerups[PW_REDFLAG] ) {
                CG_DrawNHFlag(x, y, PW_REDFLAG, 1, cgs.media.redFlagShader[ 1 ]);
                y += ICON_SIZE / 2 + TEXT_ICON_SPACE;
                y = CG_DrawNHLine(y);
            }
            return y;
        }
        
        // Display flag status
        CG_DrawNHFlag(x, y, PW_BLUEFLAG, cgs.blueflag, cgs.media.blueFlagShader[ cgs.blueflag ]);
        CG_DrawNHFlag(x * 5 + ICON_SIZE / 2, y, PW_REDFLAG, cgs.redflag, cgs.media.redFlagShader[ cgs.redflag ]);
        if (cg_hud.integer != 5)
               y += ICON_SIZE / 2 + TEXT_ICON_SPACE;
        y = CG_DrawNHLine(y);
	}
    return y;
}

/*
=================
CG_DrawNHFlagStatus2
=================
*/
static float CG_DrawNHFlagStatus2( float y ) {
       int x;
       x = 570;
	if ( cgs.gametype == GT_CTF ) {
        if ( cgs.blueflag < 0 || cgs.blueflag > 2 )
        {
            // no leet info, just draw flag if we have it;
            if ( cg.predictedPlayerState.powerups[PW_BLUEFLAG] ) {
                CG_DrawNHFlag(x, y, PW_BLUEFLAG, 1, cgs.media.blueFlagShader[ 1 ]);
                y += ICON_SIZE / 2 + TEXT_ICON_SPACE;
                y = CG_DrawNHLine(y);
            } else if ( cg.predictedPlayerState.powerups[PW_REDFLAG] ) {
                CG_DrawNHFlag(x, y, PW_REDFLAG, 1, cgs.media.redFlagShader[ 1 ]);
                y += ICON_SIZE / 2 + TEXT_ICON_SPACE;
                y = CG_DrawNHLine(y);
            }
            return y;
        }
        // Display flag status
        CG_DrawNHFlag(x, y, PW_BLUEFLAG, cgs.blueflag, cgs.media.blueFlagShader[ cgs.blueflag ]);
        CG_DrawNHFlag(x-(TEXT_ICON_SPACE) + (TEXT_ICON_SPACE * 5 ) + ICON_SIZE / 2, y, PW_REDFLAG, cgs.redflag, cgs.media.redFlagShader[ cgs.redflag ]);
        y+= ICON_SIZE/2 + TEXT_ICON_SPACE;
        y = CG_DrawNHLine(y);
	}
    return y;
}

/*
=====================
CG_DrawNHCrosshairNames
=====================
*/
static void CG_DrawNHCrosshairNames( void ) {
	float		*color;
	char		*name;
	float		w;
    char        temp[128];
    int         iconsize = ICON_SIZE / 3;
    float       h = 270;
    clientInfo_t target;
//EQ--
   char  ename[100];
   int   i;
	float ecolor[4];
//--EQ

	if ( !cg_drawCrosshair.integer ) {
		return;
	}
	if ( !cg_drawCrosshairNames.integer ) {
		return;
	}
//       if ( cg.renderingThirdPerson ) {
//              return;
//       }

	// scan the known entities to see if the crosshair is sighted on one
	CG_ScanForCrosshairEntity();

	// draw the name of the player being looked at
	color = CG_FadeColor( cg.crosshairClientTime, 1000 );
	if ( !color ) {
		trap_R_SetColor( NULL );
		return;
	}

    target = cgs.clientinfo[ cg.crosshairClientNum ];
       if (cgs.clientinfo[ cg.crosshairClientNum ].powerups & ( 1 << PW_PROTECT ))
              name = va("%s - Spawn Protection",cgs.clientinfo[ cg.crosshairClientNum ].name);
       else
           name = va("%s",cgs.clientinfo[ cg.crosshairClientNum ].name);
	w = CG_DrawStrlen( name ) * BIGCHAR_WIDTH;
       CG_DrawBigString( 320 - w / 2, h, name, color[3] * 0.5 );
    if ( cgs.gametype >= GT_TEAM && target.team == cgs.clientinfo[ cg.snap->ps.clientNum ].team && cg_drawTeamOverlay.integer)
    {
        h += BIGCHAR_HEIGHT;
        Com_sprintf(temp, sizeof(temp), "[  Health:%3i   Armor:%3i]", target.health, target.armor);
        w = 320 - (CG_DrawStrlen(temp) * SMALLCHAR_WIDTH) / 2;
        CG_DrawSmallString( w, h, temp, color[3] * 0.5 );
        w += SMALLCHAR_WIDTH;
        CG_DrawPic(w, h, iconsize, iconsize, target.modelIcon);
        w += (SMALLCHAR_WIDTH * 4.0) + iconsize;
        if (target.armor >= 100)
               CG_DrawPic(w+50, h, iconsize, iconsize, cgs.media.armorIconRed);
        else
               CG_DrawPic(w+50, h, iconsize, iconsize, cgs.media.armorIcon);
    }
//EQ--
   	for (i=0; i<cg.numScores; i++)
   		if (cg.scores[i].client == cg.crosshairClientNum)
            break;

      switch (i)
      {
         case 0:
            ecolor[0] = 0;
            ecolor[1] = 0;
            ecolor[2] = 1;
            Com_sprintf(ename, sizeof(ename), "[1st with %d frags]", cgs.clientinfo[cg.crosshairClientNum].score);
            break;
         case 1:
            ecolor[0] = 1;
            ecolor[1] = 0;
            ecolor[2] = 0;
            Com_sprintf(ename, sizeof(ename), "[2nd with %d frags]", cgs.clientinfo[cg.crosshairClientNum].score);
            break;
         case 2:
            ecolor[0] = 1;
            ecolor[1] = 1;
            ecolor[2] = 0;
            Com_sprintf(ename, sizeof(ename), "[3rd with %d frags]", cgs.clientinfo[cg.crosshairClientNum].score);
            break;
         default:
            ecolor[0] = 1;
            ecolor[1] = 1;
            ecolor[2] = 1;
            Com_sprintf(ename, sizeof(ename), "[%dth with %d frags]", i+1, cgs.clientinfo[cg.crosshairClientNum].score);
      }
//      ecolor[3] = color[3] * 0.5;
      ecolor[3] = 1;
      if (cg_draw3dIcons.integer)
      {
         w = CG_DrawStrlen(ename) * SMALLCHAR_WIDTH;
          CG_DrawSmallStringColor(320 - w / 2, h+BIGCHAR_HEIGHT+2, ename, ecolor);
      }
      else
      {
         w = CG_DrawStrlen(ename) * BIGCHAR_WIDTH;
          CG_DrawBigStringColor(320 - w / 2, h+BIGCHAR_HEIGHT+2, ename, ecolor);
      }
//--EQ

	trap_R_SetColor( NULL );
}

/*
=================
CG_DrawNHScores

Draw the small two score display
=================
*/
static float CG_DrawNHScores( float y ) {
	const char	*s;
	int			s1, s2, score;
	int			x, w;
	int			v;
	vec4_t		color;
	float		y1;

    s1 = cgs.scores1;
	s2 = cgs.scores2;

	y -=  BIGCHAR_HEIGHT + 8;

	y1 = y;

	// draw from the right side to left
	if ( cgs.gametype >= GT_TEAM ) {
		x = 640;

		color[0] = 0;
		color[1] = 0;
		color[2] = 1;
		color[3] = 0.33;
		s = va( "%2i", s2 );
		w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
		x -= w;
		CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
		if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_BLUE ) {
			CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
		}
		CG_DrawBigString( x + 4, y, s, 1.0F);

		color[0] = 1;
		color[1] = 0;
		color[2] = 0;
		color[3] = 0.33;
		s = va( "%2i", s1 );
		w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
		x -= w;
		CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
		if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_RED ) {
			CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
		}
		CG_DrawBigString( x + 4, y, s, 1.0F);

		if ( cgs.gametype == GT_CTF ) {
			v = cgs.capturelimit;
		} else {
			v = cgs.fraglimit;
		}
		if ( v ) {
			s = va( "%2i", v );
			w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
			x -= w;
			CG_DrawBigString( x + 4, y, s, 1.0F);
		}

	} else {
		qboolean	spectator;

		x = 640;
		score = cg.snap->ps.persistant[PERS_SCORE];
		spectator = ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR );

		// always show your score in the second box if not in first place
		if ( s1 != score ) {
			s2 = score;
		}
		if ( s2 != SCORE_NOT_PRESENT ) {
			s = va( "%2i", s2 );
			w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
			x -= w;
			if ( !spectator && score == s2 && score != s1 ) {
				color[0] = 1;
				color[1] = 0;
				color[2] = 0;
				color[3] = 0.33;
				CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
				CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
			} else {
				color[0] = 0.5;
				color[1] = 0.5;
				color[2] = 0.5;
				color[3] = 0.33;
				CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
			}	
			CG_DrawBigString( x + 4, y, s, 1.0F);
		}

		// first place
		if ( s1 != SCORE_NOT_PRESENT ) {
			s = va( "%2i", s1 );
			w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
			x -= w;
			if ( !spectator && score == s1 ) {
				color[0] = 0;
				color[1] = 0;
				color[2] = 1;
				color[3] = 0.33;
				CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
				CG_DrawPic( x, y-4, w, BIGCHAR_HEIGHT+8, cgs.media.selectShader );
			} else {
				color[0] = 0.5;
				color[1] = 0.5;
				color[2] = 0.5;
				color[3] = 0.33;
				CG_FillRect( x, y-4,  w, BIGCHAR_HEIGHT+8, color );
			}	
			CG_DrawBigString( x + 4, y, s, 1.0F);
		}

		if ( cgs.fraglimit ) {
			s = va( "%2i", cgs.fraglimit );
			w = CG_DrawStrlen( s ) * BIGCHAR_WIDTH + 8;
			x -= w;
			CG_DrawBigString( x + 4, y, s, 1.0F);
		}

	}

	return y1 - 8;
}

    #define kSCANNER_UNIT 24
    #define kSCANNER_RANGE 100
    
    /*
    =================
    CG_DrawScanner
    =================
    */
    static float CG_DrawScanner( float y ) 
    {
        int x;
        int w,h;
        int sx,sy;
        int rv;
        vec3_t v, norm, dp;
        float len;
        float height;
        playerpos_t *player;
        centity_t *scanner;

        if ( CG_DrawScoreboard() ) {
               return;
        }
        //dont draw anything if the scanner is off
        if (!cg_drawscanner.integer || g_scanner.integer == 0)
               return;
        if (cg_hud.integer >= 1 && cg_hud.integer <= 3)
              x = 470; //offset from left of screen
        else if (cg_hud.integer == 4)
              x = 38;
        else
              x = 20; //offset from left of screen
        rv = y + 155;
        y = 30; //offset from top of scren
        if (cg_hud.integer == 5)
              y = 20;
        w = 160; //width of scanner on screen
        h = w; //height of scanner on screen

        //draw the scanner
               CG_DrawPic( x, y, w, h, cgs.media.scannerShader);

        //get info about the scanner positon
        scanner = &cg_entities[cg.snap->ps.clientNum];

        //check the stored player positions
        player = cg_playerOrigins;
        while(player->valid != kENTRY_EOL)
        {
            //get a vector from the scanner position to the current player
            VectorSubtract (scanner->lerpOrigin, player->pos, v);

            //store the height component
            height = v[2];

            //remove the height component from the vector
            v[2] = 0;

            //calc the distance to the player and scale it to the scanner scale
            len = VectorLength( v ) / kSCANNER_UNIT;

            //is the player within range?
            if(len < kSCANNER_RANGE)
            {
                //create a vector pointing stright down
                norm[0] = 0;
                norm[1] = 0;
                norm[2] = -1;

                //normalise the vector to the player
                VectorNormalize( v );

                //rotate the player about the scanners view angle
                RotatePointAroundVector( dp, norm, v, scanner->lerpAngles[1]);

                //scale to fit current size of scanner
                VectorScale(dp,len*(w/2)/kSCANNER_RANGE,dp);

                // calc screen (x,y) (-4 = half dot width, so we can centre the graphic)
                sx = (x + (w/2) + dp[1]) - 4;
                sy = (y + (h/2) + dp[0]) - 4;

                //draw the dot
                if (height < -32) //player is above scanner
                       CG_DrawPic( sx, sy, 8, 8, cgs.media.scannerBlipUpShader);
                else if (height > 32) //player is below scanner
                       CG_DrawPic( sx, sy, 8, 8, cgs.media.scannerBlipDownShader);
                else
                       CG_DrawPic( sx, sy, 8, 8, cgs.media.scannerBlipShader);
            }
            
            //move on to next entry in the array
            player++;
        }
       return rv;
    }

/*
=================
CG_DrawNewHud
=================
*/
static void CG_DrawNewHud( void ) {
    float y;

	// if we are taking a levelshot for the menu, don't draw anything
	if ( cg.levelShot ) {
		return;
	}
       if ( CG_DrawScoreboard() ) {
              return;
       }

       if ( cg_draw2D.integer == 0 ) {
		return;
	}
	if ( cg.snap->ps.pm_type == PM_INTERMISSION ) {
		CG_DrawIntermission();
		return;
	}
	if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR ) {
		CG_DrawSpectator();
		CG_DrawCrosshair();
		CG_DrawNHCrosshairNames();
              CG_DrawSpecMenu();
	} else {
		// don't draw any status if dead
		if ( cg.snap->ps.stats[STAT_HEALTH] > 0 ) {
	        CG_DrawTeamBackground( 0, 420, 640, 60, 0.33, cg.snap->ps.persistant[PERS_TEAM] );
                     if ( cg_drawStatus.integer != 0 ) {
                            CG_DrawNHHealth();
                     }
                     if( cg_directionaldamage.integer )
                            CG_DrawDamageIcons();
			CG_DrawAmmoWarning();
			CG_DrawCrosshair();
			CG_DrawNHCrosshairNames();
                     if (cg_weaponselectstyle.integer == 1)
                            CG_DrawNHAmmo2();
                     else
                            CG_DrawWeaponSelect();
			CG_DrawReward();
                     CG_DrawModule();            // The SARACEN's Modules
                     
                     CG_DrawScanner(y);

            y = 48;
            if ( cg_drawStatus.integer != 0 ) {
                   y = CG_DrawNHLine(y);
                   y = CG_DrawNHFlagStatus(y);
                   y = CG_DrawNHAmmo(y);
                   y = CG_DrawNHPowerups(y);
            }
		}
		if ( cgs.gametype >= GT_TEAM ) {
			CG_DrawTeamInfo();
		}
	}


    y = 480 - ICON_SIZE;
	if ( cgs.gametype >= GT_TEAM && cg_drawTeamOverlay.integer == 2 ) {
		y = CG_DrawTeamOverlay( y, qtrue, qfalse );
	} 

//       y = CG_DrawNHScores( y );
       y = CG_DrawScores( y );

    CG_DrawVote();
//       CG_DrawModule();
	CG_DrawLagometer();

	CG_DrawUpperRight();

	CG_DrawLowerLeft();

	if ( !CG_DrawFollow() ) {
		CG_DrawWarmup();
	}
       if (randomweap.integer == 1 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawWeaponWarning();
       if (startwithpowerups.integer == 7 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawPowerupWarning();
       if (g_campprotect.integer && !cg.predictedPlayerState.powerups[PW_PROTECT])
              CG_DrawCampWarning();

	// don't draw center string if scoreboard is up
	if ( !CG_DrawScoreboard() ) {
		CG_DrawCenterString();
	}
}

/*
=================
CG_DrawNewHud2 (Normal health bar)
=================
*/
static void CG_DrawNewHud2( void ) {
    float y;

	// if we are taking a levelshot for the menu, don't draw anything
	if ( cg.levelShot ) {
		return;
	}
       if ( CG_DrawScoreboard() ) {
              return;
       }

	if ( cg_draw2D.integer == 0 ) {
		return;
	}
	if ( cg.snap->ps.pm_type == PM_INTERMISSION ) {
		CG_DrawIntermission();
		return;
	}
	if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR ) {
		CG_DrawSpectator();
		CG_DrawCrosshair();
		CG_DrawNHCrosshairNames();
              CG_DrawSpecMenu();
	} else {
		// don't draw any status if dead
		if ( cg.snap->ps.stats[STAT_HEALTH] > 0 ) {
	        CG_DrawTeamBackground( 0, 420, 640, 60, 0.33, cg.snap->ps.persistant[PERS_TEAM] );
			CG_DrawStatusBar();
                     if( cg_directionaldamage.integer )
                            CG_DrawDamageIcons();
			CG_DrawAmmoWarning();
			CG_DrawCrosshair();
			CG_DrawNHCrosshairNames();
                     if (cg_weaponselectstyle.integer == 1)
                            CG_DrawNHAmmo2();
                     else
                            CG_DrawWeaponSelect();
			CG_DrawReward();
                     CG_DrawModule();            // The SARACEN's Modules
                     CG_DrawScanner(y);

            y = 48;
            if ( cg_drawStatus.integer != 0 ) {
                   y = CG_DrawNHLine(y);
                   y = CG_DrawNHFlagStatus(y);
                   y = CG_DrawNHAmmo(y);
                   y = CG_DrawNHPowerups(y);
            }

		}
		if ( cgs.gametype >= GT_TEAM ) {
			CG_DrawTeamInfo();
		}
	}


    y = 480 - ICON_SIZE;
	if ( cgs.gametype >= GT_TEAM && cg_drawTeamOverlay.integer == 2 ) {
		y = CG_DrawTeamOverlay( y, qtrue, qfalse );
	} 

//       y = CG_DrawNHScores( y );
       y = CG_DrawScores( y );

    CG_DrawVote();
//                     CG_DrawModule();
	CG_DrawLagometer();

	CG_DrawUpperRight();

	CG_DrawLowerLeft();

	if ( !CG_DrawFollow() ) {
		CG_DrawWarmup();
	}
       if (randomweap.integer == 1 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawWeaponWarning();
       if (startwithpowerups.integer == 7 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawPowerupWarning();
       if (g_campprotect.integer && !cg.predictedPlayerState.powerups[PW_PROTECT])
              CG_DrawCampWarning();

	// don't draw center string if scoreboard is up
	if ( !CG_DrawScoreboard() ) {
		CG_DrawCenterString();
	}
}

/*
=================
CG_DrawNewHud3 (Every stat is on left hand corner)
=================
*/
static void CG_DrawNewHud3( void ) {
    float y;

	// if we are taking a levelshot for the menu, don't draw anything
	if ( cg.levelShot ) {
		return;
	}
       if ( CG_DrawScoreboard() ) {
              return;
       }

	if ( cg_draw2D.integer == 0 ) {
		return;
	}
	if ( cg.snap->ps.pm_type == PM_INTERMISSION ) {
		CG_DrawIntermission();
		return;
	}
	if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR ) {
		CG_DrawSpectator();
		CG_DrawCrosshair();
		CG_DrawNHCrosshairNames();
              CG_DrawSpecMenu();
	} else {
		// don't draw any status if dead
		if ( cg.snap->ps.stats[STAT_HEALTH] > 0 ) {
	        CG_DrawTeamBackground( 0, 420, 640, 60, 0.33, cg.snap->ps.persistant[PERS_TEAM] );
                     if( cg_directionaldamage.integer )
                            CG_DrawDamageIcons();
			CG_DrawAmmoWarning();
			CG_DrawCrosshair();
			CG_DrawNHCrosshairNames();
                     if (cg_weaponselectstyle.integer == 1)
                            CG_DrawNHAmmo2();
                     else
                            CG_DrawWeaponSelect();
			CG_DrawReward();
                     CG_DrawModule();            // The SARACEN's Modules
                     CG_DrawScanner(y);

            y = 48;
            if ( cg_drawStatus.integer != 0 ) {
                   y = CG_DrawNHLine(y);
                   y = CG_DrawNHFlagStatus(y);
                   y = CG_DrawNHAmmo(y);
                   y = CG_DrawNHPowerups(y);
                   y = CG_DrawNHHealth2(y);
            }
		}
		if ( cgs.gametype >= GT_TEAM ) {
			CG_DrawTeamInfo();
		}
	}


    y = 480 - ICON_SIZE;
	if ( cgs.gametype >= GT_TEAM && cg_drawTeamOverlay.integer == 2 ) {
		y = CG_DrawTeamOverlay( y, qtrue, qfalse );
	} 

//       y = CG_DrawNHScores( y );
       y = CG_DrawScores( y );

    CG_DrawVote();
//                     CG_DrawModule();
	CG_DrawLagometer();

	CG_DrawUpperRight();

	CG_DrawLowerLeft();

	if ( !CG_DrawFollow() ) {
		CG_DrawWarmup();
	}
       if (randomweap.integer == 1 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawWeaponWarning();
       if (startwithpowerups.integer == 7 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawPowerupWarning();
       if (g_campprotect.integer && !cg.predictedPlayerState.powerups[PW_PROTECT])
              CG_DrawCampWarning();

	// don't draw center string if scoreboard is up
	if ( !CG_DrawScoreboard() ) {
		CG_DrawCenterString();
	}
}

/*
=================
CG_DrawNewHud5
=================
*/
static void CG_DrawNewHud5( void ) {
    float y;

	// if we are taking a levelshot for the menu, don't draw anything
	if ( cg.levelShot ) {
		return;
	}
       if ( CG_DrawScoreboard() ) {
              return;
       }

	if ( cg_draw2D.integer == 0 ) {
		return;
	}
	if ( cg.snap->ps.pm_type == PM_INTERMISSION ) {
		CG_DrawIntermission();
		return;
	}
	if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR ) {
		CG_DrawSpectator();
		CG_DrawCrosshair();
		CG_DrawNHCrosshairNames();
              CG_DrawSpecMenu();
	} else {
		// don't draw any status if dead
		if ( cg.snap->ps.stats[STAT_HEALTH] > 0 ) {
	        CG_DrawTeamBackground( 0, 420, 640, 60, 0.33, cg.snap->ps.persistant[PERS_TEAM] );
                     if ( cg_drawStatus.integer != 0 ) {
                            CG_DrawNHHealth3();
                     }
                     if( cg_directionaldamage.integer )
                            CG_DrawDamageIcons();
			CG_DrawAmmoWarning();
			CG_DrawCrosshair();
			CG_DrawNHCrosshairNames();
			CG_DrawReward();
                     CG_DrawModule();            // The SARACEN's Modules

            y = 48;
            if ( cg_drawStatus.integer != 0 ) {
               if (cg_drawscanner.integer && g_scanner.integer)
                   y = CG_DrawScanner(y);
               else
                   y += 155;
                   if (cgs.gametype == GT_CTF)
                          y = CG_DrawNHLine(y);
                   y = CG_DrawNHFlagStatus2(y);
                   CG_DrawNHAmmo2();
                   if (cgs.gametype == GT_CTF)
                          y -= ICON_SIZE;
                   y = CG_DrawNHPowerups2(y);
            }
		}
		if ( cgs.gametype >= GT_TEAM ) {
			CG_DrawTeamInfo();
		}
	}


    y = 480 - ICON_SIZE;
	if ( cgs.gametype >= GT_TEAM && cg_drawTeamOverlay.integer == 2 ) {
		y = CG_DrawTeamOverlay( y, qtrue, qfalse );
	} 

//       y = CG_DrawNHScores( y );
//       y = CG_DrawScores( y );
       y = CG_DrawScores2( y );

    CG_DrawVote();
//                     CG_DrawModule();
	CG_DrawLagometer();

       CG_DrawUpperRight();

       CG_DrawLowerLeft();

	if ( !CG_DrawFollow() ) {
		CG_DrawWarmup();
	}
       if (randomweap.integer == 1 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawWeaponWarning();
       if (startwithpowerups.integer == 7 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawPowerupWarning();
       if (g_campprotect.integer && !cg.predictedPlayerState.powerups[PW_PROTECT])
              CG_DrawCampWarning();

	// don't draw center string if scoreboard is up
	if ( !CG_DrawScoreboard() ) {
		CG_DrawCenterString();
	}
}


//==================================================================================

//EQ--    Wow! I actually bothered to make my own function to put my code in!
// Also see cg_weapons.c: 1129
void E_DrawWeaponSelect (void)
{
       int      i,weap,count,t;
	int      bits;
	int      x, y;
   float    colour[4]      = {1.0, 1.0, 1.0, 1.0},
            barcolour[4]   = {1.0, 1.0, 1.0, 0.9},
            barbgcolour[4] = {0.7, 0.6, 0.1, 0.2};
   int      ammoheight;
	centity_t	*cent;

	// don't display if dead
	if ( cg.predictedPlayerState.stats[STAT_HEALTH] <= 0 )
		return;

	bits = cg.snap->ps.stats[ STAT_WEAPONS ];

	x = 6;
       y = 66;
       count = 0;
      for ( i = 1 ; i < NUM_WEAPS ; i++ ) { //WarZone 
        if ( bits & ( 1 << i ) ) { 
          count++; 
        } 
      } 

    weap = weaponRawOrder[NUM_WEAPS - 1]; //WarZone -- select last weapon 
    for ( t = 1; t < NUM_WEAPS+1; t++ ) {
        weap = NextWeapon( weap );

        if ( !( bits & ( 1 << weap ) ) ) {
          continue; 
        } 

              CG_RegisterWeapon(weap);

		// draw weapon icon
              CG_DrawPic(x, y, 16, 16, cg_weapons[weap].weaponIcon);

		// draw selection marker
              cent = &cg_entities[cg.snap->ps.clientNum];
              if (weap == cent->currentState.weapon)
                     CG_DrawPic(x-2, y-2, 20, 20, cgs.media.selectShader);

		// no ammo, draw cross on top
              if (!cg.snap->ps.ammo[weap])
                     CG_DrawPic(x, y, 16, 16, cgs.media.noammoShader);

       #define MIN(A,B)  ((A)>(B)?(B):(A))  /* 'Min' macro definition */

      // Draw ammo bar
      if (cg.snap->ps.ammo[weap] >= 0 && cg.snap->ps.ammo[weap] < 1500)
             ammoheight = MIN(20, (int)(0.9 + cg.snap->ps.ammo[weap] / 5.0));
      else
             ammoheight = MIN(20, (int)(0.9 - 1 / 5.0));
      CG_FillRect(x+19, y-2,                    2, 20-ammoheight, barbgcolour);
      CG_FillRect(x+19, y-2+(20-ammoheight),   2, ammoheight,    barcolour);
              y += 29;
	}
}
//--EQ

//EQ--
void E_DrawKilledDisplay (void)
{
   //CG_DrawStringExt( 320 - w * cw/2, 70, s, colorWhite, qfalse, qtrue, cw, (int)(cw * 1.5), 0 );
}
//--EQ

/*
=================
CG_Draw2D
=================
*/
static void CG_Draw2D( void ) {
#ifdef MISSIONPACK
	if (cgs.orderPending && cg.time > cgs.orderTime) {
		CG_CheckOrderPending();
	}
#endif
	// if we are taking a levelshot for the menu, don't draw anything
	if ( cg.levelShot ) {
		return;
	}

	if ( cg_draw2D.integer == 0 ) {
		return;
	}

	if ( cg.snap->ps.pm_type == PM_INTERMISSION ) {
		CG_DrawIntermission();
		return;
	}

	if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR ) {
		CG_DrawSpectator();
		CG_DrawCrosshair();
//              CG_DrawCrosshairNames();
              CG_DrawNHCrosshairNames();
              CG_DrawSpecMenu();
       } else {
		// don't draw any status if dead or the scoreboard is being explicitly shown
		if ( !cg.showScores && cg.snap->ps.stats[STAT_HEALTH] > 0 ) {

#ifdef MISSIONPACK
			if ( cg_drawStatus.integer ) {
				Menu_PaintAll();
				CG_DrawTimedMenus();
			}
#else
			CG_DrawStatusBar();
#endif
      
			CG_DrawAmmoWarning();
                     if( cg_directionaldamage.integer )
                            CG_DrawDamageIcons();

#ifdef MISSIONPACK
			CG_DrawProxWarning();
#endif      
			CG_DrawCrosshair();
//                     CG_DrawCrosshairNames();
                     CG_DrawNHCrosshairNames();
                     if (cg_weaponselectstyle.integer == 1)
                            CG_DrawNHAmmo2();
                     else
                            CG_DrawWeaponSelect();

#ifndef MISSIONPACK
			CG_DrawHoldableItem();
//                     CG_DrawModule();
#else
			//CG_DrawPersistantPowerup();
#endif
			CG_DrawReward();
                     CG_DrawModule();            // The SARACEN's Modules
                     CG_DrawScanner(0);
                   if (cg_hud.integer == 4)
                          E_DrawWeaponSelect();
		}
    
		if ( cgs.gametype >= GT_TEAM ) {
#ifndef MISSIONPACK
			CG_DrawTeamInfo();
#endif
		}
	}

	CG_DrawVote();

	CG_DrawTeamVote();

	CG_DrawLagometer();

#ifdef MISSIONPACK
	if (!cg_paused.integer) {
		CG_DrawUpperRight();
	}
#else
       if (cg_hud.integer != 5)
	CG_DrawUpperRight();
#endif

#ifndef MISSIONPACK
       if (cg_hud.integer != 5)
	CG_DrawLowerRight();

	CG_DrawLowerLeft();
#endif

	if ( !CG_DrawFollow() ) {
		CG_DrawWarmup();
	}
       if (randomweap.integer == 1 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawWeaponWarning();
       if (startwithpowerups.integer == 7 && !g_instagib.integer && !cg.predictedPlayerState.powerups[PW_CAMP])
              CG_DrawPowerupWarning();
       if (g_campprotect.integer && !cg.predictedPlayerState.powerups[PW_PROTECT])
              CG_DrawCampWarning();

	// don't draw center string if scoreboard is up
	cg.scoreBoardShowing = CG_DrawScoreboard();
	if ( !cg.scoreBoardShowing) {
		CG_DrawCenterString();
	}
}


static void CG_DrawTourneyScoreboard() {
#ifdef MISSIONPACK
#else
	CG_DrawOldTourneyScoreboard();
#endif
}

/*
=====================
CG_DrawActive

Perform all drawing needed to completely fill the screen
=====================
*/
void CG_DrawActive( stereoFrame_t stereoView ) {
	float		separation;
	vec3_t		baseOrg;

	// optionally draw the info screen instead
	if ( !cg.snap ) {
		CG_DrawInformation();
		return;
	}

	// optionally draw the tournement scoreboard instead
	if ( cg.snap->ps.persistant[PERS_TEAM] == TEAM_SPECTATOR &&
		( cg.snap->ps.pm_flags & PMF_SCOREBOARD ) ) {
		CG_DrawTourneyScoreboard();
		return;
	}

	switch ( stereoView ) {
	case STEREO_CENTER:
		separation = 0;
		break;
	case STEREO_LEFT:
		separation = -cg_stereoSeparation.value / 2;
		break;
	case STEREO_RIGHT:
		separation = cg_stereoSeparation.value / 2;
		break;
	default:
		separation = 0;
		CG_Error( "CG_DrawActive: Undefined stereoView" );
	}


	// clear around the rendered view if sized down
	CG_TileClear();

	// offset vieworg appropriately if we're doing stereo separation
	VectorCopy( cg.refdef.vieworg, baseOrg );
	if ( separation != 0 ) {
		VectorMA( cg.refdef.vieworg, -separation, cg.refdef.viewaxis[1], cg.refdef.vieworg );
	}

	// draw 3D view
	trap_R_RenderScene( &cg.refdef );

	// restore original viewpoint if running stereo
	if ( separation != 0 ) {
		VectorCopy( baseOrg, cg.refdef.vieworg );
	}

	// draw status bar and other floating elements
       switch(cg_hud.integer)
       {
       case 1:
              CG_DrawNewHud();
              break;
       case 2:
              CG_DrawNewHud2();
              break;
       case 3:
              CG_DrawNewHud3();
              break;
       case 5:
              CG_DrawNewHud5();
              break;
       default:
              CG_Draw2D();
              break;
       }
}



