unit board;

{
   Main unit for "Nuclear" game.

   Current version:            0.03b
   Previous Released Version:  0.03a

   Date: (this version) v0.03b - 12 November 2001

   Authors: Peter E. Williams
        &
            Michael J. Phillips

   Other units in "Nuclear" project are:

         About    - about (program info) window, called
                    with "showmodal" call & closed with
                    OK button.

         Hiscores - Used to display High Score table.

         Congrats - Used to input winning Player's
                    name for High Score Table (called by
                    Hiscores unit), when a new high score is
                    added to the table.

                          This unit is *not* called if the
                    High Score Table is simply being displayed
                    but not updated (the HiScores unit is called
                    instead).

         Unit3    - used to store a boolean variable passed to
                    the about unit.

         MyIniFiles - Based on IniFiles.pas with 2 new procedures:
                      MyReadBool & MyWriteBool, and data type of
                      tMyIniFile

         BTOdeum.pas - (TBTBeeper) freeware component from
                       Delphi Super Page with my modifications.

   Known bugs:
   (Previously there were 3 known bugs... all now fixed, and documented
    below).

   (this list added at v0.01m)

     #1 *** Fixed at v0.01m by PEW. Caused by tstringgrid row and height being set too
          small at 241 each. Corrected by setting row and height to 243.

          Description of bug: When the user makes a move on the bottom row the
          entire tstringgrid scroll down leaving a gap of white space along the
          bottom of the stringgrid (row 1 goes out of view). Similarly when the
          user make a move in the right hand column the stringgrid scrolls right
          leaving a gap down the right hand edge of the stringgrid (column 1
          goes out of view).

     #2 : *** Fixed at v0.01m by PEW. Fixed by changing moving code in
          procedure StringGrid1Click to new procedure CellSelected and also
          adding new procedures StringGrid1KeyPress and StringGrid1DblClick.
          StringGrid1KeyPress then detects vk_return and calls CellSelected procedure.

          Description of bug: When the user uses the arrow keys to move around
          the tstringgrid the first cell moved to (eg the cell the arrow key leads
          to) becomes the  user's move. It was originally intended that the user
          would be able to move to a cell with the arrow key (without making it
          his move) then press enter to confirm his move. This is not how it works!

     #3 : *** Fixed with last version. Beeper component used for PC speaker
          sound.

          Sound working now.

}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, Menus,
  ExtCtrls, Grids, shellapi,

  BTOdeum, // BTBeeper component

  Myinifiles; // my customized IniFile routines

const

  bad_move_beep_freq = 800;
  sound_effect_start_freq = 350;
  sound_effect_step = 40;

  VACANT              = '';
  X_PIECE             = 'X';
  O_PIECE             = 'O';
  EXPLODED            = '*';
  testing_border_code = '.';

type

  TBoardForm = class(TForm)
    MainMenu1: TMainMenu;
    Game1: TMenuItem;
    Help1: TMenuItem;
    Exit1: TMenuItem;
    Howtoplay1: TMenuItem;
    PlayerX1: TMenuItem;
    Computer1: TMenuItem;
    Human1: TMenuItem;
    turn_counter: TEdit;
    Label1: TLabel;
    ContinuousBoard2: TMenuItem;
    NewGame1: TMenuItem;
    Sound1: TMenuItem;
    StringGrid1: TStringGrid;
    Label2: TLabel;
    Os_last_move: TEdit;
    Label3: TLabel;
    Xs_last_move: TEdit;
    ViewHighScores1: TMenuItem;
    Continuous1: TMenuItem;
    NonContinuous1: TMenuItem;
    Website1: TMenuItem;
    About1: TMenuItem;
    Random1: TMenuItem;
    Wheel1: TMenuItem;
    Beeper1: TbtBeeper;
    procedure setup_new_game( {var ArrayOfButns : ArrayOfButns_type; }
                          var whose_move : char );
    procedure highlight_cell( row, column : integer; ch : string );
    procedure unhighlight_cell( row, column : integer; ch : string );
    procedure my_beep( freq : integer );
    procedure randomly_do_player_move( var row, column : integer );
    procedure do_move( ch : char; row, column : integer;
                       var success : boolean );
    procedure kill_cell( row, column, start_sound : integer );
    procedure kill_neighbours_to_cell( row, column, start_sound : integer );
    procedure check_for_chain_reaction(
              player_ch : char;
              row, column : integer;
              var Neighbour_count : integer;
              are_we_checking_a_neighbouring_cell : boolean;
              var has_chain_reaction_previously_occured : boolean );
    procedure count_pieces( var X_player_count, O_player_count : integer );
    procedure FormCreate(Sender: TObject);
    procedure do_end_of_game_stuff( winner : char;
                                    player_count, turn_no, score : integer;
                                    player_x_is_human : boolean );
    procedure check_for_winner;
    procedure do_a_computer_move;
    procedure Howtoplay1Click(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
    procedure Computer1Click(Sender: TObject);
    procedure Human1Click(Sender: TObject);
    procedure NewGame1Click(Sender: TObject);
    procedure ViewHighScores1Click(Sender: TObject);
    procedure NonContinuous1Click(Sender: TObject);
    procedure Continuous1Click(Sender: TObject);
    procedure AboutRandom1Click(Sender: TObject);
    procedure AboutWheel1Click(Sender: TObject);
    procedure StringGrid1MouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure CellSelected;
    procedure StringGrid1KeyPress(Sender: TObject; var Key: Char);
    procedure StringGrid1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ChecktoClose( var OK_to_close: boolean; game_over, player_quits : boolean );
    procedure Website1Click(Sender: TObject);
    procedure SetSoundMenu;
    procedure Sound1Click(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  BoardForm: TBoardForm;
  X_player_count, O_player_count, turn_no,
  current_row, current_col : integer;
  make_sound, play_a_game,
  player_X_is_human, both_players_have_moved,
  player_X_has_not_moved,
  player_O_has_not_moved,
  checked_to_close_yet,
  continuous_board : boolean;
  whose_move : char;
  max_row, max_column : Integer;

{----------------------------------------------------------}

implementation

{$R *.DFM}

uses
  Unit3, hiscores, about;

{----------------------------------------------------------}

procedure tBoardForm.setup_new_game( var whose_move : char );

var
  x, y : integer;

begin
  randomize;
  max_row := StringGrid1.RowCount-1;
  max_column := StringGrid1.ColCount-1;
  Os_last_move.text := 'Yet to move.';
  Xs_last_move.text := 'Yet to move.';
  X_player_count := 0;
  O_player_count := 0;
  whose_move := O_piece;
  both_players_have_moved := false;
  player_X_has_not_moved := true;
  player_O_has_not_moved := true;
  turn_no := 1;
  turn_counter.text := inttostr( turn_no );

  for x := 0 to max_row do
    for y := 0 to max_column do
    begin
      stringgrid1.cells[ y, x ] := vacant;
    end;
end; { setup_new_game }
{-----------------------------------------------------------}

procedure tBoardForm.highlight_cell( row, column : integer; ch : string );

begin
  stringgrid1.cells[ column, row ] := ch;
end; { highlight_cell }
{-----------------------------------------------------------}

procedure tBoardForm.unhighlight_cell( row, column : integer; ch : string );

begin
  stringgrid1.cells[ column, row ] := ch;
end; { unhighlight_cell }
{-----------------------------------------------------------}

procedure tBoardForm.my_beep( freq : integer );
begin
  // test if the user has sound switched on
  if not make_sound then
    exit;
  Beeper1.BeepFor( freq, 25 )
end; { my_beep }
{-----------------------------------------------------------}

procedure tBoardForm.randomly_do_player_move( var row, column : integer );

begin
  repeat
    row := random( max_row+1 );
    column := random( max_column+1 );
  until stringgrid1.cells[ column, row ] = vacant;
end; { randomly_do_player_move }
{-----------------------------------------------------------}

procedure tBoardForm.do_move( ch : char; row, column : integer;
                          var success : boolean );

begin
  success := false;
  if stringgrid1.cells[ column, row ] <> vacant then
  begin
    my_beep( bad_move_beep_freq );
    ShowMessage( 'Error: trying to move into non-vacant cell. [' +
                 inttostr( column ) + ',' + inttostr( row ) + ']' +
                 ' contains "' + stringgrid1.cells[ column, row ] + '"' + #13 +
                 'Please choose another move.');
  end
  else
    case ch of
      x_piece : begin
                  stringgrid1.cells[ column, row ] := X_piece;
                  success := true;
                end;
      o_piece : begin
                  stringgrid1.cells[ column, row ] := O_piece;
                  success := true;
                end;
  end;
  if success then
    unhighlight_cell( row, column, ch );
end; { do_move }
{-----------------------------------------------------------}

procedure tBoardForm.kill_cell( row, column, start_sound : integer );

begin
  stringgrid1.cells[ column, row ] := exploded;
  stringgrid1.refresh;
  unhighlight_cell( row, column, exploded );
  my_beep( start_sound );
end; { kill_cell }
{-----------------------------------------------------------}

function ValidateRowCol(x : Integer; RowColMax : integer) : Integer;
begin
  result := -1;
  if (x >= 0) and (x <= max_row) then
    result := x
  else
    if continuous_board then
    begin
      if (x < 0) then
        result := RowColMax;
      if (x > RowColMax) then
        result := 0;
    end;
end; { ValidateRowCol }
{-----------------------------------------------------------}

procedure tBoardForm.kill_neighbours_to_cell( row, column, start_sound : integer );

var
  ndxrow, ndxcol, r, c : Integer;

begin
  { destroy all adjacent occupied cells }
  for ndxrow := row-1 to row+1 do
    for ndxcol := column-1 to column+1 do
    begin

      { validate r for continous board }
      r := ValidateRowCol( ndxrow, max_row );

     { validate c for continous board }
      c := ValidateRowCol( ndxcol, max_column );

      if (c >= 0) and (r >= 0) then
      begin
        if (StringGrid1.Cells[c, r] = X_PIECE) or
           (StringGrid1.Cells[c, r] = O_PIECE) then
        begin
           kill_cell( r, c, start_sound );
           start_sound := start_sound + sound_effect_step;

           StringGrid1.Refresh; {useful for debugging}
           if not ((c = column) and (r = row)) then
             kill_neighbours_to_cell(r, c, start_sound );
        end;
      end;
    end;
end; { kill_neighbours_to_cell }
{-----------------------------------------------------------}

procedure tBoardForm.check_for_chain_reaction(
  player_ch : char;
  row, column : integer;
  var Neighbour_count : integer;
  are_we_checking_a_neighbouring_cell : boolean;
  var has_chain_reaction_previously_occured : boolean );

var
  ndxrow, ndxcol, r, c : Integer;
  new_Neighbour_Count : Integer;

{...........................................................}

procedure test_neighbour_count( neighbour_count : integer );
var
  ndxrow, ndxcol : integer;
begin
  if (Neighbour_Count > 3) and
     (not has_chain_reaction_previously_occured) then
    begin
      has_chain_reaction_previously_occured := true;
      kill_neighbours_to_cell(row, column, sound_effect_start_freq );
      if player_ch = testing_border_code then
        showmessage( 'Gone critical when you changed the board to continuous.' +
                     #13 + 'A chain reaction occured.' )
      else
        showmessage( 'Gone critical when "' + player_ch + '" was moving!' + #13 +
                     'A chain reaction occured.' );
      for ndxrow := 0 to max_row do
        for ndxcol := 0 to max_column do
          if StringGrid1.Cells[ndxcol, ndxrow] = EXPLODED then
          begin
            StringGrid1.Cells[ndxcol, ndxrow] := VACANT;
            unhighlight_cell(ndxcol, ndxrow, VACANT );
          end;
    end;
end; { test_neighbour_count }
{...........................................................}

begin
  { check to see if cell has gone critical )
  { critical = more than 2 adjacent cells already occupied }

  {    When a move is made, by "O" (who is always the user)
    or "X" (who can be a 2nd user or the computer [default]),
    the algorithm for testing if a chain reaction has occured
    is to count the occupied cells around the "move" (if it
    is greater than 3, including the "move" then a chain
    reaction occurs) *and* in the process of counting (e.g.
    checking) neighbouring cells, if the current cell being
    checked is a neighbour of the "move" and is occupied
    *and* the "are_we_checking_a_neighbouring_cell" flag is
    set to false (which is initially set to false) then
    recursively call the same procedure with this flag
    hard-coded as a "true" parameter. This is the stopping
    condition for the recursion.

           1
          4 2
           3

        Note that in this diagram (above), move 4 does *not*
    cause a chain reaction - because the cell in the middle
    is *not* occupied.

    -------------------------

           4
           123

        4 causes a chain reaction (at 2).

    -------------------------
           2
           13
            4

        4 causes a chain reaction (at 1 and again at 3).

    -------------------------

            6
            54
             7
             123

        7 causes a chain reaction, but we don't want it
    reported twice (actually, this would cause a chain reaction
    4 times... at 7, 5, 4 & 2)

        So, in order to stop a chain reaction from being
    reported more than *once*, make the test for "if a chain
    reaction occurs" have and *and* condition with a boolean
    flag "has_chain_reaction_previously_occured" (initially
    false) being *false*.

        e.g.

        if (neighbour_count > 3) and
           (not has_chain_reaction_previously_occured) then
          // we have a chain reaction ...
          // handle it and set flag.

  }

//  has_chain_reaction_previously_occured := false;
  Neighbour_Count := 0;
  for ndxrow := row-1 to row+1 do
    for ndxcol := column-1 to column+1 do
    begin

      { validate r for continous board }
      r := ValidateRowCol( ndxrow, max_row );

      { validate c for continous board }
      c := ValidateRowCol( ndxcol, max_column );

      if (c >= 0) and (r >= 0) then
      begin
        if (StringGrid1.Cells[c, r] <> VACANT) then
        begin
          Inc(Neighbour_Count);
          if (not(( r = row ) and ( c = column ))) and
             (not has_chain_reaction_previously_occured) and
             (not are_we_checking_a_neighbouring_cell) then
              // we are not testing the move
          begin
            check_for_chain_reaction( player_ch,
                                      r, c, new_neighbour_count, true,
                                      has_chain_reaction_previously_occured );
            test_neighbour_count( new_neighbour_count );
          end;
        end;
      end;
    end;
  test_neighbour_count( neighbour_count );
end; { check_for_chain_reaction }
{-----------------------------------------------------------}

procedure tBoardForm.count_pieces( var X_player_count, O_player_count : integer );

var
  x, y : integer;

begin
  X_player_count := 0;
  O_player_count := 0;
  for x := 0 to max_row do
    for y := 0 to max_column do
      if stringgrid1.cells[ y, x ] = X_piece then
        inc( X_player_count )
      else
        if stringgrid1.cells[ y, x ] = O_piece then
          inc( O_player_count );
end; { count_pieces }
{-----------------------------------------------------------}

procedure TBoardForm.FormCreate(Sender: TObject);
var
  a : TGridRect;
  ConfigIni : TMyIniFile;
  config_filename : string;
begin
  checked_to_close_yet := false;
  a.Left := -1;
  a.Top := -1;
  a.Bottom := -1;
  a.Right := -1;
  StringGrid1.Selection := a;
  current_row := 0;
  current_col := 0;
  randomize;
  want_random := true;
  AboutForm := tAboutForm.create( nil );
  try
    AboutForm.showmodal;
  finally
    AboutForm.free;
  end;

  setup_new_game( whose_move );
  caption := 'It''s your move, "O".';

  make_sound := true; // default is sound on.
  continuous_board := true;
  player_X_is_human := false;

  config_filename := ChangeFileExt( Application.ExeName, '.ini' );
  if FileExists( config_filename ) then
  begin
    ConfigIni := TMyIniFile.Create( config_filename );
    try
      Make_sound := ConfigIni.MyReadBool( 'User Preferences', 'Make Sound', Make_sound );
      continuous_board := ConfigIni.MyReadBool( 'User Preferences', 'Continuous Board', continuous_board );
      player_X_is_human := ConfigIni.MyReadBool( 'User Preferences', 'Player X is Human', player_X_is_human );
    finally
      ConfigIni.Free;
    end;
  end;

  Continuous1.Checked := continuous_board;
  NonContinuous1.Checked := not continuous_board;

  Computer1.Checked := not player_X_is_human;
  Human1.Checked := player_X_is_human;

  SetSoundMenu;
end; { FormCreate }
{----------------------------------------------------------}

procedure tBoardForm.do_end_of_game_stuff( winner : char;
                                       player_count, turn_no, score : integer;
                                       player_x_is_human : boolean );
var
  rank : integer;
  ok : boolean;
begin
  HiScTab := THiScTab.create( nil );
  try
    HiScTab.AddScore( winner, player_count, turn_no, score, player_x_is_human, rank );
    if rank = 0 then
    begin
      if ((not player_x_is_human ) and (winner = 'X')) then
        showmessage( 'The computer''s score was: ' + inttostr(score) + #13 +
                     'But it didn''t make the High Score Table.' )
      else
        showmessage( 'Your score was: ' + inttostr(score) + #13 +
                     'I''m sorry, you didn''t make the High Score Table.' );
    end
    else
      showmessage( 'That score ranked #' + inttostr( rank ));
    HiScTab.ShowModal;
  finally
    Hisctab.release;
  end;

  ChecktoClose( ok, true, false );
  if ok then
    close;
end; { do_end_of_game_stuff }
{----------------------------------------------------------}

procedure tBoardForm.check_for_winner;
var
  score : integer;
begin
  score := 0;
  count_pieces( X_player_count, O_player_count );
  if both_players_have_moved then
  begin
    if (X_player_count > 0) and (O_player_count = 0) then
    begin
      score := 10 * X_player_count * turn_no;
      if player_X_is_human then
        ShowMessage( 'Congratulations Player "X" - you are the winner.' +
                     #13 + 'Your score is ' + inttostr( score ) )
      else
        ShowMessage( 'The Computer wins!!!' );
      do_end_of_game_stuff( x_piece, x_player_count, turn_no, score, player_x_is_human );
    end
    else
      if (O_player_count > 0) and (X_player_count = 0) then
      begin
        score := 10 * O_player_count * turn_no;
        ShowMessage( 'Congratulations Player "O" - you are the winner.' +
                     #13 + 'Your score is ' + inttostr( score ) );
        do_end_of_game_stuff( o_piece, o_player_count, turn_no, score, player_x_is_human );
      end
      else
        if (X_player_count = 0) and (O_player_count = 0) then
        begin
          ShowMessage( 'Both players were wiped out. The game is a draw.' );
          do_end_of_game_stuff( exploded, x_player_count, turn_no, score, player_x_is_human );
        end;
  end;
end; { check_for_winner }
{----------------------------------------------------------}

procedure tBoardForm.do_a_computer_move;
var
  row, column, neighbour_count : integer;
  previous_chain_reaction, success : boolean;
begin
  count_pieces( X_player_count, O_player_count );
  if ((X_player_count <> 0) and (O_player_count <> 0)) or
     (Player_X_has_not_moved) then
  begin
    if not player_X_is_human then
      randomly_do_player_move( row, column );
    if (not player_X_is_human) then
    begin
      do_move( x_piece, row, column, success );
      if success then
      begin
        Xs_last_move.text := 'Row ' + inttostr( row ) +
                             ' Column ' + inttostr( column );
        both_players_have_moved := true;
        previous_chain_reaction := false;
        check_for_chain_reaction( x_piece, row, column,
                                  neighbour_count, false,
                                  previous_chain_reaction );
        caption := 'It''s your move, "O".';
        whose_move := o_piece;
        Player_X_has_not_moved := false;
        inc( turn_no );
        turn_counter.text := inttostr( turn_no );
      end;
    end;
  end;
end; { do_a_computer_move }
{----------------------------------------------------------}

procedure TBoardForm.Howtoplay1Click(Sender: TObject);
begin
  application.helpjump( 'Title' );
end; { howtoplay1click }
{----------------------------------------------------------}

procedure TBoardForm.Exit1Click(Sender: TObject);
begin
  Close;
end; { exit1click }
{----------------------------------------------------------}

procedure TBoardForm.Computer1Click(Sender: TObject);
begin
  player_X_is_human := false;
  mainmenu1.items[ 0 ].items[ 2 ].items[ 0 ].checked := true;
  mainmenu1.items[ 0 ].items[ 2 ].items[ 1 ].checked := false;
  if whose_move = x_piece then
  begin
    do_a_computer_move;
    check_for_winner;
  end;
end; { computer1click }
{----------------------------------------------------------}

procedure TBoardForm.Human1Click(Sender: TObject);
begin
  player_X_is_human := true;
  mainmenu1.items[ 0 ].items[ 2 ].items[ 0 ].checked := false;
  mainmenu1.items[ 0 ].items[ 2 ].items[ 1 ].checked := true;
end; { human1click }
{----------------------------------------------------------}

procedure TBoardForm.NewGame1Click(Sender: TObject);
begin
  if MessageDlg('Quit This Game && Start A New Game ?',
    mtConfirmation, [mbYes, mbNo], 0) = mrYes then
  begin
    setup_new_game( whose_move );
    caption := 'It''s your move, "O".';
  end
end; { newgame1click }
{----------------------------------------------------------}

procedure TBoardForm.ViewHighScores1Click(Sender: TObject);
begin
  HiScTab := THiScTab.create( nil );
  try
    HiScTab.display_table( 0 );
    HiScTab.ShowModal;
  finally
    Hisctab.release;
  end;
end; { ViewHighScores1Click }
{----------------------------------------------------------}

procedure TBoardForm.NonContinuous1Click(Sender: TObject);
begin
  if Player_O_has_not_moved then
  begin
    continuous_board := false;
    mainmenu1.items[ 0 ].items[ 0 ].items[ 0 ].checked := false;
    mainmenu1.items[ 0 ].items[ 0 ].items[ 1 ].checked := true;
  end
  else
    showmessage( 'You are not permitted to change the board mid-game.' + #13 +
                 'To change the board, first start a new game.' );
end; { NonContinuous1Click }
{----------------------------------------------------------}

procedure TBoardForm.Continuous1Click(Sender: TObject);
begin
  if Player_O_has_not_moved then
  begin
    continuous_board := true;
    mainmenu1.items[ 0 ].items[ 0 ].items[ 0 ].checked := true;
    mainmenu1.items[ 0 ].items[ 0 ].items[ 1 ].checked := false;
  end
  else
    showmessage( 'You are not permitted to change the board mid-game.' + #13 +
                 'To change the board, first start a new game.' );
end; { Continuous1Click }
{----------------------------------------------------------}

procedure TBoardForm.AboutRandom1Click(Sender: TObject);
begin
  want_random := true;
  AboutForm := tAboutForm.create( nil );
  try
    AboutForm.showmodal;
  finally
    AboutForm.release;
  end;
end; { AboutRandom1Click }
{----------------------------------------------------------}

procedure TBoardForm.AboutWheel1Click(Sender: TObject);
begin
  want_random := false;
  AboutForm := tAboutForm.create( nil );
  try
    AboutForm.showmodal;
  finally
    AboutForm.release;
  end;
end; { AboutWheel1Click }
{----------------------------------------------------------}

procedure TBoardForm.StringGrid1MouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  CellX,
  CellY : Longint;
begin
  { convert to stringgrid coords }
  StringGrid1.MouseToCell(X, Y, CellX, CellY);

  if (cellx >= 0) and (cellx < stringgrid1.colcount) and
     (celly >= 0) and (celly < stringgrid1.rowcount) then
    if stringgrid1.cells[ cellx, celly ] = vacant then
    begin
      { valid selection }
      StringGrid1.Cursor := crDefault;
    end
    else
    begin
      { invalid selection }
      StringGrid1.Cursor := crNo;
    end
  else
    { outside the stringgrid }
    stringgrid1.cursor := crNo;

end; { StringGrid1MouseMove }
{----------------------------------------------------------}

procedure TBoardForm.CellSelected;
var
  previous_chain_reaction, success : boolean;
  neighbour_count : integer;
begin
  Current_row := StringGrid1.Row;
  Current_col := StringGrid1.Col;
  if (whose_move = x_piece) and (not player_X_is_human) then
  // it's the computer's move not the user's.
    exit;

  success := false;
  if whose_move = x_piece then
  // x is the computer (or 2nd player)
  begin
    do_move( x_piece, current_row, current_col, success );
    if success then
    begin
      Xs_last_move.text := 'Row ' + inttostr( current_row ) +
                           ' Column ' + inttostr( current_col );
      previous_chain_reaction := false;
      check_for_chain_reaction( x_piece,
        current_row, current_col,
        neighbour_count, false, previous_chain_reaction );
      caption := 'It''s your move, "O".';
      whose_move := o_piece;
      Player_X_has_not_moved := false;
      inc( turn_no );
      turn_counter.text := inttostr( turn_no );
    end;
  end
  else
  // o is the user
  begin
    do_move( o_piece, current_row, current_col, success );
    if success then
    begin
      Os_last_move.text := 'Row ' + inttostr( current_row ) +
                           ' Column ' + inttostr( current_col );
      previous_chain_reaction := false;
      check_for_chain_reaction( o_piece,
          stringgrid1.row, stringgrid1.col,
          neighbour_count, false, previous_chain_reaction );
      caption := 'It''s your move, "X".';
      whose_move := x_piece;
      player_O_has_not_moved := false;
    end;
  end;
  if (whose_move = x_piece) and (not player_X_is_human) then
  begin
    do_a_computer_move;
    check_for_winner;
  end;
end; { CellSelected }
{----------------------------------------------------------}

procedure TBoardForm.StringGrid1KeyPress(Sender: TObject; var Key: Char);
begin
  case key of
    chr(vk_return) :
      begin
        CellSelected;
        key := #0;
      end;
    'X','x' :
      begin
        if (whose_move = X_piece) and
           player_X_is_human then
        begin
          CellSelected;
          key := #0;
        end;
      end;
    'O','o' :
      begin
        if (whose_move = O_piece) then
        begin
          CellSelected;
          key := #0;
        end;
      end;
  end; { case }
end; { StringGrid1KeyPress }
{----------------------------------------------------------}

procedure TBoardForm.StringGrid1MouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  CellSelected;
end; { StringGrid1MouseDown }
{----------------------------------------------------------}

procedure TBoardForm.ChecktoClose( var OK_to_close : boolean; game_over, player_quits : boolean );
var
  ConfigIni : TMyIniFile;
begin
  if MessageDlg('Would you like to play another game ?' + #13 +
    '"No" quits Nuclear',
    mtConfirmation, [mbYes, mbNo], 0) = mrYes then
  begin
    refresh;
    play_a_game := true;
    setup_new_game( whose_move );
    current_row := StringGrid1.row + 1;
    current_col := StringGrid1.col + 1;
    caption := 'It''s your move, "O".';
    OK_to_close := false;
    player_quits := false;
  end
  else
    player_quits := true;

  if player_quits then
  begin
    // yes quit & close application.

    // save .ini file
    ConfigIni := TMyIniFile.Create( ChangeFileExt( Application.ExeName, '.ini' ));
    try
      ConfigIni.MyWriteBool( 'User Preferences', 'Make Sound', Make_sound );
      ConfigIni.MyWriteBool( 'User Preferences', 'Continuous Board', continuous_board );
      ConfigIni.MyWriteBool( 'User Preferences', 'Player X is Human', player_X_is_human );
      ConfigIni.UpdateFile;
    finally
      ConfigIni.Free;
    end;

    OK_to_close := true;
    checked_to_close_yet := true;
  end;
end; { ChecktoClose }
{----------------------------------------------------------}

procedure TBoardForm.FormClose(Sender: TObject; var Action: TCloseAction);
var
  ok_to_close : boolean;
begin
  if not checked_to_close_yet then
  begin
    ChecktoClose( Ok_to_close, false, true );
    if ok_to_close then
      Action := caFree
    else
      Action := caNone;
  end
  else
    Action := caFree;
end; { FormClose }
{----------------------------------------------------------}

procedure TBoardForm.Website1Click(Sender: TObject);
begin
  if ShellExecute(Handle, nil, 'http://members.fortunecity.com/pew', nil, nil, SW_SHOWNORMAL) <= 32 then
    showmessage( 'Error: unable to open website:' + #13 +
    'http://members.fortunecity.com/pew'); 
end; { Website1Click }
{----------------------------------------------------------}

procedure TBoardForm.SetSoundMenu;
begin
  Sound1.Checked := make_sound;
  if make_sound then
    Sound1.caption := 'Sound (is on)'
  else
    Sound1.caption := 'Sound (is off)';
end; { SetSoundMenu }
{----------------------------------------------------------}

procedure TBoardForm.Sound1Click(Sender: TObject);
begin
  make_sound := not make_sound;
  SetSoundMenu;
end; { Sound1Click }
{----------------------------------------------------------}

end.

