Unit Board_unit;

Interface

Uses
  Windows, Messages, sysutils, Classes, Graphics,
  Controls, Forms, Dialogs,
  Stdctrls, comctrls, Menus, Grids, Aligrid, Math,
  gomoku_common_unit;

Type
  string1 = string[1];

Const
  Max_x = 15;
  Max_y = 15;
  win_length = 5;
  blank_cell : string1 = '';
  X_piece : string1 = 'X';
  O_piece : string1 = 'O';
  WM_USER_APPLICATION_MINIMIZE = WM_USER+1;

Type
  tplayers = (X_player, O_player, X_computer, O_computer);

  twinner = (no_winner, XXX, OOO, tie, ended_with_error);

  TBoard_form = class(tform)
    MainMenu1: TMainMenu;
    Quit1: TMenuItem;
    board: TStringAlignGrid;
    X_move_label: TLabel;
    O_move_label: TLabel;
    Help1: TMenuItem;
    Game1: TMenuItem;
    NewGame1: TMenuItem;
    Computeris1: TMenuItem;
    O1: TMenuItem;
    X1: TMenuItem;
    AutoplayMode1: TMenuItem;
    N1: TMenuItem;
    SuggestAMove1: TMenuItem;
    HowtoPlay1: TMenuItem;
    Settingupagame1: TMenuItem;
    DontLetComputerMoveMode1: TMenuItem;
    About1: TMenuItem;
    RowandColumnNumbers1: TMenuItem;
    N2HumanOpponentsMode1: TMenuItem;
    ViewWeights1: TMenuItem;
    procedure print_rules(Sender: tobject);
    procedure Settingupagame1Click(Sender: TObject);
    procedure Quick_sort( Lo, Hi : integer );
    procedure check_any_direction_for_piece(
      my_piece : string1;
      X, Y, X_delta, Y_delta : integer );
    procedure Assess_moves_for_piece(
      pieceA : string1;
      X, Y : integer );
    procedure check_any_direction_for_blank(
      X, Y, X_delta, Y_delta : integer );
    procedure Assess_moves_for_blank(
      X, Y : integer );
    procedure weigh_move_options;
    procedure report_winner;
    procedure ChecktoClose(
      var OK_to_close : boolean );
    procedure do_end_of_game_stuff;
    procedure autoplay_game;
    procedure computers_move;
    procedure play_new_game;
    procedure Adjust_StringGrid_size(
      var sg : tStringAlignGrid );
    procedure fix_form_elements;
    Procedure FormCreate(Sender: tobject);
    procedure highlight_move( X, Y : integer );
    procedure do_player_move(
        piece : string1;
        current_row, current_col : integer;
        var success : boolean );
    procedure handle_2_human_opponents;
    procedure Clear_cell;
    procedure update_dont_move_caption;
    procedure CellSelected( which_piece : string1 );
    procedure toggle_dont_move_mode;
    procedure Quit1Click(Sender: TObject);
    procedure PlayGame1Click(Sender: TObject);
    procedure BoardKeyPress(Sender: TObject; var Key: Char);
    procedure BoardMouseDown(Sender: TObject;
      Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure BoardMouseMove(Sender: TObject; Shift: TShiftState;
      X, Y: Integer);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure O1Click(Sender: TObject);
    procedure X1Click(Sender: TObject);
    procedure AutoplayMode1Click(Sender: TObject);
    procedure SuggestAMove1Click(Sender: TObject);
    procedure DontLetComputerMoveMode1Click(Sender: TObject);
    procedure About1Click(Sender: TObject);
    procedure RowandColumnNumbers1Click(Sender: TObject);
    procedure N2HumanOpponentsMode1Click(Sender: TObject);
    procedure ViewWeights1Click(Sender: TObject);
  Private
    { Private declarations }
    level, turn_counter, offset : integer;
    dont_let_computer_move,
    player_quits, play_a_game, autoplay_mode,
    two_human_opponents,
    suggest_a_move : boolean;
    computers_piece, players_piece,
    game_over_string : string;
    whose_move : tplayers;
    winner : twinner;
    gr1 : TGridRect;
    highlight_brush : tbrush;
    procedure OnUserAppMin(var M: TMessage);
      message WM_USER_APPLICATION_MINIMIZE;
  Public
    { Public declarations }
  End;

var
  Board_form: tBoard_form;

Implementation

uses
  play_again_unit, Exit2Humans_unit, weights_unit;

{$R *.DFM}

{----------------------------------------------------------}

procedure tBoard_form.OnUserAppMin(var M: TMessage);
begin
  Application.Minimize;
end; { OnUserAppMin }
{----------------------------------------------------------}

procedure tBoard_form.print_rules(Sender: tobject);
begin
  showmessage(
    'Welcome to the oriental game of GoMoku. ' +
    'This version is played on an 15 by 15 grid. ' +
    'During your play, you may cover one grid ' +
    'intersection with a marker. The object ' +
    'of the game is to get 5 adjacent markers ' +
    'in a row -- horizontally, vertically, or ' +
    'along either diagonal. On the board ' +
    'diagram, your moves are marked with a ' +
    '"X", and the computer moves with a "O". ' );
end; { print_rules }
{----------------------------------------------------------}

procedure TBoard_form.Settingupagame1Click(Sender: TObject);
begin
  showmessage( 'To set up a board position:' + #13 +
    'Press D for "Don''t Move Mode"' + #13 +
    'then press:' + #13 +
    'X to place an X piece,' + #13 +
    'O to place an O piece,' + #13 +
    'or Enter to place your piece' + #13 +
    'or space to clear a cell.' + #13 + #13 +
    'After each piece is placed to computer will tell you the move it would have made.'+ #13 +
    'Press D to exit "Don''t Move Mode" then it will still be your turn.' );
end; { Settingupagame1Click }
{----------------------------------------------------------}

procedure TBoard_Form.Quick_sort( Lo, Hi : integer );

{ QUICKSORT sorts elements in the array A with indices between  }
{ LO and HI (both inclusive). Note that the QUICKSORT proce-    }
{ dure provides only an "interface" to the program. The actual  }
{ processing takes place in the SORT procedure, which executes  }
{ itself recursively.                                           }

{..........................................................}

procedure sort( l, r : integer );

var
  i,j : integer;
  x,y : move_weight_subtype;

begin
  i := l;
  j := r;
  x := move_weights[ (l+r) DIV 2 ];
  repeat
    while move_weights[ i ].weight > x.weight do
      i := i + 1;
    while x.weight > move_weights[ j ].weight do
      j := j - 1;
    if i <= j then
    begin
      y := move_weights[ i ];
      move_weights[ i ] := move_weights[ j ];
      move_weights[ j ] := y;
      i := i + 1;
      j := j - 1;
    end;
  until i > j;
  if l < j then
    sort( l,j );
  if i < r then
    sort( i,r );
end; { sort }
{..........................................................}

begin
  sort( Lo,Hi );
end; { sort_hi_scores }
{----------------------------------------------------------}

procedure TBoard_Form.check_any_direction_for_piece(
  my_piece : string1;
  X, Y, X_delta, Y_delta : integer );
var
  line_count, extra,
  empty_X_before, empty_Y_before,
  empty_X_after, empty_Y_after : integer;
{..........................................................}

  procedure count_before_or_after(
    X_delta, Y_delta : integer;
    var empty_X, empty_Y, line_count : integer );
  var
    Q : integer;
  begin
    for Q := 1 to win_length - 1 do
    begin
      if (empty_X > 0) or (empty_Y > 0) then
        break;
      if (X + X_delta * Q <= max_x) and
         (X + X_delta * Q >= 1) and
         (Y + Y_delta * Q <= max_y) and
         (Y + Y_delta * Q >= 1) then
      begin
        if board.cells[ X + X_delta * Q - 1 + offset,
                        Y + Y_delta * Q - 1 + offset ] = my_piece then
          inc( line_count )
        else
          if board.cells[ X + X_delta * Q - 1 + offset,
                          Y + Y_delta * Q - 1 + offset ] = blank_cell then
          begin
            empty_X := X + X_delta * Q;
            empty_Y := Y + Y_delta * Q;
          end
          else
          begin
            empty_X := 0;
            empty_Y := 0;
            break;
          end;
      end
      else
      begin
        extra := 0;
        break; // hit the border
      end;
    end; // Q loop
  end; { count_before_or_after }
{..........................................................}

begin
  empty_X_before := 0;
  empty_Y_before := 0;
  empty_X_after := 0;
  empty_Y_after := 0;
  extra := 1;
  line_count := 1; // we know this is my_piece

  count_before_or_after( -X_delta, -Y_delta, empty_X_before, empty_Y_before, line_count );
  count_before_or_after( X_delta, Y_delta, empty_X_after, empty_Y_after, line_count );

  if line_count = win_length then
  begin
    if my_piece = X_piece then
      winner := XXX
    else
      winner := OOO;
  end
  else
  begin
    if (empty_X_before +
        empty_Y_before +
        empty_X_after +
        empty_Y_after > 0) then
      extra := extra + 2
    else
      extra := extra + 1;
    if line_count = 1 then
      extra := -1 * extra;

    if (empty_X_before + empty_Y_before > 0) then
    begin
      if board.cells[ empty_X_before-1 + offset,
                      empty_Y_before-1 + offset ] <> blank_cell then
      begin
        game_over_string := 'Error in "check_any_direction_for_piece" procedure, ' + #13 +
          'empty before cell not blank.';
        winner := ended_with_error;
        exit;
      end;
      move_weights[(empty_Y_before-1)*level + empty_X_before].X1 := empty_X_before;
      move_weights[(empty_Y_before-1)*level + empty_X_before].Y1 := empty_Y_before;
      move_weights[(empty_Y_before-1)*level + empty_X_before].weight :=
        integer(Round(
        move_weights[(empty_Y_before-1)*level + empty_X_before].weight +
        power(line_count,6) + extra));
    end;

    if (empty_X_after + empty_Y_after > 0) then
    begin
      if board.cells[ empty_X_after-1 + offset,
                      empty_Y_after-1 + offset ] <> blank_cell then
      begin
        game_over_string := 'Error in "check_any_direction_for_piece" procedure, ' + #13 +
          'empty after cell not blank.';
        winner := ended_with_error;
        exit;
      end;
      move_weights[(empty_Y_after-1)*level + empty_X_after].X1 := empty_X_after;
      move_weights[(empty_Y_after-1)*level + empty_X_after].Y1 := empty_Y_after;
      move_weights[(empty_Y_after-1)*level + empty_X_after].weight :=
        integer(Round(
        move_weights[(empty_Y_after-1)*level + empty_X_after].weight +
        power(line_count,6) + extra));
    end;
  end;
end; { check_any_direction_for_piece }
{----------------------------------------------------------}

procedure TBoard_Form.Assess_moves_for_piece(
  pieceA : string1;
  X, Y : integer );
begin
  check_any_direction_for_piece( pieceA, X, Y, 1, 0 );
  check_any_direction_for_piece( pieceA, X, Y, -1, 0 );
  check_any_direction_for_piece( pieceA, X, Y, 0, 1 );
  check_any_direction_for_piece( pieceA, X, Y, 0, -1 );
  check_any_direction_for_piece( pieceA, X, Y, 1, 1 );
  check_any_direction_for_piece( pieceA, X, Y, -1, -1 );
  check_any_direction_for_piece( pieceA, X, Y, -1, 1 );
  check_any_direction_for_piece( pieceA, X, Y, 1, -1 ); 
end; { Assess_moves_for_piece }
{----------------------------------------------------------}

function Rate_closeness_to_centre(
  Q, maxQ : integer ) : integer;
//
// The param Q is rated for how close it is to the
// centre (or middle) of MaxQ.
// This gives the strategy that on a blank board,
// the centre cell(s) [depending on whether MaxQ is odd or even]
// are the preferred moves.
// E.g. range is assumed to be 1..MaxQ
// for example for 1..8 (MaxQ = 8)
// Rating returned is:
// Q -> Rating
// 1 -> 1 (min = 1)
// 2 -> 2
// 3 -> 3
// 4 -> 4  ) middle 2 numbers
// 5 -> 4  )   "    "    "
// 6 -> 3
// 7 -> 2
// 8 -> 1 (maxQ)
//
// when MaxQ is odd there will be a single middle
// number, so it gets the highest rating.
begin
  if Q <= (MaxQ div 2 + ord(odd(MaxQ))) then
    Result := Q
  else
    Result := abs(Q-MaxQ-1);
end; { Rate_Closeness_to_centre }
{----------------------------------------------------------}

procedure TBoard_Form.check_any_direction_for_blank(
  X, Y, X_delta, Y_delta : integer );
var
  count, extra,
  before_X_count, before_O_count,
  after_X_count, after_O_count : integer;

{..........................................................}
  procedure count_before_or_after(
    X_delta, Y_delta : integer;
    var X_count, O_count : integer );
  var
    Q : integer;
  begin
    X_count := 0;
    O_count := 0;
    // count the X and O pieces before the current cell
    for Q := 1 to win_length - 2 do
    begin
      if (X + X_delta * Q <= max_x) and
         (X + X_delta * Q >= 1) and
         (Y + Y_delta * Q <= max_y) and
         (Y + Y_delta * Q >= 1) then
      begin
        if (board.cells[ X + X_delta - 1 + offset,
                         Y + Y_delta - 1 + offset ] = X_piece) and
           (count = 0) then
          inc(X_count)
        else
          if (board.cells[ X + X_delta - 1 + offset,
                           Y + Y_delta - 1 + offset ] = O_piece) and
             (count = 0) then
            inc(O_count)
          else
            break;
      end
      else
      begin
        extra := 0;
        break; // hit border
      end;
    end;
  end; { count_before_or_after }
{..........................................................}

begin
  extra := 1;
  count_before_or_after( -X_delta, -Y_delta, before_X_count, before_O_count );
  count_before_or_after( X_delta, Y_delta, after_X_count, after_O_count );

  count := 0;

  if (before_X_count + after_X_count > 0) or
     (before_O_count + after_O_count > 0) then
  begin
    // there were X or O pieces before and after
    if ((before_X_count + after_X_count > 0) and
       (computers_piece = O_piece)) or
       ((before_O_count + after_O_count > 0) and
       (computers_piece = X_piece)) then
      // if we are playing the O piece and found X pieces or
      // we were playing X and found O pieces
      // then decrease the weight
      extra := -1 * extra
    else
      // else we are found our type of pieces so increase the weight
      extra := extra + 1;

    // count the before and after pieces
    if (before_X_count > 0) then
      count := before_X_count + after_X_count
    else
      count := before_O_count + after_O_count;
  end;

  extra := extra + Rate_closeness_to_centre( X, max_X );
  extra := extra + Rate_closeness_to_centre( Y, max_Y );

  move_weights[(Y-1)*level + X].X1 := X;
  move_weights[(Y-1)*level + X].Y1 := Y;
  move_weights[(Y-1)*level + X].weight :=
    integer(Round(
    move_weights[(Y-1)*level + X].weight +
    count * count + extra));
end; { check_any_direction_for_blank }
{----------------------------------------------------------}

procedure TBoard_Form.Assess_moves_for_blank(
  X, Y : integer );
begin
  check_any_direction_for_blank( X, Y, 1, 0 );
  check_any_direction_for_blank( X, Y, -1, 0 );
  check_any_direction_for_blank( X, Y, 0, 1 );
  check_any_direction_for_blank( X, Y, 0, -1 );
  check_any_direction_for_blank( X, Y, 1, 1 );
  check_any_direction_for_blank( X, Y, -1, -1 );
  check_any_direction_for_blank( X, Y, -1, 1 );
  check_any_direction_for_blank( X, Y, 1, -1 );
end; { Assess_moves_for_blank }
{----------------------------------------------------------}

procedure TBoard_Form.weigh_move_options;
var
  x, y : integer;
begin
  for x := 1 to max_x do
    for y := 1 to max_y do
      if board.cells[ x-1+offset, y-1+offset ] = blank_cell then
      begin
        move_weights[(y-1)*level + x].X1 := X; // empty cell
        move_weights[(y-1)*level + x].Y1 := Y; // empty cell
        move_weights[(y-1)*level + x].weight := 0; // empty cell
      end
      else
      begin
        move_weights[(y-1)*level + x ].X1 := X;
        move_weights[(y-1)*level + x ].Y1 := Y;
        move_weights[(y-1)*level + x ].weight := -1; // illegal move
      end;

  for x := 1 to max_x do
    for y := 1 to max_y do
    begin
      if board.cells[ x-1+offset, y-1+offset ] = O_piece then
        Assess_moves_for_piece( O_piece, x,y )
      else
        if board.cells[ x-1+offset, y-1+offset ] = X_piece then
          Assess_moves_for_piece( X_piece, x,y )
        else
          if board.cells[ x-1+offset, y-1+offset ] = blank_cell then
            Assess_moves_for_blank( X, Y );
      if winner <> no_winner then
        exit;
    end;
end; { weigh_move_options }
{----------------------------------------------------------}

procedure TBoard_Form.report_winner;
begin
  caption := 'Game Over.';
  case winner of
    XXX :
      if computers_piece = X_piece then
        Game_Over_string := Game_Over_string + #13 +
          'Computer as X wins.' + #13 +
          'Game Over.'
      else
        Game_Over_string := Game_Over_string + #13 +
          'Player X wins.' + #13 +
          'Game Over.';
    OOO :
      if computers_piece = O_piece then
        Game_Over_string := Game_Over_string + #13 +
          'Computer as O wins.' + #13 +
          'Game Over.'
      else
        Game_Over_string := Game_Over_string + #13 +
          'Player O wins.' + #13 +
          'Game Over.';
    no_winner :
      Game_Over_string := Game_Over_string + #13 +
        'There is no winner' + #13 +
        'Game Over.';
    tie :
      Game_Over_string := Game_Over_string + #13 +
        'The Game is Tied!' + #13 +
        'Game Over.';
    ended_with_error :
      Game_Over_string := Game_Over_string + #13 +
        'Ooopps!!! The Game Ended in an Error.' + #13 +
        'Game Over.';
  end; // case
  showmessage( Game_Over_string );
end; { report_winner }
{----------------------------------------------------------}

procedure TBoard_Form.ChecktoClose(
  var OK_to_close : boolean );
// sets 'player_quits' to true to allow form to close,
// if the player says 'no' to next game.
begin
  OK_to_close := false;
  play_again_form := tplay_again_form.create( nil );
  try
    play_again_form.label1.caption := 'Would you like to play another game ?' + #13 +
      ' "Yes" starts a new game' + #13 +
      ' "No" quits GoMoko';
    play_again_form.autoplay.checked := autoplay_mode;
    if play_again_form.showmodal = mrYes then
    begin
      if play_again_form.autoplay.checked then
      begin
        autoplay_mode := true;
        whose_move := X_computer;
      end
      else
        autoplay_mode := false;
      if autoplay_mode then
      begin
        computers_piece := O_piece;
        players_piece := X_piece;
      end;
      play_a_game := true;
      player_quits := false;
    end
    else
    begin
      OK_to_close := true;
      player_quits := true;
    end;
  finally
    FreeAndNil(play_again_form);
  end;
end; { ChecktoClose }
{----------------------------------------------------------}

procedure TBoard_Form.do_end_of_game_stuff;
var
  ok : boolean;
begin
  ok := false;
  ChecktoClose( ok );
  if ok then
    close
  else
    play_new_game;
end; { do_end_of_game_stuff }
{----------------------------------------------------------}

procedure TBoard_Form.computers_move;
var
  X, Y, count, Random_move,
  highest_move_rating : integer;
  found : boolean;
begin
  caption := 'Turn '+ inttostr(turn_counter) +
    ' ... it''s my move as ' + computers_piece + '.';
  weigh_move_options;

  if winner <> no_winner then
  begin
    report_winner;
    do_end_of_game_stuff;
    exit;
  end;

  quick_sort(1, max_x * max_y );
  highest_move_rating := move_weights[ 1 ].weight;

  if highest_move_rating = -1 then
  begin
    game_over_string := 'There are no more moves.';
    winner := tie;
  end
  else
  begin
    count := 0;
    found := false;
    x := 1;
    while not found and (x <= max_x*max_y) do
    begin
      if move_weights[ x ].weight = highest_move_rating then
        inc(count)
      else
        found := true;
      inc(x);
    end;

    if count > 0 then
    begin
      Random_move := random( count ) + 1;
      X := move_weights[ Random_move ].X1;
      Y := move_weights[ Random_move ].Y1;
      if dont_let_computer_move then
      begin
        showmessage( format('computers move, as player ' +
                     computers_piece +
                     ' would be Row,Col (%d,%d)' + #13 +
                     'with a highest_move_rating = %d' + #13 +
                     'Number of moves to choose from = %d',
                     [Y,X,highest_move_rating,count]));
        exit;
      end;

      if suggest_a_move then
      begin
        showmessage( format('The move I suggest, as player ' +
                     computers_piece +
                     ' would be Row,Col (%d,%d)' + #13 +
                     'with a highest_move_rating = %d' + #13 +
                     'Number of moves to choose from = %d',
                     [Y,X,highest_move_rating,count]));
        exit;
      end;

      if (X >= 1) and (X <= max_x) and
         (Y >= 1) and (Y <= max_y) then
      begin
        if board.cells[ X-1+offset,Y-1+offset ] = blank_cell then
        begin
          board.cells[ X-1+offset, Y-1+offset ] := computers_piece;
          if computers_piece = O_piece then
            O_move_label.caption :=
              format('Player O (computer) moves: Row,Col (%d,%d)',
              [Y,X])
          else
            X_move_label.caption :=
              format('Player X (computer) moves: Row,Col (%d,%d)',
              [Y,X]);
          highlight_move( X-1, Y-1 );
        end
        else
        begin
          game_over_string := 'Error: computer attempted to move in non-vacant cell.';
          winner := ended_with_error;
          exit;
        end;
      end
      else
      begin
        game_over_string := format('Randomly picked move co-ords are wrong.' + #13 +
          'col, row [%d,%d]',[X,Y]);
        winner := ended_with_error;
      end;
    end
    else
    begin
      game_over_string := 'Error - there are zero moves of the indicated rating.';
      winner := ended_with_error;
    end;
  end;

  case whose_move of
    X_computer :
      if autoplay_mode then
        whose_move := O_computer
      else
        whose_move := O_player;

    O_computer :
      if autoplay_mode then
        whose_move := X_computer
      else
        whose_move := X_player;
  end;

  if winner = no_winner then
    weigh_move_options;

  if winner = no_winner then
  begin
    inc(turn_counter);
    case whose_move of
      X_player :
        caption := 'Turn ' + inttostr(turn_counter) +
          ' ... make a move player X.';
      O_player :
        caption := 'Turn ' + inttostr(turn_counter) +
          ' ... make a move player O.';
    end; // case
  end
  else
    begin
      report_winner;
      do_end_of_game_stuff;
      exit;
    end;
end; { computers_move }
{----------------------------------------------------------}

procedure TBoard_form.autoplay_game;
begin
  case whose_move of
    X_player : whose_move := X_computer;
    O_player : whose_move := O_computer;
  end;

  while (winner = no_winner) and
        (whose_move in [O_computer,X_computer]) do
  begin
    case whose_move of
      X_computer : computers_piece := X_piece;
      O_computer : computers_piece := O_piece;
    end;
    computers_move;
  end;
end; { autoplay_game }
{---------------------------------------------------------}

procedure TBoard_form.play_new_game;
var
  i,j : integer;
begin
  for i:= offset + 1 to max_x + offset do
    for j := offset + 1 to max_y + offset do
      Board.cells[ i-1,j-1 ] := blank_cell;

  player_quits := false;
  play_a_game := true;

  game_over_string := '';
  winner := no_winner;
  X_move_label.caption := 'Player X is yet to move.';
  O_move_label.caption := 'Player O is yet to move.';

  O1.Checked := (computers_piece = O_piece) or autoplay_mode;
  X1.Checked := (computers_piece = X_piece) or autoplay_mode;
  autoplaymode1.Checked := autoplay_mode;
  DontLetComputerMoveMode1.Checked := dont_let_computer_move;

  // X always moves first (whether its the player or computer)
  turn_counter := 1;
  suggest_a_move := false;

  if autoplay_mode then
    whose_move := X_computer
  else
    whose_move := X_player;

  if dont_let_computer_move then
  begin
    caption := 'Don''t let computer move mode ON.';
    whose_move := X_player;
  end
  else
    if autoplay_mode then
      autoplay_game
    else
    begin
      if computers_piece = O_piece then
      begin
        whose_move := X_player;
        caption := 'Turn 1 ... make a move player X.';
      end
      else
      begin
        whose_move := X_computer;
        computers_move;
      end;
    end;
end; { play_new_game }
{----------------------------------------------------------}

procedure TBoard_form.About1Click(Sender: TObject);
begin
  showmessage( 'GoMoko for Windows' + #13 +
    'v0.03  2002 Peter E. Williams' + #13 +
    '(a very simple game!)' + #13 +
    'written using Delphi 5.' + #13 +
    '(The best language for Windows programming!!!)' + #13 +
    'Freeware !!!' + #13 +
    'Source code available.' + #13 +
    'email: pew@pcug.org.au' + #13 +
    '(Send me an email, I''d love to hear from you :-)))' );
end; { About1Click }
{----------------------------------------------------------}

procedure tBoard_form.Adjust_StringGrid_size(
  var sg : tStringAlignGrid );
const
  YeOldeCellKudgeFactor = 3;
begin
  with SG do
    begin
      Width := (ColCount)*DefaultColWidth +
               GridLineWidth * (ColCount)+
               YeOldeCellKudgeFactor;

      Height := (RowCount)*DefaultRowHeight +
               GridLineWidth * (RowCount)+
               YeOldeCellKudgeFactor;
    end;
end; { Adjust_StringGrid_size }
{---------------------------------------------------------}

procedure tBoard_form.fix_form_elements;
var
  Q, Q2 : integer;
begin
  RowandColumnNumbers1.Checked := want_row_col_numbers;
  with board do
  begin
    if want_row_col_numbers then
    begin
      FixedRows := 1;
      FixedCols := 1;
      RowCount := Max_y + 1;
      ColCount := Max_x + 1;
      // move the board
      for Q := Max_X downto 1 do
        for Q2 := Max_Y downto 1 do
          board.cells[ Q, Q2 ] := board.cells[ Q-1, Q2-1 ];
      board.cells[ 0,0 ] := blank_cell;
      for Q := 1 to Max_x do
        cells[ Q, 0 ] := inttostr(Q);
      for Q := 1 to Max_y do
        cells[ 0, Q ] := inttostr(Q);
    end
    else
    begin
      if (RowCount > Max_Y) and (ColCount > Max_X) then
      begin
        // move the board
        for Q := 1 to Max_X + 1 do
          for Q2 := 1 to Max_Y + 1 do
          begin
            if (Q = Max_X + 1) and (Q2 = Max_Y + 1) then
              board.cells[ Q-1, Q2-1 ] := blank_cell
            else
              board.cells[ Q-1, Q2-1 ] := board.cells[ Q, Q2 ];
          end;
      end;
      FixedRows := 0;
      FixedCols := 0;
      RowCount := Max_y;
      ColCount := Max_x;
    end;
  end;
  Adjust_StringGrid_size( board );
  width := board.left + board.width + 30;
  height := board.top + board.height + 104;
  X_move_label.top := board.top + board.height + 5;
  O_move_label.top := board.top + board.height + 21;

  if want_row_col_numbers then
    offset := 1
  else
    offset := 0;
end; { fix_form_elements }
{---------------------------------------------------------}

procedure tBoard_form.FormCreate(Sender: tobject);
begin
  About1Click( nil );

  want_row_col_numbers := true;  // default
  fix_form_elements;

  Randomize;
  level := max_x;

  // now set the cell highlight colour
  highlight_brush := tbrush.create;
  highlight_brush.color := clYellow;

  gr1.Left := -1;
  gr1.Top := -1;
  gr1.Bottom := -1;
  gr1.Right := -1;
  board.Selection := gr1;

  dont_let_computer_move := false;
  two_human_opponents := false;

  show;
  showmessage( 'We alternate moves. You go first...' );

  computers_piece := O_piece;
  players_piece := X_piece;

  autoplay_mode := false;

  play_new_game;
end; { formcreate }
{----------------------------------------------------------}

procedure TBoard_form.Quit1Click(Sender: TObject);
var
  OK : boolean;
begin
  ok := true; // default is close program
  if (turn_counter > 1) or dont_let_computer_move then
    ChecktoClose( ok );
  if OK then
  begin
    player_quits := true;
    play_a_game := false;
    close;
  end
  else
    play_new_game;
end; { Quit1Click }
{----------------------------------------------------------}

procedure TBoard_form.PlayGame1Click(Sender: TObject);
begin
  play_new_game;
end; { PlayGame1Click }
{----------------------------------------------------------}

procedure TBoard_Form.highlight_move( X, Y : integer );
begin
  board.CellBrush[ X+offset,Y+offset ] := highlight_brush;
  refresh;
  if not autoplay_mode then
    sleep( 500 );
  board.ResetBrushCell( X+offset,Y+offset );
  refresh;
end; { highlight_move }
{----------------------------------------------------------}

procedure TBoard_form.do_player_move(
  piece : string1;
  current_row, current_col : integer;
  var success : boolean );
begin
  success := false;
  if (current_row - offset >= 0) and (current_row - offset < max_y) and
     (current_col - offset >= 0) and (current_col - offset < max_x) then
  begin
    if (board.cells[ current_col, current_row ] = blank_cell) or
       dont_let_computer_move then
    // if "dont_let_computer_move" is true then we can move anywhere.
    begin
      board.cells[ current_col, current_row ] := piece;
      if whose_move = X_player then
        X_move_label.caption :=
          format('Player X moves: Row,Col (%d,%d)',
          [current_row+1-offset,current_col+1-offset])
      else
        O_move_label.caption :=
          format('Player O moves: Row,Col (%d,%d)',
          [current_row+1-offset,current_col+1-offset]);
      highlight_move( current_col-offset, current_row-offset );
      success := true;
    end
    else
      showmessage( 'That cell is occupied.' + #13 +
                   'Please try again.');
  end;
end; { do_player_move }
{----------------------------------------------------------}

procedure TBoard_Form.handle_2_human_opponents;
var
  highest_move_rating : integer;
begin
  weigh_move_options;
  quick_sort(1, max_x * max_y );
  highest_move_rating := move_weights[ 1 ].weight;
  if highest_move_rating = -1 then
  begin
    game_over_string := 'There are no more moves.';
    winner := tie;
    report_winner;
    do_end_of_game_stuff;
    exit;
  end
  else
  begin
    inc(turn_counter);
    case whose_move of
      X_player :
        begin
          whose_move := O_player;
          caption := 'Turn ' + inttostr(turn_counter) +
            ' ... make a move player O.';
          players_piece := O_piece;
        end;
      O_player :
        begin
          whose_move := X_player;
          caption := 'Turn ' + inttostr(turn_counter) +
            ' ... make a move player X.';
          players_piece := X_piece;
        end;
    end; // case
  end;
end; { handle_2_human_opponents }
{----------------------------------------------------------}

procedure TBoard_Form.update_dont_move_caption;
var
  str1 : string;
begin
  if dont_let_computer_move then
  begin
    str1 := 'Don''t move mode ON.';
    case whose_move of
      X_player : str1 := str1 + 'X (player) to move';
      O_player : str1 := str1 + 'O (player) to move';
      X_computer : str1 := str1 + 'X (computer) to move';
      O_computer : str1 := str1 + 'O (computer) to move';
    end;
    caption := str1;
  end;
end; { update_dont_move_caption }
{----------------------------------------------------------}

procedure TBoard_Form.CellSelected( which_piece : string1 );
var
  Current_row, Current_col : integer;
  success : boolean;
  old_whose_move : tplayers;
begin
  Current_row := Board.Row;
  Current_col := Board.Col;

  if want_row_col_numbers then
    if (current_row = 0) or (current_col = 0) then
    begin
      showmessage('You cannot move in the row/col numbers');
      exit;
    end;

  if not dont_let_computer_move then
  begin
    if (which_piece <> players_piece) or
       (whose_move in [X_computer, O_computer]) then
       // it's the computer's move not the user's.
    begin
      showmessage( 'It''s the computers turn.');
      exit;
    end;
  end;

  do_player_move( which_piece, current_row, current_col, success );
  if success then
    if two_human_opponents then
      handle_2_human_opponents
    else
    begin
      old_whose_move := whose_move;
      case whose_move of
        X_player : whose_move := O_computer;
        O_player : whose_move := X_computer;
      end;

      computers_move;
      if dont_let_computer_move then
      begin
        whose_move := old_whose_move;
        update_dont_move_caption;
      end;
    end; // else
end; { CellSelected }
{----------------------------------------------------------}

procedure TBoard_Form.toggle_dont_move_mode;
begin
  dont_let_computer_move := not dont_let_computer_move;
  DontLetComputerMoveMode1.Checked := dont_let_computer_move;

  if dont_let_computer_move then
  begin
    update_dont_move_caption;
    showmessage( 'To see the computers move rating, place an X piece' + #13 +
                 'with "don''t let computer move mode ON".')
  end
  else
  begin
    case whose_move of
      X_player :
        caption := 'Turn ' + inttostr(turn_counter) +
                   ' ... make a move player X.';
      O_player :
        caption := 'Turn ' + inttostr(turn_counter) +
                   ' ... make a move player O.';
      O_computer, X_computer :
        computers_move;
    end; // case
  end;
end; { toggle_dont_move_mode }
{----------------------------------------------------------}

procedure TBoard_Form.clear_cell;
var
  Current_row, Current_col : integer;
begin
  if dont_let_computer_move then
  begin
    caption := 'Don''t let computer move mode ON.';
    Current_row := Board.Row;
    Current_col := Board.Col;
    board.cells[ current_col, current_row ] := blank_cell;
  end;
end;
{----------------------------------------------------------}

procedure TBoard_Form.BoardKeyPress(Sender: TObject; var Key: Char);
begin
  case key of
    '2' : // toggle 2 human opponents mode (Game / 2 Human Opponents Mode)
      N2HumanOpponentsMode1Click( nil );
    'A','a' : // autoplay mode on (Game / Autoplay Mode)
      AutoplayMode1Click( nil );

    'B','b' : // Help / About
      About1Click( nil );

    'D','d' : // toggle don't move mode (Game / Don't Let Computer Move Mode)
      toggle_dont_move_mode;

    'E','e' : // help / Setting up a game (board position)
      Settingupagame1Click( nil );

    'H','h' : // Help / How to play
      print_rules( nil );

    'N','n' : // Game / New Game
      play_new_game;

    'O','o' : // Play an 'O' piece
      CellSelected( O_piece );

    'Q','q' : // Game / Quit
      close;

    'R','r' : // Game / Row and Column Numbers
      RowandColumnNumbers1Click( nil );

    'S','s' : // Game / Suggest A Move
      SuggestAMove1Click( nil );

    'W','w' : // view weights
      ViewWeights1Click( nil );

    'X','x' : // Play an 'X' piece
      CellSelected( X_piece );

    chr(vk_escape) : // minimize the game
      PostMessage(Handle, WM_USER_APPLICATION_MINIMIZE, 0, 0);

    chr(vk_return) : // Play the current player's piece
      CellSelected( players_piece );

    ' ' : // space - clear the cell (only valid in Don't Move Mode)
      clear_cell;
  end; { case }
  key := #0;
end; { BoardKeyPress }
{----------------------------------------------------------}

procedure TBoard_Form.BoardMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if whose_move = X_player then
    CellSelected( X_piece )
  else
    if whose_move = O_player then
      CellSelected( O_piece )
    else
      showmessage( 'It''s the computers turn.');
end; { BoardMouseDown }
{----------------------------------------------------------}

procedure TBoard_Form.BoardMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  CellX,
  CellY : Longint;
begin
  { convert to stringgrid coords }
  Board.MouseToCell(X, Y, CellX, CellY);

  if (cellx >= 0) and (cellx < Board.colcount) and
     (celly >= 0) and (celly < Board.rowcount) then
    if Board.cells[ cellx, celly ] = blank_cell then
    begin
      { valid selection }
      Board.Cursor := crDefault;
    end
    else
    begin
      { invalid selection }
      Board.Cursor := crNo;
    end
  else
    { outside the stringgrid }
    Board.cursor := crNo;
end; { BoardMouseMove }
{----------------------------------------------------------}

procedure TBoard_Form.FormClose(Sender: TObject; var Action: TCloseAction);
var
  ok_to_close : boolean;
begin
  if not player_quits then
  begin
    ok_to_close := false;
    ChecktoClose( Ok_to_close );
    if ok_to_close then
    begin
      FreeAndNil( highlight_brush );
      Action := caFree;
    end
    else
    begin
      Action := caNone;
      play_new_game;
    end;
  end
  else
  begin
    FreeAndNil( highlight_brush );
    Action := caFree;
  end;
end; { FormClose }
{----------------------------------------------------------}

procedure TBoard_form.O1Click(Sender: TObject);
begin
  if whose_move in [O_computer,X_computer] then
    exit;
  // computer is O
  computers_piece := O_piece;
  players_piece := X_piece;
  O1.Checked := true;
  X1.Checked := false;
  case whose_move of
    O_player : whose_move := O_computer;
    X_player : whose_move := X_player;
  end;
  if not dont_let_computer_move then
  begin
    if whose_move = O_computer then
      computers_move;
    update_dont_move_caption;
  end;
end; { O1Click }
{----------------------------------------------------------}

procedure TBoard_form.X1Click(Sender: TObject);
begin
  if whose_move in [O_computer,X_computer] then
    exit;
  // computer is X
  computers_piece := X_piece;
  players_piece := O_piece;
  O1.Checked := false;
  X1.Checked := true;
  case whose_move of
    O_player : whose_move := O_player;
    X_player : whose_move := X_computer;
  end;
  if not dont_let_computer_move then
  begin
    if whose_move = X_computer then
      computers_move;
    update_dont_move_caption;
  end;
end; { X1Click }
{----------------------------------------------------------}

procedure TBoard_form.AutoplayMode1Click(Sender: TObject);
begin
  autoplay_mode := not autoplay_mode;
  autoplaymode1.Checked := autoplay_mode;
  if autoplay_mode then
  begin
    O1.Checked := true;
    X1.Checked := true;
    dont_let_computer_move := false;
    autoplay_game;
  end
  else
  begin
    O1.Checked := (computers_piece = O_piece);
    X1.Checked := (computers_piece = X_piece);
    case whose_move of
      O_computer :
        begin
          whose_move := O_player;
          caption := 'Turn ' + inttostr(turn_counter) +
            ' ... make a move player O.';
        end;
      X_computer :
        begin
          whose_move := X_player;
          caption := 'Turn ' + inttostr(turn_counter) +
            ' ... make a move player X.';
        end;
    end; // case
  end;
end; { AutoplayMode1Click }
{----------------------------------------------------------}

procedure TBoard_form.SuggestAMove1Click(Sender: TObject);
var
  old_computers_piece, old_caption : string;
begin
  old_computers_piece := computers_piece;
  case whose_move of
    O_player : computers_piece := O_piece;
    X_player : computers_piece := X_piece;
  end;
  suggest_a_move := true;
  old_caption := caption;
  computers_move;
  computers_piece := old_computers_piece;
  caption := old_caption;
  suggest_a_move := false;
end; { SuggestAMove1Click }
{----------------------------------------------------------}

procedure TBoard_form.DontLetComputerMoveMode1Click(Sender: TObject);
begin
  toggle_dont_move_mode;
end; { DontLetComputerMoveMode1Click }
{----------------------------------------------------------}

procedure TBoard_form.RowandColumnNumbers1Click(Sender: TObject);
begin
  want_row_col_numbers := not want_row_col_numbers;
  fix_form_elements;
end; { RowandColumnNumbers1Click }
{----------------------------------------------------------}

procedure TBoard_form.N2HumanOpponentsMode1Click(Sender: TObject);
var
  exit1 : boolean;
begin
  exit1 := false;
  two_human_opponents := not two_human_opponents;
  N2HumanOpponentsMode1.Checked := two_human_opponents;
  if two_human_opponents then
  begin
    O1.Checked := true;
    X1.Checked := false;
    showmessage( 'Now in 2 Human opponents mode.');
  end
  else
  begin
    Exit2Humans_form := tExit2Humans_form.create( nil );
    try
      case Exit2Humans_form.showmodal of
        mrYes : // Computer Plays O
          O1Click( nil );
        mrNo : // Computer Plays X
          X1Click( nil );
        mrCancel, mrOK :
          begin
            N2HumanOpponentsMode1.Checked := true;
            two_human_opponents := true;
            showmessage( 'Staying in 2 Human opponents mode.');
            exit1 := true;
          end
      end; // case
    finally
      FreeAndNil(Exit2Humans_form);
    end; // try
    if exit1 then
      exit;
  end;
end; { N2HumanOpponentsMode1Click }
{----------------------------------------------------------}

procedure TBoard_form.ViewWeights1Click(Sender: TObject);
var
  x : integer;
  highest_move_rating : integer;
  found, exit1 : boolean;
var
  old_computers_piece : string;
begin
  old_computers_piece := computers_piece;

  exit1 := false;
  // we'll use the Exit2Humans_form to as which side's weights
  // we want to look at
  Exit2Humans_form := tExit2Humans_form.create( nil );
  try
    Exit2Humans_form.caption := 'Select the side that you want to see the weights of';
    Exit2Humans_form.computerO.caption := 'Check weights for &O';
    Exit2Humans_form.computerX.caption := 'Check weights for &X';
    case Exit2Humans_form.showmodal of
      mrYes : // Computer Plays O
        computers_piece := O_piece;
      mrNo : // Computer Plays X
        computers_piece := X_piece;
      mrCancel, mrOK :
        begin
          showmessage( 'You didn''t select either so cancelled showing weights.');
          exit1 := true;
        end
    end; // case
  finally
    FreeAndNil(Exit2Humans_form);
  end; // try

  if exit1 then
    exit;

  weigh_move_options;
  // move weight must have been calculated before the weight_form is created
  weights_form := tweights_form.create( nil );
  try
    weights_form.init;
    weights_form.caption := 'Board weights as player ' + computers_piece;
    if winner <> no_winner then
    begin
      report_winner;
      weights_form.caption := game_over_string;
    end
    else
    begin
      quick_sort(1, max_x * max_y );
      highest_move_rating := move_weights[ 1 ].weight;
      if highest_move_rating = -1 then
      begin
        game_over_string := 'There are no more moves.';
        winner := tie;
      end
      else
      begin
        //count := 0;
        found := false;
        x := 1;
        while not found and (x <= max_x*max_y) do
        begin
          if move_weights[ x ].weight = highest_move_rating then
            weights_form.highlight_highest_weights(
              move_weights[ x ].X1,
              move_weights[ x ].Y1)
          else
            found := true;
          inc(x);
        end; // while
      end; // else
    end; // else
    weights_form.showmodal;
  finally
    FreeAndNil(weights_form);
  end;
  computers_piece := old_computers_piece;
end; { ViewWeights1Click }
{----------------------------------------------------------}

end.
