#ifndef _MYSTRING_H
#define _MYSTRING_H

#include <iostream.h>
#include <string.h>            // for strcmp, strlen, etc.
#include <stdlib.h>            // for exit,atof,atoi
#include <assert.h>            // for assert
#include <ctype.h>             // for tolower/toupper
//#include <stddef>
//#include "bool.h"

// modeled (loosely) on Stroustrup 2nd edition, on Budd's String class,
// and Adams, Leestma, and Nyhoff Strings (and probably a host of others)
//
// written by Owen Astrachan April 1993 (with reference counts)
//      modified May 2, 1994
//      modified July 6, 1994 [fixed some inconsistencies found in DOS
//                             implementation]
//      modified October, 1994 [to re-use existing memory rather than
//                              than deleting when =, copy constructor used]
//
// provides a "standard" string class supporting operations given below
//
// constructors:
//
//        cpstring(const char * s = 0);
//               copies a C-style '\0'-terminated string
//        cpstring(const cpstring & x);
//               copy constructor
//        cpstring (char ch);
//               make a cpstring from a single character
//
//  destructor
//        ~cpstring()  destructor frees memory associated with cpstring
//
//  operators
//
//        cpstring & operator= (const cpstring & x);
//               assigns one cpstring to another
//        cpstring & operator= (const char * s);
//               assigns a C-style cpstring to a cpstring
//
//        operator const char * s () const
//               cast/convert cpstring to C-style string
//
//        int length() const
//               returns # of characters in cpstring
//
//         char& operator[](int i)
//               returns i-th char of cpstring (signals error if out-of-bounds)
//                       (char is assignable since reference)
//
//         const char& operator[](int i) const
//               returns i-th char of cpstring (signals error if out-of-bounds)
//
//  I/O operators
//       << and >>
//               are "standard" for ostream and istream respectively
//       istream & getline(istream & is, cpstring & s, char sentinel = '\n')
//               mimics iostream.h getline for istreams and cpstrings
//
//  relational operators
//          ==, !=, <, >, >=, <=
//              for all combinations of C-style strings and cpstring
//
//  concatenation operators and functions
//          cpstring & operator +=(const cpstring & x)
//              catenates x to the right of the cpstring
//          friend cpstring operator +(const cpstring & x, const cpstring & y)
//              catenates x to y and returns result
//
//
//  search/facilitator functions
//
//          cpstring substr(int pos, int len)
//             returns cpstring of len chars starting at pos
//             truncates if len too big, returns empty cpstring
//             if pos too big
//
//          cpstring Downcase() const
//          cpstring Upcase() const
//               return lower/upper case versions of cpstring, respectively
//
//          bool Contains(const cpstring & s) const
//          bool Contains(char * s) const
//               returns true if cpstring contains s, else returns false
//
// conversion functions
//
//          char * chars()
//             similar to (char *) cast, but member function
//          char * c_str()
//             identical to chars(), but ANSI compatible
//
//          int atoi(cpstring &)
//             like <stdlib.h> atoi, but has cpstring parameter
//             ensures that calling <stdlib.h> version is done
//             correctly
//
//          int atof(cpstring &)
//             like <stdlib.h> atof, but has cpstring parameter
//             ensures that calling <stdlib.h> version is done
//             correctly
//
//          cpstring itoa(int n)
//             returns cpstring equivalent of number n
//             i.e., itoa(123) returns "123", itoa(-56) returns "-56"


extern const int NPOS;

class cpstring{
    
  public:
	 cpstring(const char * s = 0);           // for literals and C-style strings
	 cpstring(const cpstring & x);             // copy constructor
    cpstring(char ch);                      // make cpstring from char

    cpstring& operator = (const char * s);  // assign literal, C-style strings
    cpstring& operator =(const cpstring & x); // assign one cpstring to another

    ~cpstring();                            // free up memory

    // perhaps this cast should not be included

	 // this causes ambiguity problems with Borland (and others)
    // removing this requires use of c_str()
	 //operator const char *() const {return myCstring;}

    const char * chars() const {return myCstring;}    // explicit conversion to C-style
    const char * c_str() const {return myCstring;}

    int length() const {return strlen(myCstring);}  // returns # of chars
    char& operator[](int i);              // index character (range checked)
    const char& operator[](int i) const;  // constant index  (range checked)

    cpstring& operator +=(const cpstring & x);

    cpstring Downcase() const;

    cpstring Upcase() const;

    cpstring substr(int pos, int len) const;
    // precondition: 0 <= pos < s.length() && pos + len <= s.length()
    // postcondition: returns substring starting at char pos of len chars
    
	 bool Contains (const cpstring & s) const;

    bool Contains(char * s) const;

    int find(const cpstring & s) const;
    int find(char * s) const;

// relational operators for cpstring/string comparisons

    friend int operator ==(const cpstring & x, const cpstring & y)
        {return strcmp(x.myCstring, y.myCstring)==0;}

    friend int operator !=(const cpstring & x, const cpstring & y)
        {return ! (x == y);}

    friend int operator < (const cpstring & x, const cpstring & y)
         {return strcmp(x.myCstring, y.myCstring) < 0;}

    friend int operator > (const cpstring & x, const cpstring & y)
         {return strcmp(x.myCstring, y.myCstring) > 0;}

    friend int operator >= (const cpstring & x, const cpstring & y)
         {return x > y || x == y;}

    friend int operator <= (const cpstring & x, const cpstring & y)
         {return x < y || x == y;}

// relational operators for cpstring/char * comparisons

    friend int operator ==(const cpstring & x, const char * s)
        {return strcmp(x.myCstring,s)==0;}

    friend int operator !=(const cpstring & x, const char * s)
        {return ! (x == s);}

    friend int operator < (const cpstring & x, const char * s)
          {return strcmp(x.myCstring, s) < 0;}

    friend int operator > (const cpstring & x, const char *s)
          {return strcmp(x.myCstring, s) > 0;}

    friend int operator >= (const cpstring & x, const char *s)
         {return x > s || x == s;}

    friend int operator <= (const cpstring & x, const char *s)
         {return x < s || x == s;}


// relational operators for char */cpstring comparisons

    friend int operator ==(const char * s, const cpstring & x)
        {return strcmp(s, x.myCstring)==0;}

    friend int operator !=(const char * s, const cpstring & x)
        {return ! (s == x);}

    friend int operator < (const char * s, const cpstring & x)
          {return strcmp(s,x.myCstring) < 0;}

    friend int operator > (const char *s, const cpstring & x)
          {return strcmp(s,x.myCstring) > 0;}

    friend int operator >= (const char *s, const cpstring & x)
         {return s > x || s == x;}

    friend int operator <= (const char *s, const cpstring & x)
         {return s < x || s == x;}

    enum {maxLength = 1024};  // maximum length of cpstring read from input
  private:
    char * myCstring;         // private C-style string
    int myLength;
};

cpstring operator +(const cpstring & x, const cpstring & y);
ostream& operator <<(ostream& os, const cpstring& str);
istream& operator >>(istream& is, cpstring& str);
istream & getline(istream & is, cpstring & s, char sentinel = '\n');

int atoi(const cpstring & s);         // returns int equivalent
double atof(const cpstring & s);      // returns double equivalent
cpstring itoa(int n);

//****************************************************************
//**************** pre/post conditions   *************************
/*****************************************************************

cpstring::cpstring(const char * s)
// precondition: s is '\0'-terminated C-style string
// postcondition: object constructed to have same value as s
// assertion: fails if memory can't be allocated

cpstring::cpstring(const cpstring & x)
// postcondition: object constructed has same value as x
// assertion: fails if memory can't be allocated

cpstring::cpstring(char ch)
// postcondition: object is cpstring of one character ch
// assertion: fails if memory can't be allocated

cpstring& cpstring::operator = (const char * s)
// precondition: s is '\0'-terminated C-style string
// postcondition: assign object the value of C-style string s
// assertion: fails if memory can't be allocated

cpstring& cpstring::operator =(const cpstring & x)
// postcondition: object assigned cpstring x
//                memory allocated only if x.length() > length()
// assertion: fails if memory can't be allocated

cpstring::~cpstring()
// postcondition: dynamically allocated memory freed

char& cpstring::operator[](int i)
// precondition: 0 <= i < length()
// postcondition: returns i-th character
// assertion: fails if index not in range

const char& cpstring::operator[](int i) const
// precondition: 0 <= i < length()
// postcondition: returns i-th character (constant)
// assertion: fails if index not in range

ostream& operator <<(ostream& os, const cpstring& str)
// precondition: os is writeable
// postcondition: str is written to os (uses char * operator)

istream& operator >>(istream& is, cpstring& str)
// precondition: is open for reading
// postcondition: cpstring read from is, stored in str
//                maximum of cpstring::maxLength chars read

istream & getline(istream & is, cpstring & s, char sentinel)
// precondition: is open for reading
// postcondition: chars from is up to sentinel read, stored in s
//                sentinel is read, but NOT stored in s

cpstring& cpstring::operator +=(const cpstring & x)
// postcondition: concatenates x onto object
// assertion: fails if memory can't be allocated     

cpstring operator +(const cpstring & x, const cpstring & y)
// postcondition: returns concat(x,y)

cpstring cpstring::Downcase() const
// postcondition: returns lowercase equivalent of object
//                non-lowercase chars not changed

cpstring cpstring::Upcase() const
// postcondition: returns uppercase equivalent of object
//                non uppercase chars not changed

cpstring cpstring::substr(int pos, int len)
// precondition: 0 <= pos < s.length() && pos + len <= s.length()
// postcondition: returns substring starting at char pos of len chars
//                if len too large, takes s.length() chars

bool cpstring::Contains (const cpstring & s) const
// postcondition: returns true if s is a substring of object

inline int cpstring::Contains(char * s) const
// postcondition: returns true if s is a substring of object

inline int atoi(cpstring & s)
// precondition: s is a sequence of digits
// postcondition: returns integer equivalent of s
// exception: if s is not valid integer, 0 is returned

double atof(cpstring & s)
// precondition: s is a sequence of digits
// postcondition: returns double equivalent of s
// exception: if s is not valid double, 0 is returned

***************************************************************
*************** pre/post conditions   *************************
***************************************************************/

#endif /* _MYSTRING_H not defined */
