/***************************************************************
 *                          VGA VOYAGER                        *
 *   A game of solar system navigation with an unpowered space *
 * probe.                                                      *
 ***************************************************************/

/***************************************************************
 * The colours that various objects will appear.               *
 ***************************************************************/
const playerColour := 14 % The colour of the player.
const planetColour := 13 % The colour of the planets.
const velocityColour := 12 % The colour of the velocity vector in the
%                           velocity determination step.
const accelerationColour := 11 % The colour of the acceleration vector
%                               during the travelling phase.
const edgeColour := 9 % The colour of the edge when the map scale is changed.

/***************************************************************
 *                      Other constants                        *
 ***************************************************************/
const maxPlanets := 8 % maximum number of planets allowed
const totalGravity := 2500 % The gravity constant for the total of all the
%                           planets.  If this is increased, gravity
%                           becomes larger.
% The maximum initial velocity is maxSpeed / maxDivider
const maxSpeed := 24
const maxDivider := 5
const border := 100 % The closest that any planet will appear the edge of
%                    the screen.
const planetSize := 8 % The radius of the planets on the screen.
const maxTimer := 2000 % The number of moves that the probe will travel
%                        before the game ends.

/***************************************************************
 *                  Global Variables                           *
 ***************************************************************/
var scale : int := 0 % The current map scale (0 = close, 1 = just off map,
%                      2 = next step
var gravity : real % The gravity of each individual planet for this round
var timer : int % The number of moves that have passed.
var grandTotal : real := 0 % grand total for scoring

/***************************************************************
 *                  Planet and Player Type                     *
 ***************************************************************/
type singlePlanetType :
    record
	x, y : int % the integer position of the planet
	rx, ry : real % the real position of the planet
	closestApproach : real % the closest the probe has been to the planet
	distance : real % the distance between the probe and the planet
    end record
type playerType :
    record
	x, y : int % the integer position of the probe
	rx, ry : real % the real position of the probe
	vx, vy : real % the velocity of the probe
	px, py : array 0 .. maxTimer of int % the path of the probe
    end record
type planetType : array 1 .. maxPlanets of singlePlanetType

/*******************************************************************
 *                      Instructions                               *
 *   These print the instructions to play the game on the screen.  *
 *******************************************************************/
procedure Instructions
    setscreen ("text,nocursor")
    colour (3)
    locate (1, 25)
    put "VGA VOYAGER", skip
    colour (2)
    put "You are in the solar system Bagnax and wish to send a probe that"
    put "will travel as closely as possible to each of the " ..
    colour (13)
    put "Purple Planets.", skip
    colour (2)
    put "To do so, you must use the arrow keys to set the initial velocity"
    put "of the probe.  Once one has set the desired velocity, hitting the"
    put "space bar will launch the probe.   Remember that gravity will play"
    put "an important part of the probe's trajectory.", skip
    put "The game ends when the probe leaves the solar system, hits a"
    put "planet, runs out of time, or you hit a key.  At the end of the"
    put "game, you will be shown your complete route and a score, indicating"
    put "just how close you came to each of the planets in total.  (The "
    put "score is the sum of the square roots of the closest approach to"
    put "each planet.)  You want to minimize the score.", skip
    put "Scoring under 100 is amazing, under 150 is great and under 200 " ..
    put "is good"
    put ""
    put "While travelling, your acceleration due to gravity will appear as"
    put "a white line pointing away from the source of gravity", skip
    put "Hit any key to start the game..."
    var ch : string (1)
    getch (ch)
    setscreen ("graphics:v16,noecho,nocursor")
end Instructions

/*******************************************************************
 *                       GetIntDist,GetDist                        *
 *   These functions return the distance between two points.       *
 *******************************************************************/
function GetIntDist (x1, y1, x2, y2 : int) : real
    result sqrt ( ( (x1 - x2) * (x1 - x2)) + ( (y1 - y2) * (y1 - y2)))
end GetIntDist

function GetDist (x1, y1, x2, y2 : real) : real
    result sqrt ( ( (x1 - x2) * (x1 - x2)) + ( (y1 - y2) * (y1 - y2)))
end GetDist

/******************************************************************
 *                      ScaleX,ScaleY                              *
 *   These functions return the new co-ordinate of a point for a   *
 * given scale.  The argument is the co-ordinate at scale 0        *
 *******************************************************************/
function ScaleX (x : int) : int
    result round ( (x + (scale * maxx)) / (2 * scale + 1))
end ScaleX

function ScaleY (y : int) : int
    result round ( (y + (scale * maxy)) / (2 * scale + 1))
end ScaleY

/*******************************************************************
 *                       EmptyBuffer                               *
 *   This routine empties the keyboard buffer of any key presses.  *
 *******************************************************************/
procedure EmptyBuffer
    var c : string (1)
    loop
	exit when not hasch
	getch (c)
    end loop
end EmptyBuffer

/*******************************************************************
 *                     InitPlanets                                 *
 *   Places each of the planets a minimum of 4 planetSize's apart  *
 * and at least border pixels from the edge of the screen.         *
 *******************************************************************/
procedure InitPlanets (var planet : planetType, numPlanets : int)
    for i : 1 .. numPlanets
	loop
	    randint (planet (i).x, border, maxx - border)
	    randint (planet (i).y, border, maxy - border)
	    var match : boolean := false
	    for j : 1 .. i - 1
		if GetIntDist (planet (i).x, planet (i).y,
			planet (j).x, planet (j).y) < 4 * planetSize then
		    match := true
		end if
	    end for
	    exit when not match
	end loop
	planet (i).rx := planet (i).x
	planet (i).ry := planet (i).y
	planet (i).closestApproach := 1000.0
    end for
end InitPlanets

/*****************************************************************
 *                      InitPlayerPostion                        *
 *   This places the player's staring position at least three    *
 * planetSize's away from any planet.                            *
 *****************************************************************/
procedure InitPlayerPosition (var player : playerType, planet : planetType,
	numPlanets : int)
    loop
	randint (player.x, 0, maxx)
	randint (player.y, 0, maxy)
	var match : boolean := false
	for i : 1 .. numPlanets
	    if GetIntDist (player.x, player.y, planet (i).x, planet (i).y)
		    < planetSize * 3 then
		match := true
	    end if
	end for
	exit when not match
    end loop
    player.rx := player.x
    player.ry := player.y
    player.vx := 0.0
    player.vy := 0.0
end InitPlayerPosition

/*****************************************************************
 *                     InitPlayerVelocity                        *
 *   This sets up the stating velocity of the playter.           *
 *****************************************************************/
procedure InitPlayerVelocity (var player : playerType)
    var vx : int := 0
    var vy : int := 0
    var c : string (1)
    vx := 0
    drawbox (player.x - 1, player.y - 1, player.x + 1, player.y + 1,
	playerColour)
    drawfill (player.x, player.y, playerColour, playerColour)
    loop
	getch (c)
	drawline (player.x, player.y, player.x + vx, player.y + vy, 0)
	case ord (c (1)) of
		% The 7 key (on the keypad)
	    label 199 :
		vx -= 1
		vy += 1
		% The 8 key (on the keypad)
	    label 200 :
		vy += 1
		% The 9 key (on the keypad)
	    label 201 :
		vx += 1
		vy += 1
		% The 4 key (on the keypad)
	    label 203 :
		vx -= 1
		% The 6 key (on the keypad)
	    label 205 :
		vx += 1
		% The 1 key (on the keypad)
	    label 207 :
		vx -= 1
		vy -= 1
		% The 2 key (on the keypad)
	    label 208 :
		vy -= 1
		% The 3 key (on the keypad)
	    label 209 :
		vx += 1
		vy -= 1
		% The space or carriage return character
	    label ord (" "), 13 :
		exit
	    label :
	end case
	% This prevents the velocity from going above maxSpeed.
	if vx < - maxSpeed then
	    vx := - maxSpeed
	elsif vx > maxSpeed then
	    vx := maxSpeed
	end if
	if vy < - maxSpeed then
	    vy := - maxSpeed
	elsif vy > maxSpeed then
	    vy := maxSpeed
	end if
	drawline (player.x, player.y, player.x + vx, player.y + vy,
	    velocityColour)
	EmptyBuffer
    end loop
    player.vx := vx / maxDivider
    player.vy := vy / maxDivider
end InitPlayerVelocity

/*****************************************************************
 *                     DrawPlayingField                          *
 *   This draws the screen (without paths) at any scale.         *
 *****************************************************************/
procedure DrawPlayingField (player : playerType, planet : planetType,
	numPlanets : int)
    cls
    if scale = 0 then
	for i : 1 .. numPlanets
	    bind p to planet (i)
	    drawoval (p.x, p.y, planetSize, planetSize, planetColour)
	    drawfill (p.x, p.y, planetColour, planetColour)
	end for
	drawdot (player.x, player.y, playerColour)
    else
	drawline (ScaleX (0), ScaleY (0), ScaleX (maxx), ScaleY (0),
	    edgeColour)
	drawline (ScaleX (maxx), ScaleY (0), ScaleX (maxx), ScaleY (maxy),
	    edgeColour)
	drawline (ScaleX (maxx), ScaleY (maxy), ScaleX (0), ScaleY (maxy),
	    edgeColour)
	drawline (ScaleX (0), ScaleY (maxy), ScaleX (0), ScaleY (0),
	    edgeColour)

	for i : 1 .. numPlanets
	    drawdot (ScaleX (planet (i).x), ScaleY (planet (i).y),
		planetColour)
	end for
	drawdot (ScaleX (player.x), ScaleY (player.y), playerColour)
    end if
end DrawPlayingField

/*******************************************************************
 *                      ChangeScale                                *
 *   This sets up a new scale and draws the screen and the probe's *
 * path at the new scale.                                          *
 *******************************************************************/
procedure ChangeScale (newScale : int, player : playerType,
	planet : planetType, numPlanets : int)
    scale := newScale
    DrawPlayingField (player, planet, numPlanets)
    if scale = 0 then
	for i : 0 .. timer - 1
	    drawdot (player.px (i), player.py (i), playerColour)
	end for
    else
	for i : 0 .. timer - 1
	    drawdot (ScaleX (player.px (i)), ScaleY (player.py (i)),
		playerColour)
	end for
    end if
end ChangeScale

/*******************************************************************
 *                      Main Procedure                             *
 *******************************************************************/
setscreen ("graphics:v16,noecho")
randomize
Instructions

for numPlanets : 2 .. maxPlanets
    var planets : planetType
    var player : playerType
    var hitPlanet : boolean := false
    var ax, ay : real
    var lax : int := 0
    var lay : int := 0

    timer := 0
    gravity := totalGravity / numPlanets
    InitPlanets (planets, numPlanets)
    InitPlayerPosition (player, planets, numPlanets)
    ChangeScale (0, player, planets, numPlanets)
    InitPlayerVelocity (player)
    DrawPlayingField (player, planets, numPlanets)
    EmptyBuffer

    /* Start Main Loop */
    loop
	% This sets the path
	player.px (timer) := player.x
	player.py (timer) := player.y

	% For each planet, calculate the distance, checking for impact
	% or a closest approach so far.
	for i : 1 .. numPlanets
	    planets (i).distance := GetDist (player.rx, player.ry,
		planets (i).rx, planets (i).ry)
	    planets (i).closestApproach := min (planets (i).closestApproach,
		planets (i).distance)
	    if planets (i).distance < planetSize + 0.5 then
		hitPlanet := true
	    end if
	end for
	timer += 1
	% Leave the main loop when:
	%     The probe has run out of time
	exit when timer > maxTimer
	%     A key has been hit
	exit when hasch
	%     The probe hit a planet
	exit when hitPlanet

	% Calculate the scale factor necessary to display the probe on the screen.
	var sx := player.x div (maxx + 1)
	if player.x < 0 then
	    sx := abs (sx) + 1
	end if
	var sy := player.y div (maxy + 1)
	if player.y < 0 then
	    sy := abs (sy) + 1
	end if
	if sx > sy then
	    if sx not= scale then
		ChangeScale (sx, player, planets, numPlanets)
	    end if
	else
	    if sy not= scale then
		ChangeScale (sy, player, planets, numPlanets)
	    end if
	end if

	% If the probe is going too far and is unlikely to return, leave
	% the loop
	exit when scale = 2

	% ax and ay are the total accelertation in the x and y directions
	ax := 0
	ay := 0

	% For each planet, calculate the gravitational effect in the x and y
	% direction and add it to ax and ay.
	for i : 1 .. numPlanets
	    var g, gx, gy : real
	    g := gravity / (planets (i).distance ** 2)
	    const dx : real := planets (i).rx - player.rx
	    const dy : real := planets (i).ry - player.ry
	    gx := g * dx / (abs (dx) + abs (dy))
	    gy := g * dy / (abs (dx) + abs (dy))
	    ax += gx
	    ay += gy
	end for

	% Change the player's velocity by the accleration
	player.vx += ax
	player.vy += ay

	% Change the player's position by the velocity
	player.rx += player.vx
	player.ry += player.vy

	% Plot the position and the acceleration vector
	if scale = 0 then
	    % Erase last turn's acceleration vector
	    if lax not= 0 or lay not= 0 then
		drawline (player.x, player.y, player.x - lax, player.y -
		    lay, 0)
	    end if

	    % Draw a dot just in case the previous statement erased
	    % last turn's position
	    drawdot (player.x, player.y, playerColour)

	    % Calculate the new (integer) position and acceleration
	    player.x := round (player.rx)
	    player.y := round (player.ry)
	    lax := round (ax * 400)
	    lay := round (ay * 400)

	    % Draw the acceleration vector if there is one
	    if lax not= 0 or lay not= 0 then
		drawline (player.x, player.y, player.x - lax, player.y - lay,
		    accelerationColour)
	    end if

	    % Draw a dot for the player's position
	    drawdot (player.x, player.y, playerColour)
	else
	    % Erase last turn's acceleration vector
	    if lax not= 0 or lay not= 0 then
		drawline (ScaleX (player.x), ScaleY (player.y),
		    ScaleX (player.x) - lax, ScaleY (player.y) - lay, 0)
	    end if

	    % Draw a dot just in case the previous statement erased
	    % last turn's position
	    drawdot (ScaleX (player.x), ScaleY (player.y), playerColour)

	    % Calculate the new (integer) position and acceleration
	    player.x := round (player.rx)
	    player.y := round (player.ry)
	    lax := round (ax * 400)
	    lay := round (ay * 400)

	    % Draw the acceleration vector if there is one
	    if lax not= 0 or lay not= 0 then
		drawline (ScaleX (player.x), ScaleY (player.y),
		    ScaleX (player.x) - lax, ScaleY (player.y) - lay,
		    accelerationColour)
	    end if

	    % Draw a dot for the player's position
	    drawdot (ScaleX (player.x), ScaleY (player.y), playerColour)
	end if
    end loop

    % Write the ending message
    var c : string (1)
    locate (1, 1)
    if hitPlanet then
	put "The probe hit a planet!"
    elsif timer >= maxTimer then
	put "The probe ran out of time."
    elsif hasch then
	put "You gave up."
	getch (c)
    else
	put "The probe is not coming back..."
    end if

    % Wait for a key press
    getch (c)

    % Show the map at scale = 1 to show details of flight.
    ChangeScale (1, player, planets, numPlanets)

    % Calculate the score
    var total : real := 0
    locate (4, 1)
    for i : 1 .. numPlanets
	total += 20 * sqrt (max (0.0, planets (i).closestApproach -
	    planetSize))
	put i : 1, " : ", round (20 * sqrt (max (0.0, planets
	    (i).closestApproach -
	    planetSize)))
    end for
    total := total / numPlanets
    grandTotal += total

    % Display the score.
    locate (1, 1)
    put "The averaged total is ", round (total)
    locate (23, 1)
    put "[Press any key to continue]"

    % Wait for a key press
    getch (c)

end for

% Display the grand total for each round
cls
grandTotal := grandTotal / (maxPlanets - 1)
put "Average of all totals is ", round (grandTotal)
