setscreen ("graphics:v16,noecho,nocursor")              % Initialize graphics

% IBM Key constants
const Esc := chr (27)
const spacebar := chr (32)
const enter := chr (10)


% Arrow Keys
const left := chr (203)
const right := chr (205)
const down := chr (208)
const up := chr (200)


% Colour constants
const background := 0
const blue := 1
const green := 2
const cyan := 3
const red := 4
const magenta := 5
const brown := 6
const lightgray := 7
const darkgray := 8
const highblue := 9
const highgreen := 10
const highcyan := 11
const highred := 12
const highmagenta := 13
const yellow := 14
const white := 15

const border := darkgray                        % Border colour of each block
const minblkcolour := highblue
const maxblkcolour := yellow
const bombcolour := white                                   % Colour of bomb

const bombnumber := 100             % Number of pieces to fall between bombs

% Speed constants
const initspeed := 2000
		     % Initial time in miliseconds that takes column to fall

% Scoring constants
const matchscr := 5                             % Bonus for making one match
const dropscr := 2    % Bonus related to vertical distance column is dropped
const matchlvl := 50         % Number of matches to make to go to next level
const scorex := maxx - 175                  % Left pixel value of scoreboard

% Size constants
const xblks := 6                        % Blocks per column in playing field
const yblks := 13                           %Blocks per row in playing field
const blksize := 25                                % Size of block in pixels
const numblks := 3                      % Maximum number of blocks in column
const matchnum := 3    % Minimum number of matches to make before they erase

const chrsize := 8                   % Horizontal size of a letter in pixels

const mostmatches := xblks + numblks        % Maximum number matching blocks

const shift := 25         % Size of playing field shift off center in pixels


function CX (x : int) : int
    % Converts column blocks to x coordinate in pixels
    result maxx div 2 - blksize * xblks div 2 + (x - 1) * blksize + x - shift
end CX


function CY (y : int) : int
    % Converts column blocks to y coordinate in pixels
    result maxy div 2 - blksize * yblks div 2 + (y - 1) * blksize + y
end CY


% Constants for next piece
const snextpx := CX (xblks + 2)          % Shownext bottom position in pixels
const snextpy := CY (yblks - numblks + 1)  % Shownext left position in pixels
const snextx := xblks + 2                           % Shownext bottom block x
const snexty := yblks - numblks + 1                 % Shownext bottom block y

% Global Variables
var colbuffer : array 1 .. sizepic (CX (1), CY (1), CX (1) + blksize,
    CY (yblks) + blksize) of int            % Buffer to hold any size column

var blkcolour : array 1 .. numblks of int      % Current column block colours
var nblkcolour : array 1 .. numblks of int        % Next column block colours

var scrolled : boolean       % True if entire column is seen in playing field
var scrollc : int    % Number of blocks that have scrolled into playing field

var fallspeed : int           % Time in miliseconds that takes column to fall
var x, y : int                                        % Coordinates of column
var key : string (1)

var top : array 1 .. xblks of int
		     % Lowest vacant position of each column in playing field

var blkmatrix : array 1 .. xblks, 1 .. yblks + numblks - 1 of int
			% Stores colour values of each block in playing field

var matchx, matchy : array 1 .. xblks * yblks of int
		   % Stores match coordinates of all matches in playing field

var matchcount : int                        % Counts matches in one direction
var totmatch : int                                       % Counts all matches

var tempmatchx, tempmatchy : array 1 .. mostmatches of int
				      % Stores match coordinates of one block

var matchfound : boolean       % True if a match has been found on last check
var dropped : boolean                       % True if column has been dropped
var bomb : boolean                          % True if current piece is a bomb
var soundon : boolean                                   % True if sound is on
var endgame : boolean                        % True if quit button is pressed

var lastx : int                       % Last x position that has been checked

var score : int
var columnc : int                                            % Column counter
var level : int                                               % Current level
var matchc : int                         % Number of matches in current level

var start, finish : int := 0           % Start/Finish time for column to fall


function DISTANCE (c : int) : int
    % Calculates distance from column blocks to pixels
    result blksize * c + c - 1
end DISTANCE


procedure SWALLOW_KEYS
    % Empties keyboard buffer
    loop
	exit when not hasch
	getch (key)
    end loop
end SWALLOW_KEYS


procedure HEX (x, y, col, size : int)
    % Draws the ouline of a hexagon

    drawline (x, size div 2 + y, size div 4 + x, size + y, col)
    drawline (size div 4 + x, size + y, size - size div 4 + x, size + y, col)
    drawline (size - size div 4 + x, size + y, size + x, size div 2 + y, col)
    drawline (size + x, size div 2 + y, size - size div 4 + x, y, col)
    drawline (size - size div 4 + x, y, size div 4 + x, y, col)
    drawline (size div 4 + x, y, x, size div 2 + y, col)
end HEX


procedure PUTXY (x, y, col : int, str : string (*))
    % Displays a string given coordinates

    locate (y, x)
    colour (col)
    put str ..
end PUTXY


procedure TITLE_SCREEN
    var hexbuffer : array 1 .. sizepic (0, 0, 20, 20) of int
					% Buffer to hold image of one hexagon

    var hexcolour : int                           % Colour of current hexagon

    var titlex : array 1 .. 54 of int :=
	init (1, 7, 2, 6, 3, 5, 4, 3, 5, 2, 6, 1, 7, 1, 2, 3, 4, 5, 6, 7, 1,
	1, 1, 2, 3, 4, 1, 1, 1, 2, 3, 4, 5, 6, 7,
	1, 7, 1, 7, 1, 7, 1, 2, 3, 4, 5, 6, 7, 1, 7, 1, 7, 1, 7)
    var titley : array 1 .. 54 of int :=
	init (1, 1, 2, 2, 3, 3, 4, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 8, 9,
	10, 11, 11, 11, 11, 12, 13, 14, 14, 14
	, 14, 14, 14, 14, 15, 15, 16, 16, 17, 17, 18, 18, 18, 18, 18, 18,
	18, 19, 19, 20, 20, 21, 21)
	      % Variables that store coordinates of each hexagon in title HEX

    for c : 1 .. 54
	const X := titlex (c) * 21
	const Y := titley (c) * 21                   % Coordinates of hexagon

	if titley (c) <= 7 then
	    hexcolour := highblue
	elsif titley (c) <= 14 then
	    hexcolour := highred
	else
	    hexcolour := highmagenta
	end if                       % Each letter is given a different colour

	if not hasch then
	    HEX (X, 441, border, 20)
	    drawfill (X + 10, 451, hexcolour, border)
	    takepic (X, 441, X + 20, 461, hexbuffer)
	    for decreasing drop : 21 .. titley (c)
		drawpic (X, drop * 21, hexbuffer, 0)
		delay (75)
		drawpic (X, drop * 21, hexbuffer, 1)
	    end for            % Drops each hexagon into place to form letter
	end if

	HEX (X + 3, Y - 3, border, 20)
	drawfill (X + 13, Y + 7, border, border)
	HEX (X, Y, hexcolour, 20)
	drawfill (X + 10, Y + 10, hexcolour, hexcolour)
					     % Draw hexagon in final position
    end for

    if hasch then
	getch (key)
    end if

end TITLE_SCREEN


procedure BOX_3D (x1, y1, x2, y2, width : int)
    % Draws a rectangluar figure that appears lifted off the screen

    drawbox (x1 + width, y1 + width, x2 - width, y2 - width, white)
    drawbox (x1, y1, x2, y2, white)
    drawline (x1, y1, x1 + width, y1 + width, white)
    drawline (x2, y2, x2 - width, y2 - width, white)
    drawfill (x1 + 1, y1 + width, white, white)
    drawfill (x2 - 1, y2 - width, darkgray, white)

    drawline (x1, y1, x1 + width, y1 + width, darkgray)
    drawline (x2, y2, x2 - width, y2 - width, darkgray)
    drawline (x1, y1, x2, y1, darkgray)
    drawline (x1 + width, y1 + width, x2 - width, y1 + width, darkgray)
    drawline (x2, y2, x2, y1, darkgray)
    drawline (x2 - width, y2 - width, x2 - width, y1 + width, darkgray)

    drawbox (x1, y1, x2, y2, darkgray)
end BOX_3D


procedure SETUP
    % Initializes some variables and draws playing screen

    columnc := 0
    score := 0
    level := 1
    matchc := 0
    soundon := true
    endgame := false
    totmatch := 0

    for c : 1 .. xblks * yblks
	matchx (c) := 0
	matchy (c) := 0
    end for

    fallspeed := initspeed

    for x : 1 .. xblks
	for y : 1 .. yblks + numblks - 1
	    blkmatrix (x, y) := 0                       % Blanks playing field
	end for
    end for

    for x : 1 .. xblks
	top (x) := 1             % Lowest position of each column is at bottom
    end for

    randomize                             % Initialize random number generator

    for count : 1 .. numblks
	randint (nblkcolour (count), minblkcolour, maxblkcolour)
		       % Give first blocks in the column a random colour value
    end for

    cls
    colourback (lightgray)

    BOX_3D (CX (1) - 7, CY (1) - 7, CX (xblks + 1) + 6, CY (yblks + 1) + 6,
	6)                                        % Draws playing field border

    BOX_3D (scorex, maxy - 100, maxx - 10, maxy - 10, 3)
    const title := "HEXTRIS"
    locate (2, 65)
    for c : 1 .. length (title)
	if c mod 2 = 0 then
	    colour (highred)
	else
	    colour (highblue)
	end if
	put title (c) ..
    end for
    PUTXY (61, 4, white, "  Programmed by")
    PUTXY (61, 5, white, "    Mike Banik")

    BOX_3D (scorex, maxy - 275, maxx - 10, maxy - 110, 3)
    PUTXY (61, 9, blue, "  Score:")
    PUTXY (61, 11, blue, "  Pieces:")
    PUTXY (61, 13, blue, "  Level:")
    PUTXY (61, 15, blue, "  Matches:")

    BOX_3D (scorex, 10, maxx - 10, maxy - 285, 3)
    PUTXY (61, 20, highred, "  KEY TEMPLATE")
    PUTXY (64, 21, magenta, chr (16) + "     Right")
    PUTXY (64, 22, magenta, chr (17) + "     Left")
    PUTXY (64, 23, magenta, chr (31) + "     Down")
    PUTXY (64, 24, magenta, chr (30) + "     Cycle")
    PUTXY (64, 25, magenta, "Space Drop")
    PUTXY (62, 27, magenta, "'S'  Sound is on ")
    PUTXY (62, 28, magenta, "'H'  Help me!")
    PUTXY (62, 29, magenta, "Esc  Pause/Quit")

    for c : 0 .. numblks - 1
	HEX (CX (xblks + 2), CY (yblks + 1 - numblks + c), border, blksize)
    end for

    TITLE_SCREEN
    PUTXY (29, 28, highblue, "HIT ANY KEY TO PLAY!")
    getch (key)
    PUTXY (29, 28, background, "HIT ANY KEY TO PLAY!")
end SETUP


procedure GET_COLOURS
    % Assigns each block in the column and next column a random colour value

    blkcolour := nblkcolour
			    % Assigns current blocks to previous next colours

    for count : 1 .. numblks
	randint (nblkcolour (count), minblkcolour, maxblkcolour)
				  % Assigns next blocks a random colour value
    end for
end GET_COLOURS


procedure PAINT_COLUMN (x, y : int, blkcolour : array 1 .. numblks of int)
    for count : 0 .. numblks - 1
	if y + count <= yblks then
	    drawfill (CX (x) + blksize div 2, CY (y + count) + blksize div 2,
		blkcolour (count + 1), border)
	end if
    end for
end PAINT_COLUMN


procedure CYCLE
    % Rearranges the colour of the blocks in the column

    const temp := blkcolour (1)

    for count : 1 .. numblks - 1
	blkcolour (count) := blkcolour (count + 1)
		% Each block is given colour value of block above it in column
    end for

    blkcolour (numblks) := temp
			    % Top block is given colour vlaue of bottom block

    PAINT_COLUMN (x, y, blkcolour)

    if not scrolled then
	PAINT_COLUMN (snextx, snexty, blkcolour)
    end if

    takepic (CX (x), CY (y), CX (x) + blksize, CY (y + scrollc) - 1
     , colbuffer)                            % Store new column in a buffer
end CYCLE


procedure MOVE_DOWN
    drawpic (CX (x), CY (y), colbuffer, 1)                  % Erase old image
    y := y - 1                                        % New vertical location
    drawpic (CX (x), CY (y), colbuffer, 0)     % Redraw image in new location
end MOVE_DOWN


procedure DROP
    % Makes piece drop to bottom of its coloumn in playing field

    drawpic (CX (x), CY (y), colbuffer, 1)                  % Erase old piece

    score := score + level * (dropscr * (y - top (x)))
			      % Calculates bonus according to how far dropped

    y := top (x)                                % New lower vertical position

    if not scrolled then
	var gap : int
	       %Gap between top of playing field and closest block to the top

	if yblks + 1 - y < numblks then
	    gap := yblks + 1 - y
	else
	    gap := numblks
	end if

	takepic (snextpx, snextpy, snextpx + blksize, snextpy + DISTANCE
	    (gap), colbuffer)

	scrollc := yblks + 1 - y
	if scrollc >= numblks then
	    PAINT_COLUMN (snextx, snexty, nblkcolour)
	    scrolled := true
	end if
    end if

    drawpic (CX (x), CY (y), colbuffer, 0)  % Draw piece in new lower position
end DROP


procedure ERASE_MATCHES
    % Erases all of the blocks matches in one direction

    score := score + round (totmatch / 2 * (matchscr * 2 + matchscr *
	(totmatch - 1)))
		    % Calculates bcore according to number of matches made
		    % Calculated according to an arithmetic sequence
		    % with common difference equal to matchscr

    matchc := matchc + totmatch     % Increments match counter for scoreboard

    for c : 1 .. totmatch
	if matchy (c) <= yblks then
	    takepic (CX (matchx (c)), CY (matchy (c)), CX (matchx (c)) +
		blksize, CY (matchy (c)) + blksize, colbuffer)
						    % Store image of one block

	    for w : 1 .. 2
		drawpic (CX (matchx (c)), CY (matchy (c)), colbuffer, 0)
		delay (100)
		drawpic (CX (matchx (c)), CY (matchy (c)), colbuffer, 1)
								 % Erase match
	    end for                                  % Creates flashing affect

	    if soundon then
		sound (1000, 50)                 % Sounds when a match is made
	    end if
	end if

	blkmatrix (matchx (c), matchy (c)) := background
    end for
end ERASE_MATCHES


procedure INSERTION (x, y : int)
    % Inserts xy coordinates into array in order

    var pos : int                                % Temparary position in list

    matchx (totmatch + 1) := 1000
    matchy (totmatch + 1) := 1000

    for c : 1 .. totmatch + 1
	pos := c
	exit when x <= matchx (c)
    end for

    for c : pos .. totmatch + 1
	pos := c
	exit when matchx (c) > x
	exit when y < matchy (c) and x <= matchx (c)
	exit when matchx (c) = x and matchy (c) = y
    end for

    if not (x = matchx (pos) and y = matchy (pos)) then
	totmatch := totmatch + 1
	for decreasing c : totmatch + 1 .. pos + 1
	    matchx (c) := matchx (c - 1)
	    matchy (c) := matchy (c - 1)
	end for

	matchx (pos) := x
	matchy (pos) := y
    end if
end INSERTION


function IN_RANGE (x, y : int) : boolean
    % Results true if given coordinates are in playing field range

    if x < 1 or x > xblks or y < 1 or y > yblks + numblks - 1 then
	result false
    else
	result true
    end if
end IN_RANGE


procedure DIRECTION_CHECK (x, y, xinc, yinc : int)
    % Checks for matches of one block in one direction

    for c : 1 .. mostmatches
	exit when not IN_RANGE (x + c * xinc, y + c * yinc) or
	    blkmatrix (x + c * xinc, y + c * yinc) not= blkmatrix (x, y)
	tempmatchx (matchcount) := x + c * xinc
	tempmatchy (matchcount) := y + c * yinc
	matchcount := matchcount + 1
    end for
end DIRECTION_CHECK


procedure CHECK (x, y, xinc, yinc : int)
    % Checks for matches of one block in positive and negative direction

    const temptotmatch := totmatch

    matchcount := 1

    DIRECTION_CHECK (x, y, xinc, yinc)            % Checks positive direction
    DIRECTION_CHECK (x, y, - xinc, - yinc)        % Checks negative direction

    if matchcount >= matchnum then
	matchfound := true

	tempmatchx (matchcount) := x
	tempmatchy (matchcount) := y

	for c : 1 .. matchcount
	    INSERTION (tempmatchx (c), tempmatchy (c))
	end for
    else
	totmatch := temptotmatch
    end if

end CHECK



procedure COLUMN_CHECK (x, y : int)
    % Checks an entire column of blocks for matches in every direction

    for c : y .. yblks + numblks - 1
	exit when blkmatrix (x, c) = 0
	CHECK (x, c, 0, 1)
	CHECK (x, c, 1, 0)
	CHECK (x, c, 1, 1)
	CHECK (x, c, - 1, 1)
    end for
end COLUMN_CHECK


procedure GRAVITY (x, y : int)
    % Forces blocks to fall into lower vacant positions

    var dis : int := 1                          % Vertical distance to "fall"

    if y > 1 then
	for decreasing c : y .. 2
	    exit when blkmatrix (x, c - 1) not= 0
	    dis := dis + 1
	end for
    end if

    top (x) := top (x) - 1

    totmatch := totmatch - dis
    const checkx := matchx (totmatch + 1)
    const checky := matchy (totmatch + 1)

    if y < top (x) then
	for c : y - dis + 1 .. top (x)
	    if c + dis <= yblks + numblks - 1 then
		blkmatrix (x, c) := blkmatrix (x, c + dis)
	    else
		blkmatrix (x, c) := background
	    end if                                      % Reassigns matrix
	end for

    end if

    if y + 1 < yblks then  % For blocks that have scrolled into playing field
	takepic (CX (x), CY (y + 1), CX (x) + blksize, CY (top (x))
	    + blksize, colbuffer)
	drawpic (CX (x), CY (y + 1), colbuffer, 1)          % Erase old image

	drawpic (CX (x), CY (y - dis + 1), colbuffer, 0)
						% Redraw image in new location

    else                % For blocks that have NOT scrolled into playing field

	if y + 1 <= yblks then
	    takepic (CX (x), CY (y + 1), CX (x) + blksize, CY (yblks) +
		blksize, colbuffer)

	    drawpic (CX (x), CY (y + 1), colbuffer, 1)      % Erase old image

	    drawpic (CX (x), CY (y - dis + 1), colbuffer, 0)
						% Redraw image in new location
	end if

	if top (x) > yblks then
			% Draws blocks that do not yet appear in playing field

	    for c : yblks - dis .. yblks
		exit when blkmatrix (x, c) = background
		HEX (CX (x), CY (c), border, blksize)
		drawfill (CX (x) + blksize div 2, CY (c) + blksize div 2,
		    blkmatrix (x, c), border)
	    end for
	end if
    end if
    top (x) := top (x) - dis + 1

    if totmatch not= 0 then
	GRAVITY (matchx (totmatch), matchy (totmatch))
			     % Calls self for next blocks to fall into place
    end if

    if checkx not= lastx then
	COLUMN_CHECK (checkx, checky)   % Re-checks fallen column for matches
    end if
    lastx := checkx                        % Last horizontal position checked
end GRAVITY


procedure HORIZONTAL_MOVE
    % Moves column left and right in playing field

    if (key = left) and (x - 1 >= 1) and (blkmatrix (x - 1, y) = 0) then
	drawpic (CX (x), CY (y), colbuffer, 1)              % Erase old piece
	x := x - 1
	drawpic (CX (x), CY (y), colbuffer, 0) % Redraw column in new position

    elsif (key = right) and (x + 1 <= xblks) and (blkmatrix (x + 1, y) = 0)
     then
	drawpic (CX (x), CY (y), colbuffer, 1)              % Erase old piece
	x := x + 1
	drawpic (CX (x), CY (y), colbuffer, 0) % Redraw column in new position
    end if
end HORIZONTAL_MOVE


procedure SCROLL
    scrollc := yblks + 1 - y

    takepic (snextpx, snextpy, snextpx + blksize, CY (snexty + scrollc) - 1
	, colbuffer)
	   % Stores image of new larger part of piece that appears in field

    drawpic (CX (x), CY (y), colbuffer, 0)      % Draws piece in new position

    if scrollc >= numblks then
	PAINT_COLUMN (snextx, snexty, nblkcolour) % Draws next piece to fall
	scrolled := true
    end if
end SCROLL


procedure BOMB
    % Special piece which makes every block of the same colour disappear of
    % the colour of the block that it lands on

    PUTXY (51, 11, background, "BOMB!")                % Erases BOMB! warning

    for c : 0 .. numblks - 1
	INSERTION (x, y + c)
    end for

    if y > 1 then
	const bombblk := blkmatrix (x, y - 1)     % Colour of block landed on

	for X : 1 .. xblks
	    for Y : 1 .. top (X) - 1
		if blkmatrix (X, Y) = bombblk then
		    INSERTION (X, Y)
		    drawfill (CX (X) + blksize div 2, CY (Y) + blksize div 2,
			bombcolour, border)
		end if
	    end for
	end for
	delay (800)
    end if

    matchfound := true
end BOMB


procedure SCORE_BOARD
    % Updates scoreboard

    columnc := columnc + 1               % Increments number of pieces fallen

    if (columnc + 1) mod bombnumber = 0 then
	for c : 1 .. numblks
	    nblkcolour (c) := bombcolour          % Assigns next piece a bomb
	end for
    end if

    if columnc mod bombnumber = 0 then
	bomb := true
	PUTXY (51, 11, highblue, "BOMB!")             % Warns player of BOMB!
    else
	bomb := false
    end if

    if matchc >= matchlvl then                                 % Level Change
	matchc := matchc mod matchlvl
	level := level + 1                                % Increments levels
	if fallspeed > round (initspeed * 0.1) then
	    fallspeed := fallspeed - round (initspeed * 0.1) %Increment speed
	end if

	const scale := "cdefgab"
	play ("6")                              % Set to play sixteenth notes
	for c : 1 .. 7
	    colourback (c + 8)                    % Changes background colour
	    if soundon then
		play (scale (c))                         % Plays musical note
	    else
		delay (200)
	    end if
	end for
	colourback (lightgray)       % Changes background to origional colour

    end if

    colour (blue)
    locate (9, 70)
    put score : 6
    locate (11, 70)
    put columnc : 6
    locate (13, 70)
    put level : 6
    locate (15, 70)
    put matchc : 6                                       % Updates scoreboard

    const matchboxx := scorex + 25
    const matchboxy := 220
    const matchboxsize := 120
    const matchboxcolour := level mod 8 + 8

    % Draws horizontal bar graph of matches to go before next level
    drawbox (matchboxx, matchboxy, matchboxx + matchboxsize, matchboxy + 10,
	matchboxcolour)
    drawline (matchboxx + matchboxsize * matchc div matchlvl, matchboxy,
	matchboxx + matchboxsize * matchc div matchlvl, matchboxy + 10,
	matchboxcolour)

    if matchc not= 0 then
	drawfill (matchboxx + matchboxsize * matchc div matchlvl - 1,
	    matchboxy + 5, matchboxcolour, matchboxcolour)
    end if
    drawfill (matchboxx + matchboxsize * matchc div matchlvl + 1,
	matchboxy + 5, background, matchboxcolour)
end SCORE_BOARD


procedure STORE_RESTORE_SCREEN (x1, y1, x2, y2 : int, store : boolean)
    % Stores/restores a section of the screen to/from a file

    var filenum : int
    var screenbuffer : array 1 .. sizepic (x1, y1, x2, y2) of int
				     % Buffer to store image of screen to save

    takepic (x1, y1, x2, y2, screenbuffer)

    if store then                                                    % Storing
	drawpic (x1, y1, screenbuffer, 1)            % Erase section of screen
	BOX_3D (x1, y1, x2, y2, 4)
	open : filenum, "ERASE.ME", write
	write : filenum, screenbuffer            % Store image in binary file
	close (filenum)
    else                                                          % Restoring
	open : filenum, "ERASE.ME", read
	read : filenum, screenbuffer         % Read in image from binary file
	drawpic (x1, y1, screenbuffer, 0)         % Re-draw section of screen
	close (filenum)
    end if
end STORE_RESTORE_SCREEN


procedure PAUSE
    STORE_RESTORE_SCREEN (194, 194, 404, 279, true)

    PUTXY (36, 14, highred, "PAUSED")
    PUTXY (30, 16, blue, "Hit 'Q' to Quit or")
    PUTXY (26, 17, blue, "Press any key to continue")
    getch (key)

    STORE_RESTORE_SCREEN (194, 194, 404, 279, false)

    if index ("qQ", key) > 0 then
	endgame := true
    end if
end PAUSE


procedure HELP_ME
    STORE_RESTORE_SCREEN (154, 189, 434, 289, true)
    PUTXY (25, 14, highred, "   OBJECT OF THE GAME")
    PUTXY (25, 15, blue, "Align 3 or more hexagons ")
    PUTXY (25, 16, blue, "of the same colour in any")
    PUTXY (25, 17, blue, "direction to make matches")
    getch (key)

    STORE_RESTORE_SCREEN (154, 189, 434, 289, false)
end HELP_ME


procedure GAME_OVER
    var gameoverbuffer : array 1 .. sizepic (0, maxy, chrsize * 9, maxy -
	16) of int          % Buffer to store the image of the word GAME OVER

    if soundon then
	sound (50, 1000)
    end if

    locatexy (216, 32)
    colour (white)
    put "GAME OVER"

    takepic (216, 32, 216 + chrsize * 9, 48, gameoverbuffer)

    drawpic (216, 32, gameoverbuffer, 1)

    for decreasing c : yblks .. 1
	drawpic (round (maxx / 2 - chrsize * 9 / 2) - shift,
	    CY (c), gameoverbuffer, 1)
    end for

    STORE_RESTORE_SCREEN (154, 224, 434, 264, true)
    loop
	PUTXY (22, 15, yellow, "HOW ABOUT ANOTHER GAME? (y/n): ")
	getch (key)
	exit when index ("yYnN", key) > 0
    end loop
    put key

    STORE_RESTORE_SCREEN (154, 224, 434, 264, false)
end GAME_OVER



%                   * * * * * * *
%                   | MAIN LINE |
%                   * * * * * * *

loop
    SETUP

    loop
	SWALLOW_KEYS

	GET_COLOURS
	PAINT_COLUMN (snextx, snexty, blkcolour)    % Draws next piece to fall

	SCORE_BOARD

	x := ceil (xblks / 2)
	y := yblks
	scrolled := false
	dropped := false

	exit when blkmatrix (x, y) not= background
			    % Game will end when no room for new piece to fall

	loop
	    if not scrolled then
		SCROLL
	    end if

	    clock (start)
	    loop
		clock (finish)
		exit when finish - start >= fallspeed
		if hasch then
		    getch (key)

		    if (key = left) or (key = right) then
			HORIZONTAL_MOVE
		    elsif (key = enter) or (key = up) then
			CYCLE
		    elsif key = down then
			if y > top (x) then
			    MOVE_DOWN
			    start := finish
			end if
			if not scrolled then
			    SCROLL
			end if
		    elsif key = spacebar then
			dropped := true
		    elsif key = "s" or key = "S" then
			soundon := not soundon
			if soundon then
			    PUTXY (76, 27, magenta, "on ")
			else
			    PUTXY (76, 27, magenta, "off")
			end if
		    elsif key = Esc then
			PAUSE
			exit when endgame
		    elsif key = "h" or key = "H" then
			HELP_ME
		    end if
		end if
		exit when dropped
	    end loop

	    exit when dropped
	    exit when endgame
	    exit when y <= top (x)

	    MOVE_DOWN
	end loop
	exit when endgame

	if scrolled then
	    PAINT_COLUMN (snextx, snexty, nblkcolour)
	end if

	if dropped then
	    DROP
	end if

	top (x) := top (x) + numblks

	for c : 0 .. numblks - 1
	    blkmatrix (x, y + c) := blkcolour (c + 1)
			    % Stores colour values of current piece in matrix
	end for

	if not bomb then
	    matchfound := false
	    COLUMN_CHECK (x, y)
	else
	    BOMB
	end if

	loop
	    exit when not matchfound
	    matchfound := false
	    ERASE_MATCHES
	    lastx := 0
	    GRAVITY (matchx (totmatch), matchy (totmatch))
	end loop

	exit when top (x) - 1 > yblks
    end loop

    GAME_OVER
    exit when key = "n" or key = "N"  % Player has chosen No for another game
end loop

var sysvar : int
system ("DEL ERASE.ME", sysvar)        % Erases image swapping file ERASE.ME
cls
