/*
 *  users.c
 *
 *  Copyright (C) 2001, 2002  Gustavo Picon
 */

/*
 * this file is heavily based on eggdrop's users/flags
 * functions, by Robey Pointer and the Eggheads
 */

#include "aircdll.h"
#include "users.h"



static usernetw_t usernetstart  = NULL;	/* network tables */
static usernetw_t usernauxstart = NULL;	/* temporary network header that points to... */
static usermain_t userstart     = NULL;	/* temporary userlist list head */
static useribeihead_t *useribeistart;	/* ignore, ban, exempt, invite */


/*
 *
 * Internal functions
 *
 *
 */



/*
 * users_init()
 */
void users_init ()
{
	/* creating a blank network entry and assign the network header to this entry */
	usernetstart = users_netw_find("-", TRUE);
}

usernetw_t users_netw_find (char *netw, tbool create)
{
	hash_t hh = rfchash(netw);
	usernetw_t t, p = NULL;

	for (t = usernetstart; t; t = t->next) {
		if ( t->hh == hh && !rfc_cmp(netw, t->netw) ) {
			return t;
		}
		p = t;
	}
	if (create) {
		t = (usernetw_t) smalloc(sizeof(struct usernetw_s));
		t->hh = hh;
		t->netw = cmalloc(lstrlen(netw) + 1);
		lstrcpy(t->netw, netw);
		t->next = NULL;
		t->users = NULL;
		users_ibei_init(t);
		if (p) {
			p->next = t;
		} else {
			usernetstart = t;
		}
		return t;
	}
	return NULL;
}

/*
 * return value is _always_ > 0
 */
unsigned int users_netw_count ()
{
	unsigned int i = 0;
	usernetw_t t;

	for (t = usernetstart; t; t = t->next) {
		++i;
	}
	return i;
}

void users_netw_active (usernetw_t netw)
{
	usernauxstart = netw;

	if (netw) {
        userstart = netw->users;
		useribeistart = netw->ibei;
	} else {
		userstart = NULL;
		useribeistart = NULL;
	}
}

void users_netw_remove (usernetw_t netw)
{
	users_netw_active(netw);
	users_user_cleanup();
	mfree(netw->netw);
	mfree(netw->users);
	mfree(netw);
}

int users_netw_del (char *ntw)
{
	hash_t hh;
	usernetw_t t, p = NULL;

	if (!ntw || !*ntw)
		return 1;
	hh = rfchash(ntw);
	for (t = usernetstart; t; t = t->next) {
		if ( t->hh == hh && !rfc_cmp(ntw, t->netw) ) {
			if (t == usernetstart) {
				usernetstart = t->next;
			} else if (p) {
				p->next = t->next;
			}
			users_netw_remove(t);
			return 0;
		}
		p = t;
	}
	return 2;
}

int users_netw_cleanup (tbool exit)
{
	int r = 0;
	usernetw_t u = usernetstart, n;

	while (u) {
		n = u->next;
		users_netw_remove(u);
		u = n;
		++r;
	}

	usernetstart = NULL;
	if (!exit) {
		users_init(); /* there must _always_ be a global entry */
		/* --r; what the hell, let's count this one anyway*/
	}

	return r;
}

usermain_t users_user_find (char *usr, tbool create)
{
	hash_t hh = rfchash(usr);
	usermain_t t, p = NULL;

	for (t = userstart; t; t = t->next) {
		if ( t->hh == hh && !rfc_cmp(usr, t->handle) ) {
			return t;
		}
		p = t;
	}
	if (create) {
		t = (usermain_t) smalloc(sizeof(struct usermain_s));
		t->hh = hh;
		t->handle = cmalloc(lstrlen(usr) + 1);
		lstrcpy(t->handle, usr);
		t->next = NULL;
		t->flags = NULL;
		t->uflags = NULL;
		t->chans = NULL;
		t->entries = NULL;
		if (p) {
			p->next = t;
		} else {
			userstart = t;
			usernauxstart->users = t;
		}
		return t;
	}
	return NULL;
}

unsigned int users_user_count ()
{
	unsigned int i = 0;
	usermain_t t;

	for (t = userstart; t; t = t->next) {
		++i;
	}
	return i;
}

char *users_flags_nodup (char *s)
{
	char *t, *b;

	t = s;
	b = cmalloc(lstrlen(t) + 1);
	lstrcpy(b, t);
	*t = 0;
	while (*b) {
		if (*b != ' ' && *b != '+' && *b != '-' && *b != '\n' && !isincs(s, *b)) {
			*t++ = *b;
			*t   = 0;
		}
		b++;
	}
	mfree(b);
	return s;

}

userchan_t users_chan_find (usermain_t user, char *chan, tbool create)
{
	userchan_t t, p = NULL;

	for (t = user->chans; t; t = t->next) {
		if (!rfc_cmp(chan, t->channel)) {
			return t;
		}
		p = t;
	}
	if (create) {
		t = (userchan_t) smalloc(sizeof(struct userchan_s));
		t->channel = cmalloc(lstrlen(chan) + 1);
		lstrcpy(t->channel, chan);
		t->flags = NULL;
		t->info = NULL;
		t->uflags = NULL;
		t->next = NULL;
		t->laston = 0;
		if (p) {
			p->next = t;
		} else {
			user->chans = t;
		}
		return t;
	}
	return NULL;
}

void users_chan_remove (userchan_t chan)
{
	mfree(chan->flags);
	mfree(chan->channel);
	mfree(chan->uflags);
	mfree(chan->info);
	mfree(chan);
}

void users_entry_remove (userentry_t entry)
{
	stringchain_t c, n;

	mfree(entry->name);
	if (entry->type == USERENTRY_HOSTS || entry->type == USERENTRY_XTRA) {
		c = entry->list;
		while (c) {
			n = c->next;
			mfree(c->data);
			c = n;
		}
	} else {
		mfree(entry->cont);
	}
	mfree(entry);
}

void users_user_remove (usermain_t user)
{
	userchan_t cchan, nchan;
	userentry_t centry, nentry;

	mfree(user->handle);
	mfree(user->flags);
	mfree(user->uflags);
	cchan = user->chans;
	while (cchan) {
		nchan = cchan->next;
		users_chan_remove(cchan);
		cchan = nchan;
	}
	centry = user->entries;
	while (centry) {
		nentry = centry->next;
		users_entry_remove(centry);
		centry = nentry;
	}
	mfree(user);
}

int users_user_cleanup ()
{
	int r = 0;
	usermain_t u = userstart, n;

	while (u) {
		n = u->next;
		users_user_remove(u);
		u = n;
		++r;
	}
	usernauxstart->users = NULL;
	userstart = NULL;
	return r;
}

int users_user_add (char *usr)
{
	if (!*usr) {
		return 1;
	}
	if (users_user_find(usr, FALSE)) {
		return 2;
	}
	users_user_find(usr, TRUE);
	return 0;
}

int users_user_del (char *usr)
{
	hash_t hh;
	usermain_t t, p = NULL;

	if (!*usr) {
		return 1;
	}

	hh = rfchash(usr);
	for (t = userstart; t; t = t->next) {
		if ( t->hh == hh && !rfc_cmp(usr, t->handle) ) {
			if (t == userstart) {
				userstart = t->next;
				usernauxstart->users = t->next;
			} else if (p) {
				p->next = t->next;
			}
			users_user_remove(t);
			return 0;
		}
		p = t;
	}
	return 2;
}

int users_chrec_add (usermain_t user, char *chan)
{
	if (!chan || !*chan) {
		return 1;  /* Invalid channel */
	}
	if (users_chan_find(user, chan, FALSE)) {
		return 2;  /* Channel is already there */
	}
	users_chan_find(user, chan, TRUE);
	return 0;
}

int users_chrec_del (usermain_t user, char *chan)
{
	userchan_t t, p = NULL;

	if (!chan || !*chan) {
		return 1;  /* Invalid channel */
	}
	for (t = user->chans; t; t = t->next) {
		if (!rfc_cmp(chan, t->channel)) {
			if (t == user->chans) {
				user->chans = t->next;
			} else if (p) {
				p->next = t->next;
			}
			users_chan_remove(t);
			return 0;
		}
		p = t;
	}
	return 2; /* No such channel record for the user */
}

/*
 * type:
 *       0 = global
 *       1 = channel
 */

int users_flags_set (
				   usermain_t user,
				   char *flags,
				   int type,
				   char *chan
				   )
{
	userchan_t uchan;

	users_flags_nodup(flags);
	if (type == 0) {
		mfree(user->flags);
		user->flags = cmalloc(lstrlen(flags) + 1);
		lstrcpy(user->flags, flags);
		return 0;
	} else {
		uchan = users_chan_find(user, chan, TRUE);
		if (uchan) {
			mfree(uchan->flags);
			uchan->flags = cmalloc(lstrlen(flags) + 1);
			lstrcpy(uchan->flags, flags);
			return 0;
		}
		return 1; /* no such channel record for the user */
	}
}

/*
 *type:
 *       0 = global
 *       1 = channel
 */
char *users_flags_get (usermain_t user, int type, char *chan)
{
	userchan_t uchan;

	if (type == 0)
		return user->flags;
	uchan = users_chan_find(user, chan, FALSE);
	if (uchan)
		return uchan->flags;
	return NULL;
}

int users_entry_type2num (char *t)
{
	if (!lstrcmpi(t, "COMMENT"))
		return USERENTRY_COMMENT;
	if (!lstrcmpi(t, "INFO"))
		return USERENTRY_INFO;
	if (!lstrcmpi(t, "LASTON"))
		return USERENTRY_LASTON;
	if (!lstrcmpi(t, "PASS"))
		return USERENTRY_PASS;
	if (!lstrcmpi(t, "BOTADDR"))
		return USERENTRY_BOTADDR;
	if (!lstrcmpi(t, "BOTFL"))
		return USERENTRY_BOTFL;
	if (!lstrcmpi(t, "HOSTS"))
		return USERENTRY_HOSTS;
	if (!lstrcmpi(t, "XTRA"))
		return USERENTRY_XTRA;
	if (!lstrcmpi(t, "CONSOLE"))
		return USERENTRY_CONSOLE;
	if (!lstrcmpi(t, "DCCDIR"))
		return USERENTRY_DCCDIR;
	if (!lstrcmpi(t, "FWD"))
		return USERENTRY_FWD;
	if (!lstrcmpi(t, "FSTAT"))
		return USERENTRY_FSTAT;

	return -1;
}

tbool users_entry_find (
					 usermain_t umUser,
					 char *type,
					 tbool create
					 )
{
	userentry_t ue, p = NULL;
	int ntype = users_entry_type2num(type);

	for (ue = umUser->entries; ue; ue = ue->next) {
		if (ue->name && !lstrcmpi(type, ue->name)) {
			if (p) {
				p->next = ue->next;
				ue->next = umUser->entries;
			}
			umUser->entries = ue;
			return TRUE;
		}
		p = ue;
	}
	if (create) {
		ue = (userentry_t) smalloc(sizeof(struct userentry_s));
		ue->next = umUser->entries;
		umUser->entries = ue;
		ue->name = cmalloc(lstrlen(type) + 1);
		lstrcpy(ue->name, type);
		ue->type = ntype;
		if (ntype == USERENTRY_HOSTS || ntype == USERENTRY_XTRA)
			ue->list = NULL;
		else
			ue->cont = NULL;
		return TRUE;
	}
	return FALSE;
}
int users_entry_set (
				   usermain_t umUser,
				   char *type,
				   char *str
				   )
{
	int ntype = users_entry_type2num(type);
	userentry_t ue;
	stringchain_t add, curr, prev = NULL;

	if (!umUser)
		return 1;  /* "USER", "No such user" */
	if (ntype == -1)
		return 2;  /* "TYPE", "Invalid entry type" */
	if ((ntype == USERENTRY_HOSTS || ntype == USERENTRY_XTRA) && !*str)
		return 3;  /* "DATA", "Invalid data" */
	if (ntype == USERENTRY_HOSTS && !ishostmask(str))
		return 3;  /* "DATA", "Invalid data" */
	if (ntype == USERENTRY_HOSTS && users_uhost2handle_equalnomatch(str))
		return 4;  /* "DUPHOST", "That hostmask is already assigned to a user." */
	users_entry_find(umUser, type, TRUE);
	ue = umUser->entries;
	if (ntype == USERENTRY_HOSTS || ntype == USERENTRY_XTRA) {
		for (curr = ue->list; curr; curr = curr->next) {
			if (!rfc_cmp(str, curr->data)) {
				lstrcpy(curr->data, str);
				return 0; /* OK! */
			}
			prev = curr;
		}
		add = (stringchain_t) smalloc(sizeof(struct stringchain_s));
		if (prev)
			prev->next = add;
		else
			ue->list = add;
		add->next = NULL;
		add->data = cmalloc(lstrlen(str) + 1);
		lstrcpy(add->data, str);
	}
	else {
		mfree(ue->cont);
		ue->cont = cmalloc(lstrlen(str) + 1);
		lstrcpy(ue->cont, str);
	}
	return 0;
}
int users_entry_del_w (
					usermain_t umUser,
					char *type,
					char *first
					)
{
	int ntype;
	userentry_t ue;
	stringchain_t sc, p = NULL;
	char *auxd, auxs[MAXSTRINGLEN];

	if (!umUser)
		return 1;  /* r_err("USER", "No such user"); */
	ntype = users_entry_type2num(type);
	if (ntype == USERENTRY_HOSTS || ntype == USERENTRY_XTRA) {
		if (!*first)
			return 2;  /* r_err("WORD", "Invalid word"); */
	} else {
		return 3;  /* r_err("TYPE", "Invalid entry type"); */
	}
	if (users_entry_find(umUser, type, FALSE)) {
		ue = umUser->entries;
		auxd = auxs;
		for (sc = ue->list; sc; sc = sc->next) {
			lstrcpy(auxd, sc->data);
			if (!lstrcmpi(getword(&auxd), first)) {
				if (sc == ue->list)
					ue->list = sc->next;
				else if (p)
					p->next = sc->next;
				mfree(sc->data);
				mfree(sc);
				return 0; /* ok! */
			}
			p = sc;
		}
	}
	return 4; /* r_err("MATCH", "No such entry") */
}

int users_laston_set (usermain_t umUser, char *place, unsigned int when)
{
	userchan_t uc;
    unsigned int num;
	char aux[MAXSTRINGLEN];

    if (!umUser) {
		return 1;
    }
    if (!place || !*place) {
		return 2;
    }
    num = when?when:now();
	wsprintf(aux, "%d %s", num, place);
	if (isincs("!#&+", *place)) {
		uc = users_chan_find(umUser, place, TRUE);
        if (uc) {
			uc->laston = num;
        }
	}
	users_entry_set(umUser, "LASTON", aux);
	return 0;
}

char *users_flags_fix (char *f)
{
	if (f)
		return f;
	return "-";
}

usermain_t users_uhost2handle (char *address)
{
	usermain_t umUser, umBest = NULL;
	int best = 0, tmp;
	userentry_t ue;
	stringchain_t scHosts;

	for (umUser = userstart; umUser; umUser = umUser->next) {
		for (ue = umUser->entries; ue; ue = ue->next) {
			if (ue->type == USERENTRY_HOSTS) {
				for (scHosts = ue->list; scHosts; scHosts = scHosts->next) {
					tmp = match(scHosts->data, address, FALSE);
					if (tmp > best) {
						best = tmp;
						umBest = umUser;
					}
				}
				break;
			}
		}
	}
	if (umBest)
		return umBest;
	return NULL;
}

usermain_t users_uhost2handle_equalnomatch (char *address)
{
	usermain_t umUser;
	userentry_t  ue;
	stringchain_t scHosts;

	for (umUser = userstart; umUser; umUser = umUser->next) {
		for (ue = umUser->entries; ue; ue = ue->next) {
			if (ue->type == USERENTRY_HOSTS) {
				for (scHosts = ue->list; scHosts; scHosts = scHosts->next) {
					if (!rfc_cmp(scHosts->data, address))
						return umUser;
				}
				break;
			}
		}
	}
	return NULL;
}

int users_ibei_char2id (char c)
{
	switch (c) {
		case 'i':
			return USERIBEI_IGNORE;
		case 'b':
			return USERIBEI_BAN;
		case 'e':
			return USERIBEI_EXEMPT;
		case 'I':
			return USERIBEI_INVITE;
		default:
			return -1;
	}
}

void users_ibei_init (usernetw_t t)
{
	register int i;

	for (i = 0; i <= 3; ++i) {
		t->ibei[i] = (useribeihead_t) smalloc(sizeof(struct useribeihead_s));
		t->ibei[i]->chan = NULL;
		t->ibei[i]->entry = NULL;
		t->ibei[i]->hash = 0;
		t->ibei[i]->next = NULL;
		t->ibei[i]->num = 0;
	}
}

unsigned int users_ibei_expire (usernetw_t t)
{
    unsigned int i, currtime, r;
    useribeihead_t h;
    useribeientry_t e, p, a;

    currtime = now();
    r = 0;

    for (i = 0; i <= 3; ++i) {
        for (h = t->ibei[i]; h; h = h->next) {
            e = h->entry;
            p = NULL;
            while (e) {
                if (e->expire > currtime) {
                    if (h->entry == e) {
                        h->entry = e->next;
                    } else if (p) {
                        p->next = e->next;
                    }
                    a = e->next;
                    users_ibei_delnode(e);
                    e = a;
                    ++r;
                } else {
                    p = e;
                    e = e->next;
                }
            }
        }
    }
    return r;
}

void users_ibei_chan_delnode (useribeihead_t t)
{
	mfree(t->chan);
	mfree(t);
}
unsigned int users_ibei_chan_count (int ntype)
{
	useribeihead_t t;
	unsigned int c = 0;

	for (t = useribeistart[ntype]; t; t = t->next) {
		++c;
	}
	return c;
}

unsigned int users_ibei_count (useribeihead_t h)
{
	useribeientry_t t;
	unsigned int c = 0;

	for (t = h->entry; t; t = t->next) {
		++c;
	}
	return c;
}

void users_ibei_delnode (useribeientry_t t)
{
	mfree(t->mask);
	mfree(t->msg);
	mfree(t->setby);
	mfree(t);
}

void users_ibei_formatline (char *line, int ntype, useribeientry_t uie)
{
	if (ntype == USERIBEI_IGNORE) {
		wsprintf(line, "- %s:%s%lu:%s:%lu:%s", uie->mask, uie->perm?"+":"", uie->expire,
			uie->setby?uie->setby:"somebody", uie->seton, uie->msg?uie->msg:"");
	} else {
		wsprintf(line, "%s %s:%s%lu%s:+%lu:%lu:%s:%s",
			ntype == USERIBEI_EXEMPT ? "%" : ntype == USERIBEI_INVITE ? "@" : "-",
			uie->mask, uie->perm?"+":"", uie->expire, uie->sticky?"*":"",
			uie->seton, uie->lastactive, uie->setby?uie->setby:"somebody",
			uie->msg?uie->msg:"requested");
	}
}

useribeihead_t users_ibei_chan_find (int ntype, char *chan, tbool create, tbool remove)
{
	useribeihead_t  t = NULL, prev = NULL;
	hash_t hash = 0;

	if (!chan)
		return useribeistart[ntype]; /* global entry */

	hash = rfchash(chan);
	for (t = useribeistart[ntype]; t; t = t->next) {
		if (t->hash == hash && !rfc_cmp(t->chan, chan)) {
			if (remove) {
				if (prev)
					prev->next = t->next;
				else
					useribeistart[ntype] = t->next;
					users_ibei_chan_delnode(t);
				return NULL;
			}
			return t;
		}
		prev = t;
	}

	if (create) {
		t = (useribeihead_t) smalloc(sizeof(struct useribeihead_s));
		t->num   = 0;
		t->next  = NULL;
		t->entry = NULL;
		t->hash  = hash;
		t->chan  = cmalloc(lstrlen(chan) + 1);
		lstrcpy(t->chan, chan);
		prev->next = t; /* there is _always_ a previous node */
		return t;
	}
	return NULL;
}

useribeientry_t users_ibei_find (
					int ntype, char *chan, char *mask, char *setby, char *msg,
					unsigned int seton, unsigned int expire,
                    unsigned int lactive, tbool perm, tbool sticky,
					tbool create, tbool remove
					)
{
	useribeihead_t ihf;
	useribeientry_t t, prev = NULL;
	hash_t hash = 0;
	tbool ecreated = FALSE;
    unsigned int currtime;

	ihf = users_ibei_chan_find(ntype, chan, create, remove);
    currtime = now();

	if (ihf == NULL)
		return NULL;

	/* we found the type/channel node, now let's handle the entry itself */

	hash = rfchash(mask);
	prev = NULL;
	for (t = ihf->entry; t; t = t->next) {

		if (t->hash == hash && !rfc_cmp(t->mask, mask)) {
			if (remove) {
				if (prev)
					prev->next = t->next;
				else
					ihf->entry = NULL;
				(ihf->num)--;
				users_ibei_delnode(t);
				return NULL;
			}
			if (!create) {
				return t;
			}
			ecreated = TRUE;
			break;
		}
		prev = t;
	}

	if (create) {
		if (ecreated) {
			mfree(t->msg);
			mfree(t->setby);
		} else {
			t = (useribeientry_t) smalloc(sizeof(struct useribeientry_s));
			t->mask = cmalloc(lstrlen(mask) + 1);
			lstrcpy(t->mask, mask);
		}
		t->setby = cmalloc(lstrlen(setby) + 1);
		lstrcpy(t->setby, setby);
		t->msg = cmalloc(lstrlen(msg) + 1);
		lstrcpy(t->msg, msg);
		t->expire = expire;
		t->hash   = hash;
		t->next   = NULL;
		t->perm   = perm;
		t->seton  = seton;
        t->lastactive = lactive;
		t->sticky = sticky;
		if (!ecreated) {
			if (prev)
				prev->next = t;
			else
				ihf->entry = t;
		}
		(ihf->num)++;
		return t;
	}

	return NULL;
}

useribeientry_t users_ibei_match (int ntype, char *chan, char *address)
{
	useribeihead_t ihf;
	useribeientry_t t, ubest = NULL;
	int tmp, best = 0;
    unsigned int currtime;

	ihf = users_ibei_chan_find(ntype, chan, (tbool)NULL, (tbool)NULL);
    currtime = now();

	if (ihf == NULL)
		return NULL;

	for (t = ihf->entry; t; t = t->next) {
	    tmp = match(t->mask, address, FALSE);
	    if (tmp > best) {
   			best = tmp;
		    ubest = t;
	    }
	}
	if (ubest)
		return ubest;
	return NULL;
}

int users_ibei_lineadd (char type, char *chan, char *line)
{
	char *mask, *chexpire, *setby, *chseton, *msg, *chlactive;
	unsigned int seton, expire, lactive;
	tbool perm = FALSE, sticky = FALSE;
	int ntype, tmp;

	ntype = users_ibei_char2id(type);

	if (ntype == -1)
		return -1; /* unknown type */

	if (countchar(line, ':') < (unsigned int)((ntype == USERIBEI_IGNORE) ? 4:5) )
		return -2; /* not enough ':' characters */

	mask = getnexttok(&line, ':');
	chexpire = getnexttok(&line, ':');
	if (*chexpire == '+') {
		perm = TRUE;
		chexpire++;
	}
	if (ntype == USERIBEI_IGNORE) {
		setby   = getnexttok(&line, ':');
		chseton = getnexttok(&line, ':');
		msg     = getnexttok(&line, ':');
        lactive = 0;
	} else {

		tmp = lstrlen(chexpire) - 1;
		if (chexpire[tmp] == '*') {
			chexpire[tmp] = 0;
			sticky = TRUE;
		}

		chseton   = getnexttok(&line, ':');
		chlactive = getnexttok(&line, ':');
		setby     = getnexttok(&line, ':');
		msg       = getnexttok(&line, ':');

        lactive = atolp(chlactive);

		if (*chseton == '+')
			chseton++;
	}

	seton   = atolp(chseton);
	expire  = atolp(chexpire);

	users_ibei_find(ntype, chan, mask, setby, msg, seton, expire, lactive,
        perm, sticky, TRUE, FALSE);

	return 0;
}


/*
 *
 * Exported
 *
 */



/*
 * UserNetwAdd <Network>
 */
MircFunc UserNetwAdd (FUNCPARMS)
{
	char *tmp, *ntw;

	tmp = data;
	ntw = getword(&tmp);
	if (!ntw || !*ntw || !rfc_cmp(ntw, "-")) {
		r_err("NETW", "Invalid Network");
	}
	users_netw_find(ntw, TRUE);
	r_ok("");
}

/*
 * UserNetwDel <Network>
 */
MircFunc UserNetwDel (FUNCPARMS)
{
	char *tmp, *ntw;

	tmp = data;
	ntw = getword(&tmp);

	if (!ntw || !*ntw || !rfc_cmp(ntw, "-"))
		r_err("NETW", "Invalid Network");

	if (!users_netw_del(ntw)) {
	        r_ok("");
	} else {
		r_err("FOUND", "Network setting not found");
	}
}

/*
 * UserNetwGet <Network|N>
 */
MircFunc UserNetwGet (FUNCPARMS)
{
	char *tmp, *ntw;
	int num = 0, i = 0;
	tbool checknum = FALSE;
	usernetw_t naux;

	tmp = data;
	ntw = getword(&tmp);

	if (isnum(ntw, 0)) {
		num = atolp(ntw);
		if (!num)
			r_err("NUMBER", "Invalid number");
		checknum = TRUE;
	}
	for (naux = usernetstart; naux; naux = naux->next) {
		if ((checknum && ++i == num) || (!checknum && !rfc_cmp(ntw, naux->netw)))
			r_ok(naux->netw);
	}
	r_err("FOUND", "Entry not found");
}

/*
 * UserNetwCount
 *   return value is _always_ > 0
 */
MircFunc UserNetwCount (FUNCPARMS)
{
	wsprintf(data, "+OK %d", users_netw_count());
	return 3;
}

/*
 * UserNetwClear
 */
MircFunc UserNetwClear (FUNCPARMS)
{
	wsprintf(data, "+OK %d", users_netw_cleanup(FALSE));
	return 3;
}

/*
 * UserAdd <Network|-> <Handle>
 */
MircFunc UserAdd (FUNCPARMS)
{
	char *tmp;
	int result;

	getnet(tmp);

	result = users_user_add(getword(&tmp));

	if (result == 1)
		r_err("HANDLE", "Invalid Handle");
	if (result == 2)
		r_err("DUP", "That handle is already in the list");
	r_ok("");
}

/*
 * UserDel <Network|-> <Handle>
 */
MircFunc UserDel (FUNCPARMS)
{
	char *tmp;
	int result;

	getnet(tmp);

	result = users_user_del(getword(&tmp));

	if (result == 1)
		r_err("HANDLE", "Invalid Handle");
	if (result == 2)
		r_err("NOTFOUND", "Handle not found in the list");
	r_ok("");
}

/*
 * UserRen <Network|-> <oldHandle> <Handle>
 */
MircFunc UserRen (FUNCPARMS)
{
	char *tmp, *oldhandle, *newhandle;
	usermain_t olduser, newuser;

	getnet(tmp);

	oldhandle = getword(&tmp);
	newhandle = getword(&tmp);

	if (!*oldhandle)
		r_err("OLD", "Invalid old handle")
	if (!*newhandle)
		r_err("NEW", "Invalid new handle")
	olduser = users_user_find(oldhandle, FALSE);
	newuser = users_user_find(newhandle, FALSE);
	if (!olduser)
		r_err("NOTFOUND", "Handle not found in the list");
	if (newuser)
		r_err("DUP", "There is already a user with the new handle");
	mfree(olduser->handle);
	olduser->handle = cmalloc(lstrlen(newhandle) + 1);
	lstrcpy(olduser->handle, newhandle);
	r_ok("");
}

/*
 * UserCount <Network|->
 */
MircFunc UserCount (FUNCPARMS)
{
	char *tmp;

	getnet(tmp);

	wsprintf(data, "+OK %d", users_user_count());
	return 3;
}

/*
 * UserGet <Network|-> <N>
 */
MircFunc UserGet (FUNCPARMS)
{
	char *tmp, *ntmp;
	unsigned int index = 0, count = 0;
	usermain_t t;

	getnet(tmp);

	ntmp = getword(&tmp);

	if (!*ntmp) {
		index = 1;
	} else if (!isnum(ntmp, 0)) {
		r_err("NUM", "Invalid Index Number");
	} else {
		index = atolp(ntmp);
	}

	for (t = userstart; t; t = t->next) {
		if (++count == index) {
			wsprintf(data, "+OK %s", t->handle);
			return 3;
		}
	}
	r_err("NOUSER", "No user for that index");
}

/*
 * UserIs <Network|-> <Handle>
 */
MircFunc UserIs (FUNCPARMS)
{
	char *tmp;

	getnet(tmp);

	r_ok(users_user_find(getword(&tmp), FALSE)?"1":"0");
}

/*
 * UserChrecAdd <Network|-> <Handle> <Channel>
 */
MircFunc UserChrecAdd (FUNCPARMS)
{
	char *tmp;
	int result;
	usermain_t user;

	getnet(tmp);

	user = users_user_find(getword(&tmp), FALSE);

	if (!user)
		r_err("USER", "No such user");
	result = users_chrec_add(user, getword(&tmp));
	if (result == 1)
		r_err("CHAN", "Invalid Channel");
	if (result == 2)
		r_err("DUP", "Duplicated channel record for the user");
	r_ok("");
}

/*
 * UserChrecDel <Network|-> <Handle> <Channel>
 */
MircFunc UserChrecDel (FUNCPARMS)
{
	char *tmp;
	int result;
	usermain_t user;

	getnet(tmp);

	user = users_user_find(getword(&tmp), FALSE);

	if (!user)
		r_err("USER", "No such user");
	result = users_chrec_del(user, getword(&tmp));
	if (result == 1)
		r_err("CHAN", "Invalid Channel");
	if (result == 2)
		r_err("CHREC", "No such channel record for the user");
	r_ok("");
}

/*
 * UserChrecGet <Network|-> <Handle> <N>
 */
MircFunc UserChrecGet (FUNCPARMS)
{
	char *tmp, *ntmp;
	int count = 0, index;
	usermain_t user;
	userchan_t t;

	getnet(tmp);

	user = users_user_find(getword(&tmp), FALSE);
	ntmp = getword(&tmp);

	if (!user) {
		r_err("USER", "No such user");
	}
	if (!*ntmp) {
		index = 1;
	} else if (!isnum(ntmp, 0)) {
		r_err("NUM", "Invalid Index Number");
	} else {
		index = atolp(ntmp);
	}
	for (t = user->chans; t; t = t->next) {
		if (++count == index) {
			r_ok(t->channel);
		}
	}
	if (index == 0) {
		wsprintf(data, "+OK %d", count);
		return 3;
	}
	r_err("NOCHAN", "No channel for that Index");
}

/*
 * UserChrecIs <Network|-> <Handle> <Channel>
 */
MircFunc UserChrecIs (FUNCPARMS)
{
	char *tmp, *ntmp;
	usermain_t user;

	getnet(tmp);

	user = users_user_find(getword(&tmp), FALSE);
	ntmp = getword(&tmp);

	if (!user) {
		r_err("USER", "No such user");
	}
	if (!*ntmp) {
		r_err("CHAN", "Invalid Channel");
	}
	r_ok(users_chan_find(user, ntmp, FALSE)?"1":"0");
}

/*
 * UserFlagsSet <Network|-> <handle> <+[flags]> [chan]
 */
MircFunc UserFlagsSet (FUNCPARMS)
{
	char *tmp, *handle, *flags, *chan;
	int type = 0, result;
	usermain_t user;

	getnet(tmp);

	handle = getword(&tmp);
	flags = getword(&tmp);
	chan = getword(&tmp);
	user = users_user_find(handle, FALSE);

	if (!user) {
		r_err("USER", "No such user");
	}
	if (!*flags || *flags != '+') {
		r_err("FLAGS", "Invalid Flags");
	}
	if (*chan) {
		type = 1;
	}
	result = users_flags_set(user, flags, type, chan);
	if (result == 1) {
		r_err("CHAN", "Invalid Channel");
	}
	r_ok("");
}

/*
 * UserFlagsGet <Network|-> <handle> [chan]
 */
MircFunc UserFlagsGet (FUNCPARMS)
{
	char *tmp, *aux, *handle, *chan;
	int type = 0;
	usermain_t user;

	getnet(tmp);

	aux = NULL;
	handle = getword(&tmp);
	chan = getword(&tmp);
	user = users_user_find(handle, FALSE);

	if (!user) {
		r_err("USER", "No such user");
	}
	if (*chan) {
		if (!users_chan_find(user, chan, FALSE)) {
			r_err("CHAN", "No such channel record for the user");
		}
		type = 1;
	}
	aux = users_flags_get(user, type, chan);
	if (!aux) {
		aux = "";
	}
	wsprintf(data, "+OK %s", aux);
	return 3;
}

/*
 * UserFlagsGets <Network|-> <handle> [chan]
 */
MircFunc UserFlagsGets (FUNCPARMS)
{
	char *tmp, *aux, *handle, *chan;
	usermain_t user;

	getnet(tmp);

	aux = NULL;
	handle = getword(&tmp);
	chan = getword(&tmp);
	user = users_user_find(handle, FALSE);

	if (!user)
		r_err("USER", "No such user");
	if (*chan)
		aux = users_flags_get(user, 1, chan);
	if (!aux)
		aux = users_flags_get(user, 0, NULL);
	if (!aux)
		aux = "";
	wsprintf(data, "+OK %s", aux);
	return 3;
}

/*
 * UserFlagsGetd <Network|-> <handle> [chan]
 */
MircFunc UserFlagsGetd (FUNCPARMS)
{
	char *tmp, *handle, *chan, *result, gflags[666], cflags[666];
	usermain_t user;

	getnet(tmp);

	handle = getword(&tmp);
	chan = getword(&tmp);
	gflags[0] = cflags[0] = 0;
	user = users_user_find(handle, FALSE);
	if (!user)
		r_err("USER", "No such user");
	result = cmalloc(666);
	if (*chan)
		lstrcpy(cflags, users_flags_get(user, 1, chan));
	lstrcpy(gflags, users_flags_get(user, 0, NULL));
	if (isincs(cflags, 'o'))
		removechar(gflags, 'd');
	if (isincs(cflags, 'd') || isincs(cflags, 'k'))
		removechar(removechar(gflags, 'o'), 'f');
	if (isincs(cflags, 'q'))
		removechar(gflags, 'v');
	lstrcpy(result, gflags);
	lstrcat(result, cflags);
	users_flags_nodup(result);
	wsprintf(data, "+OK %s", result);
	mfree(result);
	return 3;
}

/*
 * UserEntrySet <Network|-> <handle> <type> <string>
 */
MircFunc UserEntrySet (FUNCPARMS)
{
	char *tmp, *handle, *type, *str;
	int result;
	usermain_t umUser;

	getnet(tmp);

	handle = getword(&tmp);
	type = getword(&tmp);
	str = tmp;
	umUser = users_user_find(handle, FALSE);

	result = users_entry_set(umUser, type, str);
	if (result == 1)
		r_err("USER", "No such user");
	if (result == 2)
		r_err("TYPE", "Invalid entry type");
	if (result == 3)
		r_err("DATA", "Invalid data");
	if (result == 4)
		r_err("DUPHOST", "That hostmask is already assigned to a user.");
	r_ok("");
}

/*
 * UserEntryDel <Network|-> <handle> <type [N]>
 */
MircFunc UserEntryDel (FUNCPARMS)
{
	char *tmp, *handle, *type, *first;
	int ntype, num = 0, mch = 0;
	usermain_t umUser;
	userentry_t ue, pue = NULL;

	getnet(tmp);

	handle = getword(&tmp);
	type = getword(&tmp);
	first = getword(&tmp);
	umUser = users_user_find(handle, FALSE);

	if (!umUser) {
		r_err("USER", "No such user");
	}
	ntype = users_entry_type2num(type);
	if (ntype == -1)
		r_err("TYPE", "Invalid entry type");
	if (ntype == USERENTRY_HOSTS || ntype == USERENTRY_XTRA) {
		if (!isnum(first, 0))
			r_err("NUMBER", "Invalid number");
		mch = atolp(first);
	}
	for (ue = umUser->entries; ue; ue = ue->next) {
		if (ue->name && !lstrcmpi(type, ue->name)) {
			if (ntype == USERENTRY_HOSTS || ntype == USERENTRY_XTRA) {
				stringchain_t sc, psc = NULL;
				for (sc = ue->list; sc; sc = sc->next) {
					if (++num == mch) {
						if (sc == ue->list)
							ue->list = sc->next;
						else if (psc)
							psc->next = sc->next;
						mfree(sc->data);
						mfree(sc);
						r_ok("");
					}
					psc = sc;
				}
				r_err("MATCH", "No such entry");
			} else {
				if (ue == umUser->entries)
					umUser->entries = ue->next;
				else if (pue)
					pue->next = ue->next;
				mfree(ue->cont);
				mfree(ue->name);
				mfree(ue);
				r_ok("");
			}
		}
		pue = ue;
	}
	r_err("MATCH", "No such entry");
}

/*
 * UserEntryGet <Network|-> <handle> <type [N]>
 */
MircFunc UserEntryGet (FUNCPARMS)
{
	char *tmp, *handle, *type, *first;
	int ntype, num = 0, mch = 0;
	usermain_t umUser;
	userentry_t ue;

	getnet(tmp);

	handle = getword(&tmp);
	type = getword(&tmp);
	first = getword(&tmp);
	umUser = users_user_find(handle, FALSE);

	if (!umUser) {
		r_err("USER", "No such user");
	}
	ntype = users_entry_type2num(type);
	if (ntype == -1) {
		r_err("TYPE", "Invalid entry type");
	}
	if (ntype == USERENTRY_HOSTS || ntype == USERENTRY_XTRA) {
		if (!isnum(first, 0))
			r_err("NUMBER", "Invalid number");
		mch = atolp(first);
	}
	if (users_entry_find(umUser, type, FALSE)) {
		ue = umUser->entries;
		if (ntype == USERENTRY_HOSTS || ntype == USERENTRY_XTRA) {
			stringchain_t sc;
			for (sc = ue->list; sc; sc = sc->next) {
				if (++num == mch)
					r_ok(sc->data);
			}
			if (mch == 0) {
				wsprintf(data, "+OK %d", num);
				return 3;
			}
			r_err("MATCH", "No such entry");
		} else {
			r_ok(ue->cont);
		}
	}
	r_err("MATCH", "No such entry");
}

/*
 * UserEntrySetW <Network|-> <handle> <"HOSTS"|"XTRA"> <word> <string>
 */
MircFunc UserEntrySetW (FUNCPARMS)
{
	char *tmp, *handle, *type, *first, *str, auxs[1024];
	int result, ntype;
	usermain_t umUser;

	getnet(tmp);

	handle = getword(&tmp);
	type = getword(&tmp);
	lstrcpy(auxs, tmp);
	first = getword(&tmp);
	str = tmp;
	ntype = users_entry_type2num(type);
	umUser = users_user_find(handle, FALSE);

	if (ntype == USERENTRY_HOSTS && users_uhost2handle_equalnomatch(auxs))
		r_err("DUPHOST", "That hostmask is already assigned to a user.");

	result = users_entry_del_w(umUser, type, first);
	if (result == 1)
		r_err("USER", "No such user");
	if (result == 2)
		r_err("WORD", "Invalid word");
	if (result == 3)
		r_err("TYPE", "Invalid entry type");
/*	if (result == 4)
 *		r_err("MATCH", "No such entry");
 */

	result = users_entry_set(umUser, type, auxs);

/*	if (result == 1)
 *		r_err("USER", "No such user");
 *	if (result == 2)
 *		r_err("TYPE", "Invalid entry type");
 */
	if (result == 3)
		r_err("DATA", "Invalid data");
	if (result == 4)
		r_err("DUPHOST", "That hostmask is already assigned to a user.");

	r_ok("");
}

/*
 * UserEntryDelW <Network|-> <handle> <"HOSTS"|"XTRA"> <word>
 */
MircFunc UserEntryDelW (FUNCPARMS)
{
	char *tmp, *handle, *type, *first;
	int result;
	usermain_t umUser;

	getnet(tmp);

	handle = getword(&tmp);
	type = getword(&tmp);
	first = getword(&tmp);
	umUser = users_user_find(handle, FALSE);

	result = users_entry_del_w(umUser, type, first);
	if (result == 1)
		r_err("USER", "No such user");
	if (result == 2)
		r_err("WORD", "Invalid word");
	if (result == 3)
		r_err("TYPE", "Invalid entry type");
	if (result == 4)
		r_err("MATCH", "No such entry");
	r_ok("");
}

/*
 * UserEntryGetW <Network|-> <handle> <"HOSTS"|"XTRA"> <word>
 */
MircFunc UserEntryGetW (FUNCPARMS)
{
	char *tmp, *handle, *type, *first, *auxd, auxs[MAXSTRINGLEN];
	int ntype;
	usermain_t umUser;
	userentry_t ue;
	stringchain_t sc;

	getnet(tmp);

	handle = getword(&tmp);
	type = getword(&tmp);
	first = getword(&tmp);
	umUser = users_user_find(handle, FALSE);

	if (!umUser)
		r_err("USER", "No such user");
	ntype = users_entry_type2num(type);
	if (ntype == USERENTRY_HOSTS || ntype == USERENTRY_XTRA) {
		if (!*first)
			r_err("WORD", "Invalid word");
	} else {
		r_err("TYPE", "Invalid entry type");
	}
	if (users_entry_find(umUser, type, FALSE)) {
		ue = umUser->entries;
		auxd = auxs;
		for (sc = ue->list; sc; sc = sc->next) {
			lstrcpy(auxd, sc->data);
			if (!lstrcmpi(getword(&auxd), first)) {
                if (ntype == USERENTRY_XTRA) {
                    r_ok(auxd);
                } else {
                    r_ok(sc->data);
                }
			}
		}
	}
	r_err("MATCH", "No such entry");
}

/*
 * UserHost2Flags <Network|-> <address> [channel]
 */
MircFunc UserHost2Flags (FUNCPARMS)
{
	char *tmp, *addy, *chan, *aux;
	usermain_t umUser;

	getnet(tmp);

	addy = getword(&tmp);
	chan = getword(&tmp);
	aux = NULL;
	umUser = users_uhost2handle(addy);

	if (umUser) {
		if (*chan) {
			aux = users_flags_get(umUser, 1, chan);
		}
		if (!aux)
			aux = users_flags_get(umUser, 0, NULL);
		if (!aux)
			aux = "";
		wsprintf(data, "+OK %s %s", umUser->handle, aux);
		return 3;
	}
	r_err("MATCH", "There is no user matching that address");
}

/*
 * UserHost2FlagsD <Network|-> <address> [channel]
 */
MircFunc UserHost2FlagsD (FUNCPARMS)
{
	char *tmp, *addy, *chan, *aux, gflags[666], cflags[666];
	usermain_t umUser;

	getnet(tmp);

	addy = getword(&tmp);
	chan = getword(&tmp);
	aux = NULL;
	umUser = users_uhost2handle(addy);

	gflags[0] = cflags[0] = 0;
	if (umUser) {
		if (*chan) {
			aux = users_flags_get(umUser, 1, chan);
			if (aux)
				lstrcpy(cflags, aux);
		}
		aux = users_flags_get(umUser, 0, NULL);
		if (aux)
			lstrcpy(gflags, aux);
		if (isincs(cflags, 'o'))
			removechar(gflags, 'd');
		if (isincs(cflags, 'd') || isincs(cflags, 'k'))
			removechar(removechar(gflags, 'o'), 'f');
		if (isincs(cflags, 'q'))
			removechar(gflags, 'v');
		aux = cmalloc(666);
		lstrcpy(aux, gflags);
		lstrcat(aux, cflags);
		users_flags_nodup(aux);
		wsprintf(data, "+OK %s %s", umUser->handle, aux);
		mfree(aux);
		return 3;
	}
	r_err("MATCH", "There is no user matching that address");
}

/*
 * UserHost2Handle <Network|-> <address>
 */
MircFunc UserHost2Handle (FUNCPARMS)
{
	char *tmp;
	usermain_t um;

	getnet(tmp);

	um = users_uhost2handle(getword(&tmp));

	if (um) {
		r_ok(um->handle);
	}
	r_err("MATCH", "There is no user matching that address");
}

/*
 * UserLastonSet <Network|-> <handle> <where>
 */
MircFunc UserLastonSet (FUNCPARMS)
{
	char *tmp, *handle, *place, *when;
	usermain_t umUser;
	int result;
    unsigned int num;

	getnet(tmp);

	handle = getword(&tmp);
	place = getword(&tmp);
	when = getword(&tmp);
    if (*when) {
        if (!isnum(when, 0)) {
            r_err("WHEN", "Invalid ctime number");
        }
        num = atolp(when);
    } else {
        num = 0;
    }
	umUser = users_user_find(handle, FALSE);
	result = users_laston_set(umUser, place, num);

	if (result == 1)
		r_err("USER", "No such user");
	if (result == 2)
		r_err("WHERE", "Invalid location");
	r_ok("");
}

/*
 * UserLastonGet <Network|-> <handle> [channel]
 */
MircFunc UserLastonGet (FUNCPARMS)
{
	char *tmp, *chan;
	usermain_t user;
	userchan_t uchan;

	getnet(tmp);

	user = users_user_find(getword(&tmp), FALSE);

	if (!user)
		r_err("USER", "No such user");
	chan = getword(&tmp);
	if (*chan) {
		uchan = users_chan_find(user, chan, FALSE);
		if (uchan && uchan->laston) {
			wsprintf(data, "+OK %d", uchan->laston);
			return 3;
		}
		r_err("CHAN", "No laston record for that channel");
	}
	if (users_entry_find(user, "LASTON", FALSE)) {
		r_ok(user->entries->cont);
	}
	r_err("LASTON", "Unknown laston Info");
}

/*
 * UserSetChanInfo <Network|-> <handle> <channel> <info>
 */
MircFunc UserSetChanInfo (FUNCPARMS)
{
	char *tmp, *chan;
	usermain_t user;
	userchan_t uchan;

	getnet(tmp);

	user = users_user_find(getword(&tmp), FALSE);

	if (!user)
		r_err("USER", "No such user");
	chan = getword(&tmp);
	if (!*chan)
		r_err("CHAN", "Invalid Channel");
	uchan = users_chan_find(user, chan, TRUE);
	uchan->info = cmalloc(lstrlen(tmp) + 1);
	lstrcpy(uchan->info, tmp);
	r_ok("")
}

/*
 * UserGetChanInfo <Network|-> <handle> <channel>
 */
MircFunc UserGetChanInfo (FUNCPARMS)
{
	char *tmp, *chan;
	usermain_t user;
	userchan_t uchan;

	getnet(tmp);

	user = users_user_find(getword(&tmp), FALSE);

	if (!user)
		r_err("USER", "No such user");
	chan = getword(&tmp);
	if (!*chan)
		r_err("CHAN", "Invalid Channel");
	uchan = users_chan_find(user, chan, TRUE);
	r_ok(uchan->info ? uchan->info : "")
}

/*
 * UserDelChanInfo <Network|-> <handle> <channel>
 */
MircFunc UserDelChanInfo (FUNCPARMS)
{
	char *tmp, *chan, *info;
	usermain_t user;
	userchan_t uchan;

	getnet(tmp);

	user = users_user_find(getword(&tmp), FALSE);

	if (!user)
		r_err("USER", "No such user");
	chan = getword(&tmp);
	if (!*chan)
		r_err("CHAN", "Invalid Channel");
	info = tmp;
	uchan = users_chan_find(user, chan, TRUE);
	if (uchan->info)
		mfree(uchan->info);
	uchan->info = NULL;
	r_ok("")
}

/*
 * UserLoad <Network|-> <file>
 */
MircFunc UserLoad (FUNCPARMS)
{
	char *tmp, *auxl, *auxb, *code, *pass, *attr, *s, *lasthand, *chname, *st, *fl;
 	char buff[4096 + 1], line[MAXSTRINGLEN], aux[4096]; /* what the hell, i'm lazy */
	usermain_t u = NULL;
	HANDLE hfile = NULL;
	DWORD  bytesread;
	unsigned int wpos = 0, nline = 0, nusers = 0;
	register int i;
	userchan_t uc;

	getnet(tmp);

	lasthand = cmalloc(1024);

	if (myfileopen(&hfile, data, tmp, FALSE))
		return 3;

	users_user_cleanup();
	auxl = line;
	do {
		for (i = 0; i < 4096 + 1; ++i)
			buff[i] = 0;
		if (ReadFile(hfile, buff, 4096, &bytesread, NULL)) {
			for (auxb = buff, i = 0; *auxb && i < 4906 + 1; auxb++, ++i) {
				if (++wpos == (MAXSTRINGLEN - 4) || *auxb == 10) {
					*auxl = 0;
					lstrcpy(aux, line);
					if (++nline > 1) {
						s = aux;
						if (s[0] && s[0] != '#' && s[0] != ';') {
							code = getword(&s);
							if (!lstrcmp(code, "-")) {
								; /* global bans and ignores */
								if (!lasthand[0] || !s[0])
									continue;		/* Skip this entry.	*/

								if (lasthand[0] && strchr("#&!+", lasthand[0]) != NULL) {
									users_ibei_lineadd('b', lasthand, s);

								} else if (lasthand[0] == '*') {
									if (lasthand[1] == 'i') {
										users_ibei_lineadd('i', NULL, s);
									} else {
										users_ibei_lineadd('b', NULL, s);
									}
								}

							} else if (!lstrcmp(code, "%")) {
								; /* exemptmasks */
                                if (!lasthand[0] || !s[0])
									continue;		/* Skip this entry.	*/
								if (lasthand[0] && strchr("#&!+", lasthand[0]) != NULL) {
									users_ibei_lineadd('e', lasthand, s);
								} else if (lasthand[0] == '*' && lasthand[1] == 'e') {
									users_ibei_lineadd('e', NULL, s);
								}
							} else if (!lstrcmp(code, "@")) {
								; /* Invitemasks */

								if (!lasthand[0] || !s[0])
									continue;		/* Skip this entry.	*/

								if (lasthand[0] && strchr("#&!+", lasthand[0]) != NULL) {
									users_ibei_lineadd('I', lasthand, s);
								} else if (lasthand[0] == '*' && lasthand[1] == 'I') {
									users_ibei_lineadd('I', NULL, s);
								}
							} else if (!lstrcmp(code, "!")) {
								/* ! #chan laston flags [info] */
								if (u) {
									chname = getword(&s);
									st = getword(&s);
									fl = users_flags_nodup(getword(&s));
									uc = users_chan_find(u, chname, TRUE);
									uc->flags = cmalloc(lstrlen(fl) + 1);
									lstrcpy(uc->flags, fl);
									uc->laston = atolp(st);
									uc->info = cmalloc(lstrlen(s) + 1);
									lstrcpy(uc->info, s);
								}
							} else if (!mystrcmp(code, "::")) {
								/* channel-specific bans */
								lstrcpy(lasthand, &code[2]);
								u = NULL;
							} else if (!mystrcmp(code, "&&")) {
								/* channel-specific exempts */
								lstrcpy(lasthand, &code[2]);
								u = NULL;
							} else if (!mystrcmp(code, "$$")) {
								/* channel-specific invites */
								lstrcpy(lasthand, &code[2]);
								u = NULL;
							} else if (!mystrcmp(code, "--")) {
								if (u)
									users_entry_set(u, code + 2, s);
							} else if (!rfc_cmp(code, BAN_NAME)) {
								lstrcpy(lasthand, code);
								u = NULL;
							} else if (!rfc_cmp(code, IGNORE_NAME)) {
								lstrcpy(lasthand, code);
								u = NULL;
							} else if (!rfc_cmp(code, EXEMPT_NAME)) {
								lstrcpy(lasthand, code);
								u = NULL;
							} else if (!rfc_cmp(code, INVITE_NAME)) {
								lstrcpy(lasthand, code);
								u = NULL;
							} else if (code[0] == '*') {
								lasthand[0] = 0;
								u = NULL;
							} else {
								pass = getword(&s); /* pass not used */
								if (!pass || !pass[0] || !s || !s[0]) {
									lasthand[0] = 0;
								} else {
									++nusers;
									lstrcpy(lasthand, code);
                                    attr = users_flags_nodup(getword(&s));
									u = users_user_find(code, TRUE);
									u->flags = cmalloc(lstrlen(attr) + 1);
									lstrcpy(u->flags, attr);
								}
							}
						}
					}
					else if (aux[0] != '#' || aux[1] != '4') {
						CloseHandle(hfile);
						r_err("UFILE", "Invalid Userfile");
					}
					auxl = line;
					wpos = 0;
				} else if (*auxb != 13) {
					*auxl++ = *auxb;
				}
			} /* for (auxb = buff; *auxb; auxb++) { */
		} /* if (ReadFile(hfile, buff, 4096, &bytesread, NULL)) */
	} while (bytesread == 4096);
	mfree(lasthand);
	CloseHandle(hfile);
	wsprintf(data, "+OK %d %d", nusers, nline);
	return 3;
}

/*
 * UserSave <Network|-> <file>
 */
MircFunc UserSave (FUNCPARMS)
{
	char *tmp, line[4096];
	HANDLE hfile = NULL;
	usermain_t u;
	userchan_t ch;
	userentry_t ue;
	useribeihead_t uih;
	useribeientry_t uie;
	stringchain_t sc;
	register int i;

	getnet(tmp);

	if (myfileopen(&hfile, data, tmp, TRUE))
		return 3;

	/* the header */
	wsprintf(line, "#4v: %s -- %s -- written %d", VERSIONL, "aircdll", now());
	file_write(&hfile, line);

	/* adding the users */
	for (u = userstart; u; u = u->next) {
		wsprintf(line, "%-10s - %-24s", u->handle, users_flags_fix(u->flags));
		file_write(&hfile, line);
		for (ch = u->chans; ch; ch = ch->next) {
			wsprintf(line, "! %-20s %lu %-10s %s", ch->channel, ch->laston, users_flags_fix(ch->flags), ch->info ? ch->info : "");
			file_write(&hfile, line);
		}
		for (ue = u->entries; ue; ue = ue->next) {
			if (ue->type == USERENTRY_HOSTS || ue->type == USERENTRY_XTRA) {
				for (sc = ue->list; sc; sc = sc->next) {
					wsprintf(line, "--%s %s", ue->name, sc->data);
					file_write(&hfile, line);
				}
			} else {
				wsprintf(line, "--%s %s", ue->name, ue->cont);
				file_write(&hfile, line);
			}
		}
	}

	/* ibei: ignores, bans, exempts and invites */
	for (i = 0; i <= 3; ++i) {
		file_write(
			&hfile,
			i == USERIBEI_IGNORE ? IGNORE_NAME:
			i == USERIBEI_BAN    ? BAN_NAME:
			i == USERIBEI_EXEMPT ? EXEMPT_NAME:
			INVITE_NAME
			);
		for (uih = useribeistart[i]; uih; uih = uih->next) {
			if (uih->num) {
				if (uih->chan) {
					/* channel header */
					wsprintf(line, "%s%s %s",
						i == USERIBEI_EXEMPT ? "&&" : i == USERIBEI_INVITE ? "$$" : "::", uih->chan,
						i == USERIBEI_BAN ? "bans": i == USERIBEI_EXEMPT ? "exempts": "invites"
						);
					file_write(&hfile, line);
				}
				/* entries */
				for (uie = uih->entry; uie; uie = uie->next) {
					users_ibei_formatline(line, i, uie);
					file_write(&hfile, line);
				}
			}
		}
	}

	CloseHandle(hfile);
	r_ok("");
}


/*
 * UserIbeiAdd <Network|-> +<i|b|e|I>[s] <channel|-> <mask> <creator> <created> <expire> <lastactive> <comment>
 */
MircFunc UserIbeiAdd (FUNCPARMS)
{
	char *tmp, *chan, *flags, *mask, *setby, *chexpire, *chcreated, *chlactive, *msg;
	unsigned int expire, created, lactive;
	int ntype;
	tbool sticky;

	getnet(tmp);
	flags     = getword(&tmp);
	chan      = getword(&tmp);
	mask      = getword(&tmp);
	setby     = getword(&tmp);
	chcreated = getword(&tmp);
	chexpire  = getword(&tmp);
	chlactive = getword(&tmp);
	msg   = tmp;
    sticky = FALSE;

	if (!*chan || !lstrcmp(chan, "-"))
		chan = NULL; /* global */
	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");
	if (flags[2] == 's')
		sticky = TRUE;
	if (!*mask)
		r_err("MASK", "Invalid Mask");
	if (!*setby)
		r_err("CREATOR", "Invalid creator");
	expire = (unsigned int)atolp(chexpire);
	created = (unsigned int)atolp(chcreated);
	lactive = (unsigned int)atolp(chlactive);
	if (!*msg)
		msg = NULL;
    if (!created) {
        created = now();
    }

	users_ibei_find(ntype, chan, mask, setby, msg?msg:"requested", created,
        expire,
        lactive,
		expire?FALSE:TRUE,
        sticky,
        TRUE,
        FALSE);
	r_ok("");
}

/*
 * UserIbeiDel <Network|-> +<i|b|e|I> <channel|-> <mask>
 */
MircFunc UserIbeiDel (FUNCPARMS)
{
	char *tmp, *chan, *flags, *mask;
	int ntype;

	getnet(tmp);
	flags = getword(&tmp);
	chan  = getword(&tmp);
	mask  = getword(&tmp);

	if (!*chan || !lstrcmp(chan, "-"))
		chan = NULL; /* global */
	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");
	if (!*mask)
		r_err("MASK", "Invalid Mask");

	if (users_ibei_find(ntype, chan, mask, NULL, NULL, 0, 0, 0, FALSE, FALSE, FALSE, FALSE)) {
		users_ibei_find(ntype, chan, mask, NULL, NULL, 0, 0, 0, FALSE, FALSE, FALSE, TRUE);
		r_ok("");
	}
	r_err("ERR", "Entry not found");
}

/*
 * UserIbeiGet <Network|-> +<i|b|e|I> <channel|-> <mask>
 */
MircFunc UserIbeiGet (FUNCPARMS)
{
	char *tmp, *chan, *flags, *mask, line[4096];
	int ntype;
	useribeientry_t uie;

	getnet(tmp);
	flags = getword(&tmp);
	chan  = getword(&tmp);
	mask  = getword(&tmp);

	if (!*chan || !lstrcmp(chan, "-"))
		chan = NULL; /* global */
	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");
	if (!*mask)
		r_err("MASK", "Invalid Mask");

	uie = users_ibei_find(ntype, chan, mask, NULL, NULL, 0, 0, 0, FALSE, FALSE, FALSE, FALSE);
	if (uie) {
		users_ibei_formatline(line, ntype, uie);
		line[MAXSTRINGLEN - 5] = 0; /* just to be sure */
		r_ok(&line[2]); /* the first word is a single character, only for files */
	}
	r_err("ERR", "Entry not found");
}

/*
 * UserIbeiGetn <Network|-> +<i|b|e|I> <channel|-> <entry_number>
 */
MircFunc UserIbeiGetn (FUNCPARMS)
{
	char *tmp, *chan, *flags, *ntmp, line[4096];
	int ntype, number, i;
	useribeihead_t ihf;
	useribeientry_t t;

	getnet(tmp);
    lstrcpy(line, tmp);
	flags = getword(&tmp);
	chan  = getword(&tmp);
	ntmp  = getword(&tmp);
    i = 0;

	if (!*chan || !lstrcmp(chan, "-"))
		chan = NULL; /* global */
	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");
	if (!*ntmp || !isnum(ntmp, 0))
		r_err("NUMBER", "Invalid Number");
    number = atolp(ntmp);

    ihf = users_ibei_chan_find(ntype, chan, FALSE, FALSE);
    if (ihf) {
        t = ihf->entry;
	    while (t) {
    		if (++i == number) {
			    break;
		    }
            t = t->next;
    	}
        if (number == 0) {
            wsprintf(data, "+OK %d", i);
            return 3;
        }
	    if (t) {
    		users_ibei_formatline(line, ntype, t);
		    line[MAXSTRINGLEN - 5] = 0; /* just to be sure */
		    r_ok(&line[2]); /* the first word is a single character, only for files */
	    }
    }
	r_err("ERR", "Entry not found");
}

/*
 * UserIbeiMatch <Network|-> +<i|b|e|I> <channel|-> <address>
 */
MircFunc UserIbeiMatch (FUNCPARMS)
{
	char *tmp, *chan, *flags, *addy, line[4096];
	int ntype;
	useribeientry_t uie = NULL;

	getnet(tmp);
	flags = getword(&tmp);
	chan  = getword(&tmp);
	addy  = getword(&tmp);

	if (!*chan || !lstrcmp(chan, "-"))
		chan = NULL; /* global */
	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");
	if (!*addy)
		r_err("ADDRESS", "Invalid Address");

	if (chan)
		uie = users_ibei_match(ntype, chan, addy); /* channel */
	if (!uie)
		uie = users_ibei_match(ntype, NULL, addy); /* global */

	if (uie) {
		users_ibei_formatline(line, ntype, uie);
		line[MAXSTRINGLEN - 5] = 0; /* just to be sure */
		r_ok(&line[2]); /* the first word is a single character, only for files */
	}
	r_err("NOMATCH", "No entry matched the address");
}

/*
 * UserIbeiExpire <Network|->
 */
MircFunc UserIbeiExpire (FUNCPARMS)
{
	char *tmp;

	getnet(tmp);

    wsprintf(data, "+OK %d", users_ibei_expire(usernauxstart));
	return 3;
}

/*
 * UserIbeiCount <Network|-> +<i|b|e|I> <channel|->
 */
MircFunc UserIbeiCount (FUNCPARMS)
{
	char *tmp, *chan, *flags;
	int ntype;
	useribeihead_t uih = NULL;

	getnet(tmp);
	flags = getword(&tmp);
	chan  = getword(&tmp);

	if (!*chan || !lstrcmp(chan, "-"))
		chan = NULL; /* global */
	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");

	uih = users_ibei_chan_find(ntype, chan, FALSE, FALSE);
	if (uih) {
		wsprintf(data, "+OK %d", users_ibei_count(uih));
		return 3;
	}
	r_err("FOUND", "Channel not found");
}

/*
 * UserIbeiChanCount <Network|-> +<i|b|e|I>
 */
MircFunc UserIbeiChanCount (FUNCPARMS)
{
	char *tmp, *flags;
	int ntype;

	getnet(tmp);
	flags = getword(&tmp);

	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");
	wsprintf(data, "+OK %d", users_ibei_chan_count(ntype));
	return 3;
}

/*
 * UserIbeiChanDel <Network|-> +<i|b|e|I> <Channel>
 */
MircFunc UserIbeiChanDel (FUNCPARMS)
{
	char *tmp, *flags, *chan;
	int ntype;

	getnet(tmp);
	flags = getword(&tmp);
	chan  = getword(&tmp);

	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");
	if (!*chan || !lstrcmp(chan, "-"))
		r_err("CHAN", "Invalid channel");
	if ( users_ibei_chan_find(ntype, chan, FALSE, FALSE)) {
		users_ibei_chan_find(ntype, chan, FALSE, TRUE);
		r_ok("");
	}
	r_err("FOUND", "Channel not found");
}

/*
 * UserIbeiChanGet <Network|-> +<i|b|e|I> <Channel|N>
 */
MircFunc UserIbeiChanGet (FUNCPARMS)
{
	char *tmp, *flags, *chan;
	int ntype, num = 0, curr = 0;
	useribeihead_t  t = NULL;
	hash_t hash = 0;
	tbool found = FALSE;

	getnet(tmp);
	flags = getword(&tmp);
	chan  = getword(&tmp);

	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");
	if (!*chan)
		r_err("CHAN", "Invalid channel");
	num = atolp(chan);
	hash = rfchash(chan);

	for (t = useribeistart[ntype]; t; t = t->next) {
		++curr;
		if (num) {
			if (curr == num) {
				found = TRUE;
			} else if (curr > num) {
				break;
			}
		} else if (hash == t->hash && !rfc_cmp(chan, t->chan)) {
			found = TRUE;
		}
		if (found) {
			r_ok(t->chan?t->chan:"-");
		}
	}
	r_err("FOUND", "Channel not found");
}

/*
 * UserIbeiMod <Network|-> +<i|b|e|I> <channel|-> <mask> <field> <value>
 *   fields: creator commend created expire lastactive sticky
 */
MircFunc UserIbeiMod (FUNCPARMS)
{
	char *tmp, *chan, *flags, *mask, *field, *tword, line[4096];
	int ntype;
	useribeientry_t uie;

	getnet(tmp);
	flags  = getword(&tmp);
	chan   = getword(&tmp);
	mask   = getword(&tmp);
	field  = getword(&tmp);

	if (!*chan || !lstrcmp(chan, "-"))
		chan = NULL; /* global */
	if (*flags != '+')
		r_err("FLAGS", "Invalid Flags");
	ntype = users_ibei_char2id(flags[1]);
	if (ntype < 0)
		r_err("TYPE", "Invalid ibeI type");
	if (!*mask)
		r_err("MASK", "Invalid Mask");

	uie = users_ibei_find(ntype, chan, mask, NULL, NULL, 0, 0, 0, FALSE, FALSE,
        FALSE, FALSE);
	if (uie) {
        if (!lstrcmp(field, "creator")) {
            tword = getword(&tmp);
            if (!*tword) {
                r_err("CREATOR", "Invalid creator");
            }
            mfree(uie->setby);
		    uie->setby = cmalloc(lstrlen(tword) + 1);
		    lstrcpy(uie->setby, tword);
            r_ok("");
        } else if (!lstrcmp(field, "comment")) {
            if (!*tmp) {
                lstrcpy(line, "requested");
            } else {
                lstrcpy(line, tmp);
            }
            mfree(uie->msg);
		    uie->msg = cmalloc(lstrlen(line) + 1);
		    lstrcpy(uie->msg, line);
            r_ok("");
        } else if (!lstrcmp(field, "sticky")) {
            tword = getword(&tmp);
            uie->sticky = atolp(tword)?TRUE:FALSE;
            r_ok("");

        } else if (!lstrcmp(field, "created") || !lstrcmp(field, "expire")
            || !lstrcmp(field, "lastactive")) {
            tword = getword(&tmp);
            if (!isnum(tword, 0)) {
                wsprintf(line, "Invalid %s value", field);
                r_err(field, line);
            }
            if (*field == 'c') {
                uie->seton = atolp(tword);
            } else if (*field == 'e') {
                uie->expire = atolp(tword);
            } else {
                uie->lastactive = atolp(tword);
            }
            r_ok("");
        } else {
            r_err("FIELD", "Unknown field for an ibeI entry");
        }
	}
	r_err("ERR", "Entry not found");
}

/*
 * UserCopy <<Network|-> <handle>> <<Network|-> <handle>> <+[o]>
 */
MircFunc UserCopy (FUNCPARMS)
{
    char *tmp, *strls, *strus, *strlt, *strut, *cpyop, aux[4096];
    usernetw_t    ns, nt;
    usermain_t    us, ut;
    userchan_t    cs, ct;
    userentry_t   es, et;
    stringchain_t ls, lt, la;
    tbool copy, add;

    tmp = data;
    strls = getword(&tmp);
    strus = getword(&tmp);
    strlt = getword(&tmp);
    strut = getword(&tmp);
    cpyop = getword(&tmp);

	if (!rfc_cmp(strls, "-")) {
        ns = usernetstart;
    } else {
		ns = users_netw_find(strls, FALSE);
        if (!ns) {
            r_err("NETW1", "Invalid Source Network");
        }
	}
    users_netw_active(ns);
    us = users_user_find(strus, FALSE);
    if (!us) {
        r_err("USER1", "Invalid Source User");
    }

	if (!rfc_cmp(strlt, "-")) {
        nt = usernetstart;
    } else {
		nt = users_netw_find(strlt, FALSE);
        if (!nt) {
            r_err("NETW2", "Invalid Target Network");
        }
	}
    users_netw_active(nt);
    ut = users_user_find(strut, TRUE);
    if (!ut) {
        r_err("USER2", "Invalid Target User");
    }

    if (ns == nt && us == ut) {
        r_err("SAME", "The source and target netw/user are the same");
    }
    copy = isincs(cpyop, 'o');

    if (copy && !us->flags) {
        mfree(ut->flags);
        ut->flags = NULL;
    } else if (us->flags && (copy || !ut->flags)) {
		mfree(ut->flags);
		ut->flags = cmalloc(lstrlen(us->flags) + 1);
		lstrcpy(ut->flags, us->flags);
    }

    if (copy && !us->uflags) {
        mfree(ut->uflags);
        ut->uflags = NULL;
    } else if (us->uflags && (copy || !ut->uflags)) {
		mfree(ut->uflags);
		ut->uflags = cmalloc(lstrlen(us->uflags) + 1);
		lstrcpy(ut->uflags, us->uflags);
    }

    for (cs = us->chans; cs; cs = cs->next) {
        ct = users_chan_find(ut, cs->channel, TRUE);
        if (copy && !cs->flags) {
            mfree(ct->flags);
            ct->flags = NULL;
        } else if (cs->flags && (copy || !ct->flags)) {
            mfree(ct->flags);
            ct->flags = cmalloc(lstrlen(cs->flags) + 1);
            lstrcpy(ct->flags, cs->flags);
        }
        if (copy && !cs->uflags) {
            mfree(ct->uflags);
            ct->uflags = NULL;
        } else if (cs->uflags && (copy || !ct->uflags)) {
            mfree(ct->uflags);
            ct->uflags = cmalloc(lstrlen(cs->uflags) + 1);
            lstrcpy(ct->uflags, cs->uflags);
        }
        if (copy && !cs->info) {
            mfree(ct->info);
            ct->info = NULL;
        } else if (cs->info && (copy || !ct->info)) {
            mfree(ct->info);
            ct->info = cmalloc(lstrlen(cs->info) + 1);
            lstrcpy(ct->info, cs->info);
        }
        if (copy && !cs->laston) {
            ct->laston = 0;
        } else if (cs->laston && (copy || !ct->laston)) {
            ct->laston = cs->laston;
        }
    }

    for (es = us->entries; es; es = es->next) {
        users_entry_find(ut, es->name, TRUE);
        et = ut->entries;
        wsprintf(aux, "tipo: %d - hosts: %d - xtra: %d", es->type, USERENTRY_HOSTS, USERENTRY_XTRA);
        if (es->type == USERENTRY_HOSTS || es->type == USERENTRY_XTRA) {
            if (copy) {
                lt = et->list;
                while (lt) {
                    la = lt;
                    lt = lt->next;
                    mfree(la->data);
                    mfree(la);
                }
                et->list = NULL;
            }
            for (ls = es->list; ls; ls = ls->next) {
                add = TRUE;
                lt = et->list;
                while (lt) {
                    if (!rfc_cmp(ls->data, lt->data)) {
                        add = FALSE;
                        break;
                    }
                    lt = lt->next;
                }
                if (add) {
                    lt = (stringchain_t) smalloc(sizeof(struct stringchain_s));
                    lt->data = cmalloc(lstrlen(ls->data) + 1);
                    lstrcpy(lt->data, ls->data);
                    lt->next = et->list;
                    et->list = lt;
                }
            }
        } else {
            if (copy && !es->cont) {
                mfree(et->cont);
                et->cont = NULL;
            } else if (es->cont && (copy || !et->cont)) {
                mfree(et->cont);
                et->cont = cmalloc(lstrlen(es->cont) + 1);
                lstrcpy(et->cont, es->cont);
            }
        }
    }

    r_ok("");
}
