/*
 * tabolib - tabo's crappy C library
 * Copyright (C) 2001, 2002 Gustavo Picon
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#define TABOLIBVERSION 20020630

#include <windows.h>
#include "tabolib.h"

#ifdef __cplusplus
extern "C" {
#endif

LPVOID tabolibheap = NULL;

/* taken from the hybrid-5.3p7 source */
unsigned char tolowertab[] = { 0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xa, 0xb, 0xc, 0xd, 0xe, 0xf, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, ' ', '!', '"', '#', '$', '%', '&', 0x27, '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|',  '}', '~', '_',  '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|',  '}', '~', 0x7f, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff };
unsigned char touppertab[] = { 0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xa, 0xb, 0xc, 0xd, 0xe, 0xf, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, ' ', '!', '"', '#', '$', '%', '&', 0x27, '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', 0x5f, '`', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', 0x7f, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff };


/**
 * tabolib_onload - assigns the process heap to tabolibheap
 */
void tabolib_onload ()
{
	tabolibheap = GetProcessHeap();
}


/**
 * cmalloc - allocates memory to hold a string
 * @size: the length of the string
 *
 * must be freed later with mfree
 *
 * returns a char pointer to the allocated memory
 * returns NULL if there is not enough memory
 */
char *cmalloc (int size)
{
	return (char *) HeapAlloc(tabolibheap, HEAP_ZERO_MEMORY, sizeof(char) * size);
}

/**
 * smalloc - allocates memory
 * @size: the length of the string

 * must be freed later with mfree
 *
 * returns a pointer to the allocated memory
 * returns NULL if there is not enough memory
 */
void *smalloc (int size)
{
	return HeapAlloc(tabolibheap, HEAP_ZERO_MEMORY, size);
}

/*
 * mfree(memblock)
 *   deallocates a memory block (memblock) that was previously allocated
 *   by a call to cmalloc or smalloc
 */
tbool mfree (void *pmem)
{
	if (pmem)
		return (tbool)HeapFree(tabolibheap, 0, pmem);
	return TRUE;
}


/*
 * getword(&string)
 *   returns a pointer to a buffer containing the first word
 *   in "string", the pointer to "string" will be moved to
 *   the next word
 */
char *getword (char ** p)
{
	return getnexttok(p, ' ');
}

/*
 * getnexttok(&string, sep)
 *   returns a pointer to the next token (separated by the "sep" char)
 *   in "string", the pointer to "string" will be moved to
 *   the next token, based On eggdrop's newsplit
 *   
 */
char *getnexttok (char **p, int c)
{
	register char *r, *o;
	if (!p)
		return "";
	o = *p;
	while (*o == c)
		o++;
	r = o;
	while (*o && *o != c)
		o++;
	if (*o) {
		*o++ = 0;
		while (*o == c)
			o++;
	}
	*p = o;
	return r;
}


/*
 * atolp(string)
 *   returns a _positive_ number extracted from the string
 *   if the string wasn't a number a 0 will be returned
 */
int atolp (char *s)
{
	register int n = 0;
	while (*s >= '0' && *s <= '9') {
		n *= 10;
		n += (*s++ - '0');
	}
	return n;
}

/*
 * isnum(string, option)
 *   returns TRUE if string is a valid number, if option is > 0,
 *   it'll take negative numbers as valid numbers
 *   returns FALSE in an invalid string
 */
tbool isnum (char *p, int f)
{
	int c = 0;
	if (*p == '-') {
		if (f)
			return FALSE;
		++p;
	}
	if (!f && *p == '-')
		return FALSE;
	while (*p) {
		if (*p < '0' || *p > '9')
			return FALSE;
		++c;
		++p;
	}
	return c?TRUE:FALSE;
}

/*
 * isincs(string, letter)
 *   returns TRUE if letter is in word (case sensitive)
 *   otherwise, a FALSE is returned
 */
tbool isincs (char *pal, char let)
{
	char *tmp = pal;
	while (*tmp) {
		if (*tmp++ == let)
			return TRUE;
	}
	return FALSE;
}

int findtok (char *str, char *tok, char c)
{
	return 0;
}

/*
 * countchar(string, character)
 */
unsigned int countchar (char *str, int chr)
{
	char *aux;
	unsigned int r = 0;

	for (aux = str; *aux; ++aux) {
		if (*aux == chr)
			++r;
	}
	return r;
}

/*
 * removechar(string, character)
 *   removes all the ocurrences of character from string
 */
char *removechar (char *str, int chr)
{
	char *s1, *s2;
        s1 = s2 = str;
	for (; *s2; ++s2) {
		if (*s2 != chr)
			*s1++ = *s2;
	}
	*s1 = 0;
	return str;
}

/*
 * removelf(string)
 *   Removes all the line feeds from a string
 */
char *removelf (char *str)
{
	return removechar(str, '\n');
}

/*
 * mirchar(string)
 *   removes all the spaces at the beginning of the string and removes "extra" spaces
 *   mirchar("    a   b") -> "a b"
 */
char *mirchar (char *str)
{
	char *s1, *s2;
	tbool begin = TRUE, space = FALSE;
        s1 = s2 = str;
	for (; *s2; ++s2) {
		if (*s2 != ' ') {
			*s1++ = *s2;
			space = FALSE;
			begin = FALSE;
		} else {
			if (!begin) {
				if (!space)
					*s1++ = *s2;
				space = TRUE;
			}
		}
	}
	*s1 = 0;
	return str;
}


/*
 * now()
 *   returns the number of seconds elapsed since
 *   midnight (00:00:00), January 1, 1970
 */
unsigned int now ()
{
#ifdef TABOLIB_USE_TIME
	return (unsigned int)time(NULL);
#else
	SYSTEMTIME t;
	GetSystemTime(&t);
	return (t.wYear - 1970) * 365 * 24 * 3600 + t.wMonth * 30 * 24 * 3600 + t.wDay * 24 * 3600 + t.wHour * 3600 + t.wMinute * 60 + t.wSecond - 1814400;
#endif
}


/*
 * mystrcmp(s, s)
 */
int mystrcmp (char *s1, char *s2)
{
	while (*s2) {
		if (*s1 < *s2)
			return -1;
		if (*s1 > *s2)
			return  1;
		s1++;
		s2++;
	}
	return 0;
}

/*
 * ishostmask(mask)
 */
tbool ishostmask (register char *c)
{
	register tbool nick = FALSE, s1 = FALSE, ident = FALSE, s2 = FALSE, host = FALSE;
	while (*c) {
		if (*c == ' ') {
			return FALSE;
		} else if (*c == '!') {
			if (s1)
				return FALSE;
			s1 = TRUE;
		} else if (*c == '@') {
			if (!s1 || s2)
				return FALSE;
			s2 = TRUE;
		} else if (s1 && s2) {
			if (!host && *c == '.')
				return FALSE;
			host = TRUE;
		} else if (s1) {
			ident = TRUE;
		}
		else {
			nick = TRUE;
		}
		++c;
	}
	if (host) {
		if (*--c == '.')
			return FALSE;
		return TRUE;
	}
	return FALSE;
}

/*
 * The next function is based on eggdrop's wild_match (by robey pointer)
 * I added an option to perform a case sensitive match
 */

#define QUOTE '\\'
#define WILDS '*'
#define WILDP '%'
#define WILDQ '?'
/* #define WILDT '~' */

#define MATCH (match+saved+sofar)
#define NOMATCH 0

/*
 * EGGDROP:   wild_match_per(char *m, char *n)
 * IrcII:     wild_match(char *m, char *n)
 *
 * Features:  Forward, case-insensitive, ?, *, %, ~(optional)
 * Best use:  Generic string matching, such as in IrcII-esque bindings
 */
int match (register char *m, register char *n, tbool cs /* case sensitive */)
{
	char *ma = m, *lsm = 0, *lsn = 0, *lpm = 0, *lpn = 0;
	int match = 1, saved = 0;
	register unsigned int sofar = 0;

#ifdef WILDT
  int space;
#endif

	/* take care of null strings (should never match) */
	if ((m == 0) || (n == 0) || (!*n))
		return NOMATCH;
	/* (!*m) test used to be here, too, but I got rid of it.  After all,
	 * If (!*n) was false, there must be a character in the name (the
	 * second string), so if the mask is empty it is a non-match.  Since
	 * the algorithm handles this correctly without testing for it here
	 * and this shouldn't be called with null masks anyway, it should be
	 * a bit faster this way */

	while (*n) {
		/* Used to test for (!*m) here, but this scheme seems to work better */
#ifdef WILDT
		if (*m == WILDT) {			/* Match >=1 space */
			space = 0;				/* Don't need any spaces */
		do {
			m++;
			space++;
		}							/* Tally 1 more space ... */
		while ((*m == WILDT) || (*m == ' '));	/*  for each space or ~ */
		sofar += space;				/* Each counts as exact */
		while (*n == ' ') {
			n++;
			space--;
		}							/* Do we have enough? */
		if (space <= 0)
			continue;				/* Had enough spaces! */
		}
		/* Do the fallback       */
		else {
#endif
			switch (*m) {
			case 0:
				do
					m--;			/* Search backwards */
				while ((m > ma) && (*m == '?'));	/* For first non-? char */
				if ((m > ma) ? ((*m == '*') && (m[-1] != QUOTE)) : (*m == '*'))
					return MATCH;	/* nonquoted * = match */
				break;
			case WILDP:
				while (*(++m) == WILDP);	/* Zap redundant %s */
				if (*m != WILDS) {		/* Don't both if next=* */
					if (*n != ' ') {	/* WILDS can't match ' ' */
						lpm = m;
						lpn = n;		/* Save '%' fallback spot */
						saved += sofar;
						sofar = 0;		/* And save tally count */
					}
					continue;		/* Done with '%' */
				}
				/* FALL THROUGH */
			case WILDS:
				do
					m++;			/* Zap redundant wilds */
				while ((*m == WILDS) || (*m == WILDP));
				lsm = m;
				lsn = n;
				lpm = 0;		/* Save '*' fallback spot */
				match += (saved + sofar);	/* Save tally count */
				saved = sofar = 0;
				continue;		/* Done with '*' */
			case WILDQ:
				m++;
				n++;
				continue;		/* Match one char */
			case QUOTE:
				m++;			/* Handle quoting */
			}
			if (cs) { /* case sensitive */
				if (*m == *n) {		/* If matching */
					m++;
					n++;
					sofar++;
					continue;		/* Tally the match */
				}
			}
			else if (rfc_toupper(*m) == rfc_toupper(*n)) {		/* If matching */
				m++;
				n++;
				sofar++;
				continue;		/* Tally the match */
			}
#ifdef WILDT
		}
#endif
		if (lpm) {			/* Try to fallback on '%' */
			n = ++lpn;
			m = lpm;
			sofar = 0;		/* Restore position */
			if ((*n | 32) == 32)
				lpm = 0;		/* Can't match 0 or ' ' */
			continue;			/* Next char, please */
		}
		if (lsm) {			/* Try to fallback on '*' */
			n = ++lsn;
			m = lsm;			/* Restore position */
			/* Used to test for (!*n) here but it wasn't necessary so it's gone */
			saved = sofar = 0;
			continue;			/* Next char, please */
		}
		return NOMATCH;		/* No fallbacks=No match */
	}
	while ((*m == WILDS) || (*m == WILDP))
		m++;			/* Zap leftover %s & *s */
	return (*m) ? NOMATCH : MATCH;	/* End of both = match */
}

/*
 * iswm(*mask*, string)
 *    returns TRUE if *mask* matches string
 *    else it returns FALSE
 *    case insensitive
 */
tbool iswm (char *a, char *b)
{
	return match(a, b, FALSE)?TRUE:FALSE;
}

/*
 * iswmcs(*mask*, string)
 *    returns TRUE if *mask* matches string
 *    else it returns FALSE
 *    case sensitive
 */
tbool iswmcs (char *a, char *b)
{
	return match(a, b, TRUE)?TRUE:FALSE;
}

/*
 * myhash(string)
 *   returns a hash value of string
 *
 * This function is based on the hash routine used
 * by MFC's CString
 *
 * very simple, very fast, and good for human-readable strings (like nicks, channels, etc)
 */
hash_t myhash (char *s)
{
	register hash_t i = 0;

	while (*s)
		i = (i<<5) + i + *s++;

	return i;
}
hash_t rfchash (char *s)
{
	register hash_t i = 0;
	register unsigned char *t = (unsigned char *) s;

	while (*t)
		i = (i<<5) + i + tolowertab[*t++];

	return i;
}

/*
 *
 * File related functions
 *
 */

/*
 * isfile(file)
 */
tbool isfile (char *file)
{
	char *result;
	result = file_get(file);
	if (result != NULL)	{
		mfree(result);
		return TRUE;
	}
	return FALSE;
}

/*
 * file_get(file)
 *         returns a char *buffer to the file (if found)
 *                 you must free the buffer after
 *                 you use it
 *         returns NULL if error (no need to "delete")
 */
char *file_get (char *file)
{
	char *result, aux[MAX_PATH];
	DWORD size = sizeof(aux), length = SearchPath(NULL, file, NULL, size, aux, NULL);

	if (length && length <= size && !(FILE_ATTRIBUTE_DIRECTORY & GetFileAttributes(aux))) {
		result = cmalloc(lstrlen(aux) + 1);
		lstrcpy(result, aux);
		return result; /* woohoo! */
	}
	return NULL;	/*
					 * file not found OR file is really a directory
					 * OR path too long
					 */
}

/*
 * file_open_read(file handle, filename)
 *   Opens a file for reading. Returns:
 *   0 - no error
 *   ERR_TABOLIB_BADFILEBUFFER   - invalid file buffer
 *   ERR_TABOLIB_COULDNTOPENFILE - couldn't open file
 *   ERR_TABOLIB_FILENOTFOUND    - file not found or invalid
 *
 *   file handle must be closed later with CloseHandle(handle);
 */
int file_open_read (HANDLE *hfile, char *rawfile)
{
	char *file;

	if (!rawfile || !*rawfile)
		return ERR_TABOLIB_BADFILEBUFFER;
	file = file_get(rawfile);
	if (!file)
		return ERR_TABOLIB_FILENOTFOUND; /* file not found or invalid */
	*hfile = CreateFile(file, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	mfree(file);
	if (*hfile == INVALID_HANDLE_VALUE)
		return ERR_TABOLIB_COULDNTOPENFILE;
	return 0;
}

/*
 * file_open_write(file handle, filename)
 *   Opens a file for writing. Returns:
 *   0 - no error
 *   ERR_TABOLIB_BADFILEBUFFER   - invalid file buffer
 *   ERR_TABOLIB_COULDNTOPENFILE - couldn't open file
 *
 *   file handle must be closed later with CloseHandle(handle);
 */
int file_open_write (HANDLE *hfile, char *file)
{
	if (!file || !*file)
		return ERR_TABOLIB_BADFILEBUFFER;
	*hfile = CreateFile(file, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
	if (*hfile == INVALID_HANDLE_VALUE)
		return ERR_TABOLIB_COULDNTOPENFILE;
	return 0;
}

/*
 * file_write(file handle, text)
 *   Writes a line of text to a file
 *   Returns TRUE on a success
 */
tbool file_write (HANDLE *hfile, char *buff)
{
	char aux[4096];
	DWORD foo;

	/* wsprintf(aux, "%s\n", buff); */
	wsprintf(aux, "%s\r\n", buff); /* CRLF, damn microsoft */
	return (tbool)WriteFile(*hfile, aux, lstrlen(aux), &foo, NULL);
}

void cmdrun (char *appname, char *cmdline)
{
	STARTUPINFO si; 
	PROCESS_INFORMATION ProcessInformation; 

	si.cb = sizeof(STARTUPINFO); 
	si.lpReserved = NULL; 
	si.lpTitle = NULL; 
	si.lpDesktop = "WinSta0\\Default"; 
	si.dwX = si.dwY = si.dwXSize = si.dwYSize = 0L; 
	si.dwFlags = 0;; 
	si.wShowWindow = SW_SHOW; 
	si.lpReserved2 = NULL; 
	si.cbReserved2 = 0; 
	if (CreateProcess(appname, cmdline, NULL, NULL, FALSE, 
					0, NULL, NULL, &si, &ProcessInformation)) 
	{ 
	    CloseHandle(ProcessInformation.hProcess); 
	    CloseHandle(ProcessInformation.hThread); 
	} 
}

char *fixpath (char *file, char *dir, char *out)
{
	wsprintf(out, "%s\\%s", dir, file);
	return out;
}

/*
 * RFC1459 lowercase/uppercase functions
 */

/*
 * taken from the eggdrop bot by Robey Pointer
 */
int rfc_cmp (char *s1, char *s2)
{
	register unsigned char *str1, *str2;
	register int res;

        str1 = (unsigned char *) s1;
        str2 = (unsigned char *) s2;
	while (!(res = touppertab[*str1] - touppertab[*str2])) {
		if (*str1 == '\0')
			return 0;
		str1++;
		str2++;
	}
	return res;
}
int rfc_tolower (int c)
{
  return tolowertab[(unsigned char)(c)];
}
int rfc_toupper (int c)
{
  return touppertab[(unsigned char)(c)];
}
char *rfc_strtoupper (char *s)
{
	register unsigned char *t = (unsigned char *) s;

	while (*t)
		*t = touppertab[*t++];
	return s;
}
char *rfc_strtolower (char *s)
{
	register unsigned char *t = (unsigned char *) s;

	while (*t)
		*t = tolowertab[*t++];
	return s;
}

#ifdef __cplusplus
}
#endif


