/*
 *  flood.c
 *
 *  Copyright (C) 2001, 2002  Gustavo Picon
 */

#include "aircdll.h"
#include "flood.h"

/*
 * Flood stuff
 */

flood_t fldstart = NULL;
floodoptnc_t fldoptstart = NULL;

void flood_checkfree (flood_t p)
{
	floodnum_t aux;
	unsigned int min = ((GetTickCount() / 1000) - p->interval);

	while (p->fnums) {
		if ( (p->fnums)->ticks < min) {
			aux = p->fnums;
			p->fnums = aux->next;
			mfree(aux);
		} else {
			return;
		}
	}
}
/*
 * This function creates a new entry if necessary.
 * The requested node will be moved to the head of the
 * list for speed porpuses.
 */
tbool flood_find (char *string, unsigned int interval, tbool create)
{
	hash_t hash = rfchash(string);
	flood_t t, fldprev = NULL;

	for (t = fldstart ; t; t = t->next ) {
		if ( t->hash == hash && !rfc_cmp(string, t->desc) ) {
			if (create)
				t->interval = interval;
			if (fldprev) {
				fldprev->next = t->next;
				t->next = fldstart;
			}
			fldstart = t;
			return TRUE;
		}
		fldprev = t;
	}
	if (create) {
		t = (flood_t) smalloc(sizeof(struct flood_s));
		t->hash = hash;
		t->interval = interval;
		t->desc = cmalloc(lstrlen(string) + 1);
		lstrcpy(t->desc, string);
		t->fnums = NULL;
		t->next = fldstart;
		fldstart = t;
		return TRUE;
	}
	return FALSE;
}
floodnum_t flood_numnode_create (unsigned int count, unsigned int ticks)
{
	floodnum_t fl;

	fl = (floodnum_t) smalloc(sizeof(struct floodnum_s));
	fl->next = NULL;
	fl->count = count;
	fl->ticks = ticks;
	return fl;
}
unsigned int flood_getcount ()
{
	unsigned int r = 0;
	flood_t t;

	for (t = fldstart ; t; t = t->next)
		++r;
	return r;
}
void flood_numnode_del (floodnum_t t)
{
	floodnum_t a;

	while (t) {
		a = t->next;
		mfree(t);
		t = a;
	}
}
void flood_node_del (flood_t t)
{
	flood_numnode_del(t->fnums);
	mfree(t->desc);
	mfree(t);
}
unsigned int flood_cleanup ()
{
	flood_t t = fldstart, next;
	unsigned int c = 0;

	while (t) {
		next = t->next;
		flood_node_del(t);
		++c;
		t = next;
	}
	fldstart = NULL;
	return c;
}
/*
 * I usually listen to classical music when I program
 * and i have _no_ imagination to name functions
 */
int flood_bach (
				char *data,
				char *desc,
				unsigned int inc,
				unsigned int max,
				unsigned int secs,
				char *flags
				)
{
	unsigned int count = 0, tmpi = 0, cursecs = (unsigned int)(GetTickCount() / 1000);
	flood_t p, aux;
	floodnum_t pfnums;

	if (max  == 0)
		r_err("MAX", "Invalid max value");
	if (secs == 0)
		r_err("SECS", "Invalid secs value");
	flood_find(desc, secs, TRUE);
	p = fldstart;
	flood_checkfree(p);
	count += inc;
	if (!p->fnums) {
		p->fnums = flood_numnode_create(inc, cursecs);
	} else {
		pfnums = p->fnums;
		while (pfnums) {
			count += pfnums->count;
			if (pfnums->next) {
				pfnums = pfnums->next;
			} else {
				pfnums->next = flood_numnode_create(inc, cursecs);
				break;
			}
		}
	}
	tmpi = (cursecs - ((p->fnums)->ticks) );
	if (count >= max && !isincs(flags, 't')) {
		if (!isincs(flags, 'k')) {
			aux = fldstart;
			fldstart = fldstart->next;
			flood_node_del(aux);
		}
		wsprintf(data, "+FLOOD %d %d %d", count, tmpi, max);
	} else {
		wsprintf(data, "+OK %d %d %d", count, tmpi, max);
	}
	return 3;
}
/*
 * Flood <desc> <inc> <max> <secs> [+flags]
 */
MircFunc Flood (FUNCPARMS)
{
	char *tmp         = data;
	char *desc        = getword(&tmp);
	unsigned int inc  = atolp(getword(&tmp));
	unsigned int max  = atolp(getword(&tmp));
	unsigned int secs = atolp(getword(&tmp));
	char *flags       = getword(&tmp);

	if (inc == 0)
		r_err("INC", "Invalid inc value");
	if (max == 0)
		r_err("MAX", "Invalid max value");
	if (secs == 0)
		r_err("SECS", "Invalid secs value");
	if (!*desc)
		r_err("DESC", "Invalid description");
	return flood_bach(data, desc, inc, max, secs, flags); /* returns "+CODE count interval" */
}

/*
 * Flood2 <desc> <inc> <floodtype> [+flags] [network] [#channel]
 */
MircFunc Flood2 (FUNCPARMS)
{
	floodopt_t found   = NULL;
	char *tmp        = data;
	char *desc       = getword(&tmp);
	unsigned int inc = atolp(getword(&tmp));
	char *type       = getword(&tmp);
	char *flags      = getword(&tmp);
	char *netw       = NULL;
	char *chan       = NULL;

	if (isincs(flags, 'n'))
		netw = getword(&tmp);
	if (isincs(flags, 'c'))
		chan = getword(&tmp);
	if (inc == 0)
		r_err("INC", "Invalid inc value");
	if (!*desc)
		r_err("DESC", "Invalid description");
	if (!*type)
		r_err("TYPE", "Invalid Flood Type");
	if (chan && *chan) {
		if (netw && *netw)
			found = flood_opt_find(type, 0, 0, TRUE, netw, chan, FALSE, FALSE);
		if (!found)
			found = flood_opt_find(type, 0, 0, TRUE, NULL, chan, FALSE, FALSE);
	}
	if (!found && netw && *netw)
		found = flood_opt_find(type, 0, 0, TRUE, netw, NULL, FALSE, FALSE);
	if (!found)
		found = flood_opt_find(type, 0, 0, TRUE, NULL, NULL, FALSE, FALSE);
	if (found) {
		/* returns "+CODE count interval" */
		if (found->state)
			return flood_bach(data, desc, inc, found->max, found->secs, flags);
		r_err("OFF", "Flood setting off");
	}
	r_err("ERR", "Flood setting not found");
}

/*
 * FloodExpire
 */
MircFunc FloodExpire(FUNCPARMS)
{
	flood_t t = fldstart, p = NULL, next;
	unsigned int count = 0, total = 0;

	while (t) {
		flood_checkfree(t);
		if (!t->fnums) {
			next = t->next;
			if (t == fldstart)
				fldstart = next;
			else if (p)
				p->next = next;
			flood_node_del(t);
			t = next;
			++count;
		} else {
			p = t;
			++total;
			t = t->next;
		}
	}
	wsprintf(data, "+OK %d %d", count, total);
	return 3; /* returns "removed total" */
}
/*
 * FloodDel <desc>
 */
MircFunc FloodDel (FUNCPARMS)
{
	char *tmp  = data;
	char *desc = getword(&tmp);
	flood_t aux;

	if (!*desc)
		r_err("DESC", "Invalid description");
	if (flood_find(desc, 0, FALSE)) {
		aux = fldstart;
		fldstart = fldstart->next;
		flood_node_del(aux);
		r_ok("");
	} else {
		r_err("NOTFOUND", "Flood entry not found");
	}
}
/*
 * FloodDelMatch <match>
 */
MircFunc FloodDelMatch (FUNCPARMS)
{
	char *tmp  = data;
	char *desc = getword(&tmp);
	flood_t t, p = NULL, aux = NULL;
	unsigned int count = 0, total = 0;

	if (!*desc)
		r_err("MATCH", "Invalid match expression");
	t = fldstart;
	while (t) {
		if (iswm(desc, t->desc)) {
			if (t == fldstart)
				fldstart = fldstart->next;
			else if (p)
				p->next = t->next;
			aux = t;
			t = aux->next;
			flood_node_del(aux);
			++count;
		} else {
			p = t;
			++total;
			t = t->next;
		}
	}
	wsprintf(data, "+OK %d %d", count, total);
	return 3; /* returns "removed total" */
}
/*
 * FloodGet <desc>
 */
MircFunc FloodGet (FUNCPARMS)
{
	char *tmp          = data;
	char *desc         = getword(&tmp);
	unsigned int count = 0;
	flood_t p;
	floodnum_t pfnums;

	if (!*desc)
		r_err("DESC", "Invalid description");

	if (flood_find(desc, 0, FALSE)) {
		p = fldstart;
		flood_checkfree(p);
		pfnums = p->fnums;
		while (pfnums) {
			count += pfnums->count;
			if (pfnums->next)
				pfnums = pfnums->next;
			else
				break;
		}
		wsprintf(data, "+OK %d", count);
		return 3;
	} else {
		r_err("NOTFOUND", "Flood entry not found");
	}
}

/*
 * FloodCount
 */
MircFunc FloodCount (FUNCPARMS)
{
	wsprintf(data, "+OK %d", flood_getcount());
	return 3;
}

/*
 * FloodClear
 */
MircFunc FloodClear (FUNCPARMS)
{
	wsprintf(data, "+OK %d", flood_cleanup());
	return 3;
}

unsigned int flood_opt_nodes_getnum ()
{
	unsigned int c = 0;
	floodoptnc_t nct;
	floodopt_t t;

	for (nct = fldoptstart; nct; nct = nct->next)
		for (t = nct->list; t; t = t->next)
			++c;
	return c;
}
unsigned int flood_opt_cleanup ()
{
	floodoptnc_t nct = fldoptstart, ncn;
	floodopt_t nt    = NULL, nn;
	unsigned int c = 0;

	while (nct) {
		ncn = nct->next;
		nt = nct->list;
		while (nt) {
			nn = nt->next;
			flood_opt_delnode(nt);
			++c;
			nt = nn;
		}
		flood_opt_nc_delnode(nct);
		nct = ncn;
	}
	fldoptstart = NULL;
	return c;
}

floodoptnc_t flood_opt_nc_find (char *netw, char *chan, tbool create, tbool remove)
{
	hash_t hn, hc;
	floodoptnc_t t, prev = NULL;

	if (!netw)
		netw = "";
	if (!chan)
		chan = "";
	hn = rfchash(netw);
	hc = rfchash(chan);
	for (t = fldoptstart; t; t = t->next) {
		if ( hn == t->hn && hc == t->hc && !rfc_cmp(netw, t->netw) && !rfc_cmp(chan, t->chan) ) {
			if (remove) {
				if (prev)
					prev->next = t->next;
				else if (t->next)
					fldoptstart = t->next;
				else
					fldoptstart = NULL;
				flood_opt_nc_delnode(t);
				return NULL;
			}
			return t;
		}
		prev = t;
	}

	if (remove)
		return NULL;

	if (create) {
		t = (floodoptnc_t) smalloc(sizeof(struct floodoptnc_s));
		t->hn = hn;
		t->hc = hc;
		t->netw = cmalloc(lstrlen(netw) + 1);
		lstrcpy(t->netw, netw);
		t->chan = cmalloc(lstrlen(chan) + 1);
		lstrcpy(t->chan, chan);
		if (prev)
			prev->next = t;
		else
			fldoptstart = t;
		t->next = NULL;
		t->list = NULL;
		return t;
	}
	return NULL;
}
floodopt_t flood_opt_find (
					char *type,
					unsigned int max,
					unsigned int secs,
					tbool state,
					char *netw,
					char *chan,
					tbool create,
					tbool remove
					)
{
	hash_t ht = rfchash(type);
	floodopt_t t, prev = NULL;
	floodoptnc_t optnc = flood_opt_nc_find(netw, chan, create, FALSE);

	if (!optnc)
		return NULL;
	for (t = optnc->list; t; t = t->next) {
		if ( ht == t->ht && !rfc_cmp(type, t->type) ) {
			if (remove) {
				if (prev) {
					prev->next = t->next;
				} else if (t->next) {
					optnc->list = t->next;
				} else {
					/* remove the node if there are no entries left */
					flood_opt_nc_find(netw, chan, FALSE, TRUE);
				}
				flood_opt_delnode(t);
				return NULL;
			}
			if (create) {
				t->max = max;
				t->secs = secs;
				t->state = state;
			}
			return t;
		}
		prev = t;
	}
	if (remove)
		return NULL;
	if (create) {
		t = (floodopt_t) smalloc(sizeof(struct floodopt_s));
		t->ht = ht;
		t->type = cmalloc(lstrlen(type) + 1);
		lstrcpy(t->type, type);
		t->max = max;
		t->secs = secs;
		t->state = state;
		if (prev)
			prev->next = t;
		else
			optnc->list = t;
		t->next = NULL;
		return t;
	}
	return NULL;
}
void flood_opt_delnode (floodopt_t t)
{
	mfree(t->type);
	mfree(t);
}
void flood_opt_nc_delnode (floodoptnc_t t)
{
	mfree(t->netw);
	mfree(t->chan);
	mfree(t);
}

int flood_opt_intset (char *buff)
{
	char *tmp  = buff;
	char *type = getword(&tmp);
	char *netw = NULL;
	char *chan = NULL;
	char *tmax = getword(&tmp);
	unsigned int max, secs = atolp(getword(&tmp));
	char *flags  = getword(&tmp);
	tbool state = TRUE;

	if (*flags && *flags != '+')
		return 1;
	if (isincs(flags, 'n'))
		netw = getword(&tmp);
	if (isincs(flags, 'c'))
		chan = getword(&tmp);
	if (isincs(flags, 'd'))
		state = FALSE;
	if (!isnum(tmax, 0))
		return 2;
	max = atolp(tmax);
	if (secs == 0)
		return 3;
	if (!*type)
		return 4;
	flood_opt_find(type, max, secs, state, netw, chan, TRUE, FALSE);
	return 0;
}

/*
 * FloodOptSet <floodtype> <max> <secs> [+ncd] [network] [#channel]
 */
MircFunc FloodOptSet (FUNCPARMS)
{
	int result = flood_opt_intset(data);
	if (result == 1)
		r_err("FLAGS", "Invalid Flags");
	if (result == 2)
		r_err("MAX", "Invalid max value");
	if (result == 3)
		r_err("SECS", "Invalid secs value");
	if (result == 4)
		r_err("TYPE", "Invalid Flood type");
	r_ok("");
}

/*
 * FloodOptDel <floodtype> [+nc] [network] [#channel]
 */
MircFunc FloodOptDel (FUNCPARMS)
{
	char *tmp   = data;
	char *type  = getword(&tmp);
	char *netw  = NULL;
	char *chan  = NULL;
	char *flags = getword(&tmp);

	if (isincs(flags, 'n'))
		netw = getword(&tmp);
	if (isincs(flags, 'c'))
		chan = getword(&tmp);
	if (!*type)
		r_err("TYPE", "Invalid Flood type");

	if (flood_opt_find(type, 0, 0, TRUE, netw, chan, FALSE, FALSE)) {
		flood_opt_find(type, 0, 0, TRUE, netw, chan, FALSE, TRUE);
		r_ok("");
	}
	r_err("ERR", "Flood setting not found");
}

/*
 * FloodOptDelMatch <match> [+nc] [network] [#channel]
 */
MircFunc FloodOptDelMatch (FUNCPARMS)
{
	tbool matchan = FALSE, matchnetw = FALSE;
	unsigned int count = 0;
	char *tmp   = data;
	char *type  = getword(&tmp);
	char *netw  = NULL;
	char *chan  = NULL;
	char *flags = getword(&tmp);
	floodoptnc_t nct, ncp = NULL, ncaux = NULL;
	floodopt_t t = NULL, p = NULL, aux = NULL;

	if (isincs(flags, 'n'))
		netw = getword(&tmp);
	if (isincs(flags, 'c'))
		chan = getword(&tmp);
	if (!*type)
		r_err("TYPE", "Invalid Flood type");
	if (!chan || !*chan)
		matchan = TRUE;
	if (!netw || !*netw)
		matchnetw = TRUE;
	nct = fldoptstart;
	while (nct) {
		if ( (matchan || iswm(chan, nct->chan)) && (matchnetw || iswm(netw, nct->netw)) ) {
			t = nct->list;
			p = NULL;
			aux = NULL;
			while (t) {
				if (iswm(type, t->type)) {
					if (t == nct->list)
						nct->list = (nct->list)->next;
					else if (p)
						p->next = t->next;
					aux = t;
					t = t->next;
					flood_opt_delnode(aux);
					++count;
				} else {
					p = t;
					t = t->next;
				}
			}
		}
		if (!nct->list) {
			if (nct == fldoptstart)
				fldoptstart = fldoptstart->next;
			else if (ncp)
				ncp->next = nct->next;
			ncaux = nct;
			nct = nct->next;
			flood_opt_nc_delnode(ncaux);
		} else {
			ncp = nct;
			nct = nct->next;
		}
	}
	wsprintf(data, "+OK %d", count);
	return 3; /*  returns "removed" */
}

/*
 * FloodOptGet <floodtype> [+nc] [network] [#channel]
 */
MircFunc FloodOptGet (FUNCPARMS)
{
	char *tmp   = data;
	char *type  = getword(&tmp);
	char *netw  = NULL;
	char *chan  = NULL;
	char *flags = getword(&tmp);
	floodopt_t fopt;

	if (isincs(flags, 'n'))
		netw = getword(&tmp);
	if (isincs(flags, 'c'))
		chan = getword(&tmp);
	if (!*type)
		r_err("TYPE", "Invalid Flood type");
	fopt = flood_opt_find(type, 0, 0, TRUE, netw, chan, FALSE, FALSE);
	if (fopt) {
		wsprintf(data, "+OK %d %d %s", fopt->max, fopt->secs, fopt->state?"1":"0");
		return 3;
	}
	r_err("ERR", "Flood setting not found");
}

/*
 * FloodOptGets <floodtype> [+nc] [network] [#channel]
 */
MircFunc FloodOptGets (FUNCPARMS)
{
	char *tmp     = data;
	char *type    = getword(&tmp);
	char *netw    = NULL;
	char *chan    = NULL;
	char *flags   = getword(&tmp);
	floodopt_t fopt = NULL;

	if (isincs(flags, 'n'))
		netw = getword(&tmp);
	if (isincs(flags, 'c'))
		chan = getword(&tmp);
	if (!*type)
		r_err("TYPE", "Invalid Flood type");
	if (chan && netw)
		fopt = flood_opt_find(type, 0, 0, TRUE, netw, chan, FALSE, FALSE);
	if (!fopt && chan)
		fopt = flood_opt_find(type, 0, 0, TRUE, NULL, chan, FALSE, FALSE);
	if (!fopt && netw)
		fopt = flood_opt_find(type, 0, 0, TRUE, netw, NULL, FALSE, FALSE);
        if (!fopt)
		fopt = flood_opt_find(type, 0, 0, TRUE, NULL, NULL, FALSE, FALSE);
	if (fopt) {
		wsprintf(data, "+OK %d %d %s", fopt->max, fopt->secs, fopt->state?"1":"0");
		return 3;
	}
	r_err("ERR", "Flood setting not found");
}

/*
 * FloodOptGetMatch <match> <N> [+nc] [network] [#channel]
 */
MircFunc FloodOptGetMatch (FUNCPARMS)
{
	tbool matchan = FALSE, matchnetw = FALSE;
	unsigned int count = 0, nmatch = 0;
	char *tmp   = data;
	char *type  = getword(&tmp);
	char *netw  = NULL;
	char *chan  = NULL;
	char *ntmp  = getword(&tmp);
	char *flags = getword(&tmp);
	floodoptnc_t nct;
	floodopt_t t;

	if (isincs(flags, 'n'))
		netw = getword(&tmp);
	if (isincs(flags, 'c'))
		chan = getword(&tmp);
	if (!*type)
		r_err("TYPE", "Invalid Flood type");
	if (!*ntmp) {
		nmatch = 1;
	} else if (!isnum(ntmp, 0)) {
		r_err("NUM", "Invalid Match Number");
	} else {
		nmatch = atolp(ntmp);
	}
	if (!chan || !*chan)
		matchan = TRUE;
	if (!netw || !*netw)
		matchnetw = TRUE;
	for (nct = fldoptstart; nct; nct = nct->next) {
		if ( (matchan || iswm(chan, nct->chan)) && (matchnetw || iswm(netw, nct->netw)) ) {
			for (t = nct->list; t; t = t->next) {
				if (iswm(type, t->type) && ++count == nmatch) {
					char rflags[10] = "+", rnetw[MAXSTRINGLEN] = "", rchan[MAXSTRINGLEN] = "";
					if (nct->netw && *(nct->netw)) {
						lstrcat(rflags, "n");
						wsprintf(rnetw, " %s", nct->netw);
					}
					if (nct->chan && *(nct->chan)) {
						lstrcat(rflags, "c");
						wsprintf(rchan, " %s", nct->chan);
					}
					if (!t->state)
						lstrcat(rflags, "d");
					wsprintf(data, "+OK %s %d %d %s%s%s", t->type, t->max, t->secs, rflags, rnetw, rchan);
					return 3;
				}
			}
		}
	}
	if (nmatch == 0) {
		wsprintf(data, "+OK %d", count);
		return 3;
	}
	r_err("NOMATCH", "No match found");
}

/*
 * FloodOptNetchan <N>
 */
MircFunc FloodOptNetchan (FUNCPARMS)
{
	unsigned int count = 0, nmatch;
	char *tmp  = data;
	char *ntmp = getword(&tmp);
	floodoptnc_t t;

	if (!*ntmp) {
		nmatch = 1;
	} else if (!isnum(ntmp, 0)) {
		r_err("NUM", "Invalid Match Number");
	} else {
		nmatch = atolp(ntmp);
	}
	for (t = fldoptstart; t; t = t->next) {
		if (*(t->chan) || *(t->netw)) {
			if (++count == nmatch) {
				char rflags[10] = "+", rnetw[MAXSTRINGLEN] = "", rchan[MAXSTRINGLEN] = "";
				if (t->netw && *(t->netw)) {
					lstrcat(rflags, "n");
					lstrcpy(rnetw, t->netw);
				}
				if (t->chan && *(t->chan)) {
					lstrcat(rflags, "c");
					lstrcpy(rchan, t->chan);
				}
				wsprintf(data, "+OK %s %s %s", rflags, rnetw, rchan);
				return 3;
			}
		}
	}
	if (nmatch == 0) {
		wsprintf(data, "+OK %d", count);
		return 3;
	}
	r_err("NOMATCH", "No match found");
}

/*
 * FloodOptCount
 */
MircFunc FloodOptCount (FUNCPARMS)
{
	wsprintf(data, "+OK %d", flood_opt_nodes_getnum());
	return 3;
}

/*
 * FloodOptClear
 */
MircFunc FloodOptClear (FUNCPARMS)
{
	wsprintf(data, "+OK %d", flood_opt_cleanup());
	return 3;
}

/*
 * FloodOptLoad <file>
 */
MircFunc FloodOptLoad (FUNCPARMS)
{
	char *tmp = data;
	char *auxl;
	char *auxb;
	HANDLE hfile = NULL;
	DWORD  bytesread;
 	char buff[4096 + 1];
	char line[MAXSTRINGLEN];
	char aux[4096]; /* what the hell, i'm lazy */
	unsigned int wpos = 0, totalread = 0, nline = 0;
	register int i;

	if (myfileopen(&hfile, data, tmp, FALSE))
		return 3;

	flood_opt_cleanup();
	auxl = line;
	do {
		for (i = 0; i < 4096 + 1; ++i)
			buff[i] = 0;
		if (ReadFile(hfile, buff, 4096, &bytesread, NULL)) {
			for (auxb = buff, i = 0; *auxb && i < 4096 + 1; auxb++, ++i) {
				if (++wpos == (MAXSTRINGLEN - 4) || *auxb == 10) {
					*auxl = 0;
					lstrcpy(aux, line);
					if (++nline > 1) {
						if (!flood_opt_intset(aux))
							++totalread;
					} else if (lstrcmpi(aux, "#aircdll FloodOptTable 1")) {
						CloseHandle(hfile);
						r_err("FOPTFILE", "Invalid Flood Options File");
					}
					auxl = line;
					wpos = 0;
				}
				else if (*auxb != 13)
					*auxl++ = *auxb;
			}
		}
	} while (bytesread == 4096);
	CloseHandle(hfile);
	wsprintf(data, "+OK %d", totalread);
	return 3;
}

/*
 * FloodOptSave <file>
 */
MircFunc FloodOptSave (FUNCPARMS)
{
	char *tmp = data;
	char line[1024];
	HANDLE hfile = NULL;
	unsigned int count = 0;
	floodoptnc_t nct;
	floodopt_t t;

	if (myfileopen(&hfile, data, tmp, TRUE))
		return 3;
	lstrcpy(line, "#aircdll FloodOptTable 1");
	file_write(&hfile, line);
	for (nct = fldoptstart; nct; nct = nct->next) {
		for (t = nct->list; t; t = t->next) {
			char rflags[10] = "+", rnetw[MAXSTRINGLEN] = "", rchan[MAXSTRINGLEN] = "";
			if (nct->netw && *(nct->netw)) {
				lstrcat(rflags, "n");
				wsprintf(rnetw, " %s", nct->netw);
			}
			if (nct->chan && *(nct->chan)) {
				lstrcat(rflags, "c");
				wsprintf(rchan, " %s", nct->chan);
			}
			if (!t->state)
				lstrcat(rflags, "d");
			wsprintf(line, "%s %d %d %s%s%s", t->type, t->max, t->secs, rflags, rnetw, rchan);
			if (file_write(&hfile, line))
				count++;
		}
	}
	CloseHandle(hfile);
	wsprintf(data, "+OK %d", count);
	return 3;
}

